/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.servermgmt.pe;

import java.io.File;
import java.io.IOException;

import com.sun.enterprise.util.OS;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.util.i18n.StringManager;

import com.sun.enterprise.admin.servermgmt.DomainException;
import com.sun.enterprise.admin.servermgmt.DomainConfig;
import com.sun.enterprise.admin.servermgmt.RepositoryConfig;
import com.sun.enterprise.admin.servermgmt.RepositoryException;
import com.sun.enterprise.admin.servermgmt.DomainsManager;
import com.sun.enterprise.admin.servermgmt.InstancesManager;
import com.sun.enterprise.admin.servermgmt.DomainConfigValidator;
import com.sun.enterprise.admin.servermgmt.StringValidator;
import com.sun.enterprise.admin.servermgmt.FileValidator;
import com.sun.enterprise.admin.servermgmt.RepositoryNameValidator;
import com.sun.enterprise.admin.servermgmt.DomainXmlEventListener;
import com.sun.enterprise.admin.servermgmt.RepositoryManager;
import com.sun.enterprise.admin.servermgmt.util.DomainXmlSAXParser;

import com.sun.enterprise.admin.common.Status;

import com.sun.enterprise.admin.util.TokenValueSet;

import java.util.HashMap;
import java.util.BitSet;

public class PEDomainsManager extends RepositoryManager 
    implements DomainsManager
{
    /**
     * i18n strings manager object
     */
    private static final StringManager strMgr = 
        StringManager.getManager(PEDomainsManager.class);
  
    public PEDomainsManager()
    {
        super();
    }
    
    //PE does not require that an admin user / password is available at start-domain time.
    //SE/SEE does require it.
    public BitSet getDomainFlags()
    {
        BitSet bs = new BitSet();        
        bs.set(DomainConfig.K_FLAG_START_DOMAIN_NEEDS_ADMIN_USER, false);
        return bs;
    }
    
    public void validateDomain(DomainConfig domainConfig, boolean domainExists)
        throws DomainException
    {
        try {
            checkRepository(domainConfig, domainExists, domainExists);
        } catch (RepositoryException ex) {
            throw new DomainException(ex);
        }
    }

    public void validateAdminUserAndPassword(DomainConfig domainConfig) 
        throws DomainException
    {
        try {
            validateAdminUserAndPassword(domainConfig, getDomainUser(domainConfig),
                getDomainPasswordClear(domainConfig));
        } catch (RepositoryException ex) {
            throw new DomainException(ex);
        }
    }
    
    public void validateMasterPassword(DomainConfig domainConfig) 
        throws DomainException
    {
        try {
            validateMasterPassword(domainConfig, getMasterPasswordClear(domainConfig));
        } catch (RepositoryException ex) {
            throw new DomainException(ex);
        }
    }
    
    public void createDomain(DomainConfig domainConfig) 
        throws DomainException
    {
        PEFileLayout layout = getFileLayout(domainConfig);
        
        try {
            new RepositoryNameValidator(strMgr.getString("domainsRoot")).
                    checkValidXmlToken(
                        layout.getRepositoryRootDir().getAbsolutePath());
            layout.createRepositoryRoot();        
            new PEDomainConfigValidator().validate(domainConfig);
            checkRepository(domainConfig, false);
        } catch (Exception ex) {
            throw new DomainException(ex);
        }
        
        try {            
            String masterPassword = getMasterPasswordClear(domainConfig);
            layout.createRepositoryDirectories();
            createDomainXml(domainConfig);
            createDomainXmlEvents(domainConfig);
            createScripts(domainConfig);
            createServerPolicyFile(domainConfig);
            createAdminKeyFile(domainConfig, getDomainUser(domainConfig), 
                getDomainPasswordClear(domainConfig));
            createKeyFile(domainConfig, getDomainUser(domainConfig),
                getDomainPasswordClear(domainConfig));
            createAppClientContainerXml(domainConfig);
            createIndexFile(domainConfig);
            createDefaultWebXml(domainConfig);
            createLoginConf(domainConfig);
            createWssServerConfig(domainConfig);
            createWssServerConfigOld(domainConfig);
            createSSLCertificateDatabase(domainConfig, masterPassword);                                     
            changeMasterPasswordInMasterPasswordFile(domainConfig, masterPassword, 
                saveMasterPassword(domainConfig));
            createPasswordAliasKeystore(domainConfig, masterPassword);        
            createTimerWal(domainConfig);
            createTimerDbn(domainConfig);
            createMQInstance(domainConfig);
            if (layout.getDerbyEjbTimerSqlFile().exists()) //will be cleaned up once derby integrates, 05/19/2005
                handleDerby(domainConfig);
            setPermissions(domainConfig);
        } catch (DomainException de) {
            //rollback
            FileUtils.liquidate(getDomainDir(domainConfig));
            throw de;
        } catch (Exception ex) {
            //rollback
            FileUtils.liquidate(getDomainDir(domainConfig));
            throw new DomainException(ex);
        }
    }
    
    /**
     * Sets the permissions for the domain directory, its config directory,
     * startserv/stopserv scripts etc.
     */
    protected void setPermissions(DomainConfig domainConfig) throws DomainException
    {        
        final PEFileLayout layout = getFileLayout(domainConfig);
        try {
            //4958533
            chmod("-R u+x ", layout.getBinDir());
            chmod("-R g-rwx,o-rwx ", layout.getConfigRoot());
            //4958533
        } catch (Exception e) {
            throw new DomainException(
                strMgr.getString("setPermissionError"), e);
        }   
    }

    public void deleteDomain(DomainConfig domainConfig) 
        throws DomainException
    {               
        try {
            deleteRepository(domainConfig);
        } catch (Exception e) {
            throw new DomainException(e);
        }
    }

    public void startDomain(DomainConfig domainConfig) 
        throws DomainException
    {                        
        try {
            checkRepository(domainConfig);            
            String[] options = getInteractiveOptions(
                (String)domainConfig.get(DomainConfig.K_USER), 
                (String)domainConfig.get(DomainConfig.K_PASSWORD),
                (String)domainConfig.get(DomainConfig.K_MASTER_PASSWORD),
                (HashMap)domainConfig.get(DomainConfig.K_EXTRA_PASSWORDS));            
            getInstancesManager(domainConfig).startInstance(options);            
        } catch (Exception e) {
            throw new DomainException(e);
        }
    }

    public void stopDomain(DomainConfig domainConfig) 
        throws DomainException
    {                        
        try {
            checkRepository(domainConfig);            
            getInstancesManager(domainConfig).stopInstance();
        } catch (Exception e) {
            throw new DomainException(e);
        }
    }

    public String[] listDomainsAndStatus(DomainConfig domainConfig)
        throws DomainException
    {
        try {
            return listDomainsAndStatusAsString(domainConfig);
        } catch (Exception e) {
            throw new DomainException(e);
        }        
    }

    /**
     * Lists all the domains.
     */
    public String[] listDomains(DomainConfig domainConfig)
        throws DomainException
    {        
        try {
            return listRepository(domainConfig); 
        } catch (Exception e) {
            throw new DomainException(e);
        }        
    }


    protected void createDomainXmlEvents(DomainConfig domainConfig) 
        throws DomainException {
            try {
                final PEFileLayout layout = getFileLayout(domainConfig);
                final File domainXml = layout.getDomainConfigFile();
                DomainXmlSAXParser parser = new DomainXmlSAXParser();
                try {
                    parser.parse(domainXml,layout.getDtdFile());
                }
                catch(Exception e) {
                    throw new DomainException(
                        strMgr.getString("domainXmlNotParsed"), e);
                }
                String className = parser.getDomainXmlEventListenerClass();
                if(className!=null) {
                    DomainXmlEventListener listener = (DomainXmlEventListener) Class.forName(className).newInstance();
                    listener.handleCreateEvent(domainConfig);
                }
            }
            catch(Exception e) {
                throw new DomainException(
                    strMgr.getString("domainXmlEventsNotCreated"), e);
            }
    }

    /**
     * The EEDomains manager needs to have an augmented set of tokens
     */
    protected TokenValueSet getDomainXmlTokens(DomainConfig domainConfig) {
        return PEDomainXmlTokens.getTokenValueSet(domainConfig);
    }
    
    protected void createDomainXml(DomainConfig domainConfig) 
        throws DomainException
    {
        try
        {
            final PEFileLayout layout = getFileLayout(domainConfig);
            final File domainXml = layout.getDomainConfigFile();
            final File domainXmlTemplate;
            String templateName = (String)domainConfig.get(DomainConfig.K_TEMPLATE_NAME);
            if((templateName == null)||(templateName.equals("")))
                domainXmlTemplate = layout.getDomainXmlTemplate();
            else
                domainXmlTemplate = layout.getDomainXmlTemplate(templateName);

            TokenValueSet tokens = getDomainXmlTokens(domainConfig);
            generateFromTemplate(tokens, domainXmlTemplate, domainXml);
        }
        catch(Exception e)
        {
            throw new DomainException(
                strMgr.getString("domainXmlNotCreated"), e);
        }
    }

    protected void createScripts(DomainConfig domainConfig)
        throws DomainException
    {
        final TokenValueSet tokens = PEScriptsTokens.getTokenValueSet(domainConfig);
        createStartServ(domainConfig, tokens);
        createStopServ(domainConfig, tokens);
    }

    void createStartServ(DomainConfig domainConfig, 
        TokenValueSet  tokens) throws DomainException
    {
        try
        {
            final PEFileLayout  layout = getFileLayout(domainConfig);
            final File startServTemplate = layout.getStartServTemplate();
            final File startServ = layout.getStartServ();
            generateFromTemplate(tokens, startServTemplate, startServ);
        }
        catch (Exception e)
        {
            throw new DomainException(
                strMgr.getString("startServNotCreated"), e);
        }
    }

    void createStopServ(DomainConfig domainConfig, 
        TokenValueSet  tokens) throws DomainException
    {
        try
        {
            final PEFileLayout  layout = getFileLayout(domainConfig);
            final File stopServTemplate = layout.getStopServTemplate();
            final File stopServ = layout.getStopServ();
            generateFromTemplate(tokens, stopServTemplate, stopServ);
        }
        catch (Exception e)
        {
            throw new DomainException(
                strMgr.getString("stopServNotCreated"), e);
        }
    } 

    protected void createAppClientContainerXml(
        DomainConfig domainConfig) throws DomainException
    {
        try
        {
            final PEFileLayout layout = getFileLayout(domainConfig);
            final File accXmlTemplate = 
                layout.getAppClientContainerXmlTemplate();
            final File accXml = layout.getAppClientContainerXml();
            TokenValueSet tokens = PEAccXmlTokens.getTokenValueSet(domainConfig);
            generateFromTemplate(tokens, accXmlTemplate, accXml);
        }
        catch(Exception e)
        {
            throw new DomainException(strMgr.getString("accXmlNotCreated"), e);
        }
    }

    protected void createIndexFile(
        DomainConfig domainConfig) throws DomainException
    {
        final PEFileLayout layout = getFileLayout(domainConfig);
        final File src = layout.getIndexFileTemplate();
        final File dest = layout.getIndexFile();
        try
        {
            FileUtils.copy(src, dest);
        }
        catch (IOException ioe)
        {
            throw new DomainException(
                strMgr.getString("indexFileNotCreated"), ioe);
        }
    }

    protected void createDefaultWebXml(
        DomainConfig domainConfig) throws DomainException
    {
        final PEFileLayout layout = getFileLayout(domainConfig);
        final File src = layout.getDefaultWebXmlTemplate();
        final File dest = layout.getDefaultWebXml();
        try
        {
            FileUtils.copy(src, dest);
        }
        catch (IOException ioe)
        {
            throw new DomainException(
                strMgr.getString("defaultWebXmlNotCreated"), ioe);
        }
    }
    
    protected void createLoginConf(
        RepositoryConfig config) throws DomainException
    {
        final PEFileLayout layout = getFileLayout(config);
        final File src = layout.getLoginConfTemplate();
        final File dest = layout.getLoginConf();
        try
        {
            FileUtils.copy(src, dest);
        }
        catch (IOException ioe)
        {
            throw new DomainException(
                strMgr.getString("loginConfNotCreated"), ioe);
        }
    }    

    protected void createWssServerConfigOld(RepositoryConfig config)
        throws DomainException
    {
        final PEFileLayout layout = getFileLayout(config);
        final File src = layout.getWssServerConfigOldTemplate();
        final File dest = layout.getWssServerConfigOld();
        try
        {
            FileUtils.copy(src, dest);
        }
        catch (IOException ioe)
        {
            throw new DomainException(
                strMgr.getString("wssserverconfignotcreated"), ioe);
        }

    }

    protected void createWssServerConfig(RepositoryConfig config)
        throws DomainException
    {
        final PEFileLayout layout = getFileLayout(config);
        final File src = layout.getWssServerConfigTemplate();
        final File dest = layout.getWssServerConfig();
        try
        {
            FileUtils.copy(src, dest);
        }
        catch (IOException ioe)
        {
            throw new DomainException(
                strMgr.getString("wssserverconfignotcreated"), ioe);
        }

    }

    protected File getDomainDir(DomainConfig domainConfig)
    {
        return getRepositoryDir(domainConfig);
    }

    protected File getDomainRoot(DomainConfig domainConfig)
    {
        return getRepositoryRootDir(domainConfig);
    }

    String getDefaultInstance()
    {
        return PEFileLayout.DEFAULT_INSTANCE_NAME;
    }
     
    /** Returns the domain user from the domainConfig.
     *  @param Map that represents the domain configuration
     *  @return String representing the domain user if the given map contains
     *  it, null otherwise
    */

    protected static String getDomainUser(DomainConfig domainConfig) 
    {
        return ( (String) domainConfig.get(DomainConfig.K_USER) );
    }
    
    /** Returns the domain user's password in cleartext from the domainConfig.
     *  @param Map that represents the domain configuration
     *  @return String representing the domain user password if the 
     *  given map contains it, null otherwise
    */

    protected static String getDomainPasswordClear(DomainConfig domainConfig) 
    {
        return ( (String) domainConfig.get(DomainConfig.K_PASSWORD) );
    } 
    
    protected static String getMasterPasswordClear(DomainConfig domainConfig)
    {
        return ((String)domainConfig.get(DomainConfig.K_MASTER_PASSWORD));
    }
    
    protected static String getNewMasterPasswordClear(DomainConfig domainConfig)
    {
        return ((String)domainConfig.get(DomainConfig.K_NEW_MASTER_PASSWORD));
    }
    
    protected static boolean saveMasterPassword(DomainConfig domainConfig) {
        Boolean b = (Boolean)domainConfig.get(DomainConfig.K_SAVE_MASTER_PASSWORD);
        return b.booleanValue();
    }
    
    /**
     * Changes the master password for the domain
     */    
    public void changeMasterPassword(DomainConfig config) throws DomainException
    {                                      
        try {
            //Ensure that the entity is stopped
            final int status = getInstancesManager(config).getInstanceStatus();
            if (status != Status.kInstanceNotRunningCode) {
                throw new DomainException(
                    strMgr.getString("cannotChangePassword_invalidState",
                        config.getDisplayName(), Status.getStatusString(status)));
            }
        
            String oldPass = getMasterPasswordClear(config);
            String newPass = getNewMasterPasswordClear(config);                        
            
            //Change the password of the keystore alias file
            changePasswordAliasKeystorePassword(config, oldPass, newPass);
            
            //Change the password of the keystore and truststore
            changeSSLCertificateDatabasePassword(config, oldPass, newPass);

            //Change the password in the masterpassword file or delete the file if it is 
            //not to be saved.
            changeMasterPasswordInMasterPasswordFile(config, newPass, saveMasterPassword(config));
        } catch (Exception ex) {
            throw new DomainException(
                strMgr.getString("masterPasswordNotChanged"), ex);
        }
    }
    
    public String[] getExtraPasswordOptions(DomainConfig config)
        throws DomainException
    {
        return null;
    }
}
