/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.util.io;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.File;



public class StreamFlusher extends Thread {
    
    private InputStream _input=null;
    private OutputStream _output=null;
    private String _logFile=null;

    
    public StreamFlusher(InputStream input, OutputStream output) {
        this(input, output, null);
    }

    
    public StreamFlusher(InputStream input, OutputStream output, String logFile) {
        this._input=input;
        this._output=output;
        this._logFile=logFile;
    }
    
    public void run() {
        
        // check for null stream
        if (_input == null) return;
        
        PrintStream printStream=null;
        
        // If applicable, write to a log file
        if (_logFile != null) {
            try {
                if(createFileStructure(_logFile)) {
                    // reset streams to logfile
                    printStream = new PrintStream(new FileOutputStream(_logFile, true), true);
                } else {
                    // could not write to log for some reason
                    _logFile=null;
                }
            } catch (IOException ie) {
                ie.printStackTrace();
                _logFile=null;
            }
        }
        
        // transfer bytes from input to output stream
        try {
            int byteCnt=0;
            byte[] buffer=new byte[4096];
            while ((byteCnt=_input.read(buffer)) != -1) {
                if (_output != null && byteCnt > 0) {
                    _output.write(buffer, 0, byteCnt);
                    _output.flush();
                    
                    // also send to log, if it exists
                    if (_logFile != null) {
                        printStream.write(buffer, 0, byteCnt);
                        printStream.flush();
                    }
                }
                yield();
            }
        } catch (IOException e) {
            // shouldn't matter
        }
    }

    
    /**
     * createFileStructure - This method validates that that the file can be written to.  It the
     * if the parent directory structure does not exist, it will be created
     *
     * @param logFile - fully qualified path of the logfile
     */
    protected boolean createFileStructure(String logFile) {
        boolean bRet=false;
        File outputFile=new File(logFile);
        
        try {
            // Verify that we can write to the output file
            File parentFile = new File(outputFile.getParent());
            // To take care of non-existent log directories
            if ( !parentFile.exists() ) {
                // Trying to create non-existent parent directories
                parentFile.mkdirs();
            }
            // create the file if it doesn't exist
            if (!outputFile.exists()) {
                outputFile.createNewFile();
            }
            if (outputFile.canWrite()) {
                // everything is okay to logfile
                bRet=true;
            }
        } catch (IOException e) {
            // will only see on verbose more, so okay
            e.printStackTrace();
        }

        return bRet;
    }    
}