/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.appclient.jws;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;


/**
 *Ad hoc servlet registered to receive requests for the user-specified (or
 *defaulted) context-root.
 *<p>
 *The user can (but is not required to) specify a context root in the runtime
 *deployment descriptor for the app client.  That context root - of, if omitted,
 *a default value computed by the app server - is registered dynamically so
 *requests to that context root are routed to an instance of this servlet.  In
 *turn, this servlet forwards such requests to the system web application that
 *actually processes all Java Web Start requests for app clients.
 *
 * @author tjquinn
 */
public class JWSAdHocServlet extends HttpServlet {

    /** servlet init parameter name for specifying the context root value */
    public static final String CONTEXT_ROOT_PARAMETER_NAME = "context-root";
    
    /** servlet init parameter name for specifying the category value (appclient or application) */
    public static final String CATEGORY_PARAMETER_NAME = "category";
    
    /** the context for the system web app that handles all Java Web Start requests */
    private ServletContext jwsAppContext = null;
    
    /** a dispatcher used in forwarding requests to the system web app */
    private RequestDispatcher systemWebAppDispatcher = null;
    
    /** the actual context root, obtained from the init parameter, to which reqs should be dispatched */
    private String targetContextRoot = null;
    
    /** the category from the init parameter */
    private String category = null;
    
    /** Creates a new instance of JWSAdHocServlet */
    public JWSAdHocServlet() {
    }
    
    /**
     *Responds to any method.
     *@param the incoming request
     *@param the outgoing response
     *@throws ServletException in case of any errors
     */
    protected void service(HttpServletRequest request, HttpServletResponse response) throws ServletException {
        /*
         *Use the context for the Java Web Start system web app and forward this request
         *to that app.
         */
        JWSAdHocServletRequestWrapper wrappedRequest = new JWSAdHocServletRequestWrapper(request, targetContextRoot, category);
        try {
            getJWSRequestDispatcher().forward(wrappedRequest, response);
        } catch (Throwable thr) {
            throw new ServletException("Error dispatching request to Java Web Start app client application", thr);
        }
    }
    
    /**
     *Init method responsible for retrieiving init params.
     */
    public void init() {
        ServletConfig config = getServletConfig();
        
        targetContextRoot = config.getInitParameter(CONTEXT_ROOT_PARAMETER_NAME);
        category = config.getInitParameter(CATEGORY_PARAMETER_NAME);
    }
    
    /**
     *Returns the context for the Java Web Start app client system web app.
     *@return the servlet context for the system web app
     */
    private ServletContext getJWSAppContext() {
        if (jwsAppContext == null) {
            String uri = NamingConventions.webAppURI();
            jwsAppContext = getServletContext().getContext(uri);
        }
        return jwsAppContext;
    }
    
    /**
     *Returns the dispatcher to the system web app.
     *@return the request dispatcher
     */
    private RequestDispatcher getJWSRequestDispatcher() {
        if (systemWebAppDispatcher == null) {
            String uri = NamingConventions.webAppURI();
            WebPath webPath = new WebPath(uri);
            
            ServletContext sc = getServletContext().getContext(webPath.contextRoot());
            String servletContextName = sc.getServletContextName();
            systemWebAppDispatcher = sc.getRequestDispatcher(webPath.path() + "/" + category + targetContextRoot);
        }
        return systemWebAppDispatcher;
    }
}
