/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.appclient.jws;

import com.sun.enterprise.deployment.backend.DeploymentLogger;
import com.sun.enterprise.security.SSLUtils;
import com.sun.enterprise.util.i18n.StringManager;
import java.io.File;
import java.net.URI;
import java.security.AccessControlException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.Permission;
import java.util.ArrayList;
import java.util.logging.Logger;
import sun.security.tools.JarSigner;

/**
 * Represents a jar file that is signed before being served in 
 * response to a Java Web Start request.
 *
 * For PE, signing occurs when the jar is first requested.  
 *
 * @author tjquinn
 */
public class SignedStaticContent extends StaticContent {
    
    /** file name suffix used to create the name for the signed jar */
    private static final String SIGNEDJAR_OPTION = "-signedjar";

    /** jarsigner option name introducing the keystore file spec */
    private static final String KEYSTORE_OPTION = "-keystore";

    /** jarsigner option name introducing the keystore password value */
    private static final String STOREPASS_OPTION = "-storepass";

    /** the unsigned jar to be served in signed form */
    private File unsignedJar;
    
    /** the signed jar */
    private File signedJar;
    
    /** URI for the app server installation root */
    private URI installRootURI;

    /** path to the app server's keystore */
    private String keystoreAbsolutePath = null;
    
    /** property name the value of which points to the keystore */
    private String KEYSTORE_PATH_PROPERTYNAME = "javax.net.ssl.keyStore";

    /** property name optionally set by the admin in domain.xml to select an alias for signing */
    private String USER_SPECIFIED_ALIAS_PROPERTYNAME = "com.sun.aas.jws.signing.alias";

    /** default alias for signing if the admin does not specify one */
    private String DEFAULT_ALIAS_VALUE = "s1as";

    private Logger logger = DeploymentLogger.get();

    private StringManager localStrings;
    
    /**
     * Creates a new instance of SignedStaticContent
     * 
     * @param origin the origin from which the jar to be signed comes
     * @param contentKey key by which this jar will be retrievable when requested
     * @param path the relative path within the app server to the jar
     * @param signedJar specifies what the resulting signed jar file should be
     * @param unsignedjar the existing unsigned jar to be signed just-in-time
     * @param installRootURI the app server's installation directory
     * @param isMain indicates if the jar contains the mail class that Java Web Start should launch
     */
    
    public SignedStaticContent(
            ContentOrigin origin, 
            String contentKey, 
            String path, 
            File signedJar,
            File unsignedJar, 
            URI installRootURI,
            StringManager localStrings,
            boolean isMain) throws Exception {
        super(origin, contentKey, path, signedJar, installRootURI, isMain);
        
        /*
         *Find out as much as we can in order to sign the jar, but do not sign it yet.
         */
        this.installRootURI = installRootURI;
        this.unsignedJar = unsignedJar;
        this.signedJar = signedJar;
        this.localStrings = localStrings;
    }
    
    /**
     *Returns the URI, relative to the app server installation directory, of
     *the signed jar file to be published, signing the unsigned jar if needed
     *to create the signed one to serve.
     *@return relative URI to the jar
     */
    public URI getRelativeURI() {
        try {
            ensureSignedFileUpToDate();
            return installRootURI.relativize(signedJar.toURI());
        } catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }

    /**
     *Makes sure that the signed jar exists and is up-to-date compared to the
     *corresponding unsigned jar.  If not, create a new signed jar using
     *the alias provided on this object's constructor.
     *@throws KeyStoreException in case of errors reading the key store
     *@throws IllegalArgumentException if the unsigned jar does not exists
     */
    private void ensureSignedFileUpToDate() throws KeyStoreException, IllegalArgumentException, Exception {
        /*
         *Check to see if the signed version of this jar is present.
         */
        if ( ! unsignedJar.exists()) {
            throw new IllegalArgumentException(
                    localStrings.getString("jws.sign.noUnsignedJar", unsignedJar.getAbsolutePath()));
        }
        if ( ! signedJar.exists() || (signedJar.lastModified() < unsignedJar.lastModified())) {
            signJar();
        }
    }
    
    /**
     *Signs the jar file.
     *@throws KeyStoreException in case of errors accessing the keystore
     */
    private void signJar() throws KeyStoreException, Exception {
        /*
         *Compose the arguments to pass to the JarSigner class equivalent to
         *this command:
         *jarsigner 
         *  -signedjar <signed JAR file spec>
         *  -keystore <keystore file spec>
         *  -storepass <keystore password>
         *  <unsigned JAR file spec>
         *  <alias>
         *
         *Note that techniques for importing certs into the AS keystore are
         *documented as requiring that the key password and the keystore
         *password be the same.  We rely on that here because there is no
         *provision for extracting the password for an alias from the keystore.
         */
        
        ArrayList<String> args = new ArrayList<String>();
        args.add(SIGNEDJAR_OPTION);
        args.add(signedJar.getAbsolutePath());
        
        args.add(KEYSTORE_OPTION);
        args.add(getKeystoreAbsolutePath());
        
        args.add(STOREPASS_OPTION);
        int passwordSlot = args.size();
        args.add(getKeystorePassword());
        
        args.add(unsignedJar.getAbsolutePath());
        
        args.add(getAlias());
        long startTime = System.currentTimeMillis();
        /*
         *Save the current security manager; restored later.
         */
        SecurityManager mgr = System.getSecurityManager();
        
        try {
            /*
             *While running the JarSigner use a security manager that forbids
             *VM exits, because the JarSigner uses them as part of its
             *error handling and we do not want to exit the app server in case
             *of jar signing errors.
             */
            NoExitSecurityManager noExitMgr = new NoExitSecurityManager(mgr);
            System.setSecurityManager(noExitMgr);

            /*
             *Run the jar signer.
             */
            JarSigner.main(args.toArray(new String[args.size()]));
        } catch (Throwable t) {
            /*
             *In case of any problems, make sure there is no ill-formed signed
             *jar file left behind.
             */
            signedJar.delete();
            
            /*
             *The jar signer will have written some information to System.out
             *and/or System.err.  Refer the user to those earlier messages.
             */
            throw new Exception(localStrings.getString("jws.sign.errorSigning", signedJar.getAbsolutePath()), t);
        } finally {
            /*
             *Restore the saved security manager.
             */
            System.setSecurityManager(mgr);

            /*
             *Clear out the password.
             */
            args.set(passwordSlot, null);

            long duration = System.currentTimeMillis() - startTime;
            logger.fine("Signing " + unsignedJar.getAbsolutePath() + " took " + duration + " ms");
        }
    }

    /**
     *Returns the absolute path to the keystore.
     *@return path to the keystore
     */
    private String getKeystoreAbsolutePath() {
        if (keystoreAbsolutePath == null) {
            keystoreAbsolutePath = System.getProperty(KEYSTORE_PATH_PROPERTYNAME);
        }
        return keystoreAbsolutePath;
    }

    /**
     *Returns the password for the keystore.
     *@return the keystore password
     */
    private String getKeystorePassword() {
        return SSLUtils.getKeyStorePass();
    }

    /**
     *Returns the alias to use for signing the jar file.
     *@return the alias to use for signing the jar file
     *@throws KeyStoreException in case of errors accessing the keystore
     */
    private String getAlias() throws KeyStoreException, Exception {
        /*
         *Choose what alias to use for signing the jar.
         *If the user specified one, make sure it exists in the keystore.
         *Even if we fall back to the default alias, make sure it is there
         *in the keystore.  Throw exceptions if neither alias is present.
         */
        KeyStore keystore = SSLUtils.getKeyStore();
        
        String alias = System.getProperty(USER_SPECIFIED_ALIAS_PROPERTYNAME);
        if (alias == null || ! checkUserAlias(keystore, alias)) {
            /*
             *Either the admin did not specify an alias or s/he did and it
             *is not in the keystore.  Use the default alias in either case,
             *making sure the default is in the keystore first.
             */
            checkDefaultAlias(keystore); // throws exception if alias is absent
            alias = DEFAULT_ALIAS_VALUE;
        }
        return alias;
    }
    
    /**
     *Makes sure the specified alias is in the keystore.  If not, throws an
     *exception.
     *@param keystore the keystore to use in checking the default alias
     *@param candidateAlias the alias to check for
     *@throws IllegalStateException if the keystore does not contain the default alias
     */
    private void checkDefaultAlias(KeyStore keystore) throws KeyStoreException {
        if ( ! keystore.containsAlias(DEFAULT_ALIAS_VALUE)) {
            throw new IllegalStateException(localStrings.getString("jws.sign.defaultAliasAbsent", DEFAULT_ALIAS_VALUE));
        }
    }
    
    /**
     *Returns whether the specified alias is present in the keystore or not.  Also
     *logs a warning if the user-specified alias is missing.
     *@param keystore the keystore to use in checking the user alias
     *@param candidateAlias the alias to look for
     *@return true if the alias is present in the keystore; false otherwise
     *@throws KeyStoreException in case of error accessing the keystore
     */
    private boolean checkUserAlias(KeyStore keystore, String candidateAlias) throws KeyStoreException {
        boolean result;
        if ( ! (result = keystore.containsAlias(candidateAlias)) ) {
            logger.warning(localStrings.getString("jws.sign.userAliasAbsent", candidateAlias));
        }
        return result;
    }
    
    /**
     *A security manager that rejects any attempt to exit the VM.
     */
    private class NoExitSecurityManager extends SecurityManager {
        
        private SecurityManager originalManager;
        
        public NoExitSecurityManager(SecurityManager originalManager) {
            this.originalManager = originalManager;
        }
        
        public void checkExit(int status) {
            /*
             *Always reject attempts to exit the VM.
             */
            throw new AccessControlException("System.exit");
        }
        
        public void checkPermission(Permission p) {
            /*
             *Delegate to the other manager, if any.
             */
            if (originalManager != null) {
                originalManager.checkPermission(p);
            }
        }
    }
}
