/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.appclient.jws;

import java.io.File;
import java.net.URI;

/**
 *Represents content that does not change from one request for it to the next.
 *
 * @author tjquinn
 */
public class StaticContent extends Content {

    /** a URI relative to the install root URI of the app server installation */
    private URI relativeURI;

    /** indicates if this static content should be marked as the main jar file
     *in a JNLP document
     */
    private boolean isMainJarFile;
    
    /**
     *Returns a new instance of StaticContent
     *@param origin the ContentOrigin that owns the new static content
     *@param contentKey content key for storing and retrieving the content
     *@param path path within the content's subcategory of this content
     *@param file File object for the physical file corresponding to this static content
     *@param installRootURI URI specifying the location of the static content file
     *@return new StaticContent object
     */
    public StaticContent(ContentOrigin origin, String contentKey, String path, File file, URI installRootURI, boolean isMainJarFile) {
        super(origin, contentKey, path);
        URI fileURI = file.toURI();
        relativeURI = installRootURI.relativize(fileURI);
        this.isMainJarFile = isMainJarFile;
    }

    /**
     *Retrieves the content's URI relative to the product's installation root.
     */
    public URI getRelativeURI() {
        return relativeURI;
    }

    /**
     *Returns a string representation of the StaticContent instance.
     */
    public String toString() {
        String result = super.toString() + ", relative URI=" + getRelativeURI() + ", isMain=" + isMainJarFile;
        return result;
    }

    /**
     *Returns a string expression suitable for use in a JNLP document's <jar>
     *element for whether this represents the main jar file of the application.
     */
    public String getMainExpr() {
        String result;
        if (isMainJarFile) {
            result = " main=\"true\"";
        } else {
            result = "";
        }
        return result;
    }

    /**
     *Returns a string expression suitable for inclusion in a JNLP document
     *listing this jar file.
     *@return string of the form <jar href="${relative-path-to-virtual-content}"/>
     */
    public String asJNLPJarElement() {
        String fullPath = "${request.scheme}://${request.host}:${request.port}" + 
                NamingConventions.fullJarPath(getContentKey());
        
        return "        <jar href=\"" + fullPath + "\" " + getMainExpr() + "/>" + lineSep;
    }
}
