/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/* 
 * AppReDeployer.java
 *
 * Created on April 27, 2002, 3:38 PM
 * 
 * @author  bnevins
 * <BR> <I>$Source: /cvs/glassfish/appserv-core/src/java/com/sun/enterprise/deployment/backend/AppReDeployer.java,v $
 *
 * Indentation Information:
 * 0. Please (try to) preserve these settings.
 * 1. Tabs are preferred over spaces.
 * 2. In vi/vim -
 *             :set tabstop=4 :set shiftwidth=4 :set softtabstop=4
 * 3. In S1 Studio -
 *             1. Tools->Options->Editor Settings->Java Editor->Tab Size = 4
 *             2. Tools->Options->Indentation Engines->Java Indentation Engine->Expand Tabs to Spaces = False.
 *             3. Tools->Options->Indentation Engines->Java Indentation Engine->Number of Spaces per Tab = 4.
 */

package com.sun.enterprise.deployment.backend;

import java.io.*;
import java.util.logging.*;
import com.sun.enterprise.instance.ApplicationEnvironment;
import com.sun.enterprise.instance.InstanceEnvironment;
import com.sun.enterprise.util.io.FileSource;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.util.diagnostics.Reminder;
import com.sun.enterprise.deployment.phasing.DeploymentServiceUtils;

public class AppReDeployer extends AppDeployer
{
	///////////////////////////////////////////////////////////////////////////
	//////////    Constructor                                //////////////////
	///////////////////////////////////////////////////////////////////////////

	AppReDeployer(DeploymentRequest r)  throws IASDeploymentException
	{
		super(r);
	}

	///////////////////////////////////////////////////////////////////////////
	//////////    Public Methods                            //////////////////
	///////////////////////////////////////////////////////////////////////////

	/** 
	 * Attempt to delete previously deployed directory-tree.
	 * <p> This method call is <b>guaranteed</b> to never throw any kind of Exception
	 */	
	public void cleanup_internal()
	{
		try
		{
			// don't whack Directory-Redeployments, because the new dir == previous dir!
			if(isArchive() && oldAppDir != null)
			{
				FileUtils.whack(oldAppDir);

				if(oldAppDir.exists())
					logger.info("Unable to delete previous deployment directory: " + oldAppDir.getPath());
				else
					logger.info("Deleted previous deployment directory: " + oldAppDir.getPath());
			}
		}
		catch(Exception e)
		{
			logger.info("Unable to delete previous deployment directory: " + oldAppDir.getPath());
		}
	}

	///////////////////////////////////////////////////////////////////////////
	//////////    Protected and Package Visibility Methods   //////////////////
	///////////////////////////////////////////////////////////////////////////

	protected void begin() throws IASDeploymentException
	{
		// this is here because by the time we get to predeploy() -- the app will no
		// longer be registered.  So we need to save the app-dir right NOW!
		
		super.begin();
		
		try
		{
                        originalAppDir = new File(DeploymentServiceUtils.getLocation(getAppName(), request.getType()));
                        getManager().unregisterDescriptor(getAppName());
                        removePolicy();
		}
		catch(Exception e)
		{
			String msg = localStrings.getString(
					"enterprise.deployment.backend.error_getting_app_location",
					getAppName() );
			throw new IASDeploymentException( msg, e);
		}
	}		

	///////////////////////////////////////////////////////////////////////////
	
	protected final void predeploy() throws IASDeploymentException
	{
		appWasUnregistered = true;
		super.predeploy();
		setOldDirs();
	}

	///////////////////////////////////////////////////////////////////////////

	protected final File setAppDir() throws IASDeploymentException
	{
		File newAppDir = null;
		
		if(isArchive())
		{
			newAppDir = setAppDirArchive();
		}
		else if(isDirectory())
		{
			newAppDir = setAppDirDirectory();
		}
		else
		{
			String msg = localStrings.getString(
					"enterprise.deployment.backend.redeployment_not_dir_or_archive" );
			throw new IASDeploymentException( msg );
		}
		
		newAppDir.mkdirs();
		return newAppDir;
	}

	///////////////////////////////////////////////////////////////////////////

	protected final File getOldStubsDir()
	{
		// this is here AND in the super-class so that ejbc-calling code
		// can exist in the super-class only.
		return oldStubsDir;
	}
	
	///////////////////////////////////////////////////////////////////////////

	protected final File getOldAppDir()
	{
		// this is here AND in the super-class so that ejbc-calling code
		// can exist in the super-class only.
		return oldAppDir;
	}

	///////////////////////////////////////////////////////////////////////////
	
	protected void postDeploy() throws ConfigException, IASDeploymentException
	{
		super.postDeploy();

		if(FileUtils.safeIsDirectory(oldStubsDir))
			FileUtils.whack(oldStubsDir);

		if(FileUtils.safeIsDirectory(oldJSPDir))
			FileUtils.whack(oldJSPDir);

                if(FileUtils.safeIsDirectory(oldXMLDir))
                        FileUtils.whack(oldXMLDir);

                if(FileUtils.safeIsDirectory(oldJWSDir))
                        FileUtils.whack(oldJWSDir);
	}
	
	///////////////////////////////////////////////////////////////////////////

	
	protected final void rollback()
	{
		// careful -- this method is called from inside of a catch block.
		// If we throw an exception from here, it'll be BIG trouble!
		try
		{
			if(isArchive())
				rollbackArchive();
			else
				rollbackDirectory();
		}
		catch(Exception e)
		{
			// Can't do anything about it!!
            logger.log( Level.WARNING,
                    "enterprise.deployment_rollback_error", e );
		}
	}
	///////////////////////////////////////////////////////////////////////////

	protected String whatAreYou()
	{
		return "Redeployment";
	}

	///////////////////////////////////////////////////////////////////////////
	//////////    Private Methods and Data                   //////////////////
	///////////////////////////////////////////////////////////////////////////
	
	private final void rollbackArchive() throws IASDeploymentException, ConfigException
	{
		// blow away new dirs, rename old ones back to new ones...
		// This is inherently complex.  Sigh.
		
		// updated comment August 2003 bnevins
		// redeploy has changed.  Right now we have a completely UNREGISTERED app
		// We need to 
		// 1) rename the _old dir back to the original
		// 2) Register it -- because it was unregistered in order to unload it long ago.

		if(appdirWasRenamed)
		{
			// hose-down new generated directories
			DeleteOrKeepFailedStubs(getStubsDir());

			if(FileUtils.safeIsDirectory(getJSPDir()))
				FileUtils.whack(getJSPDir());		

                        if(FileUtils.safeIsDirectory(getXMLDir()))
                                FileUtils.whack(getXMLDir());

                        if(FileUtils.safeIsDirectory(getJWSDir()))
                                FileUtils.whack(getJWSDir());

			// restore the generated directories back to original...
			if(FileUtils.safeIsDirectory(oldStubsDir))
				oldStubsDir.renameTo(getStubsDir());			

			if(FileUtils.safeIsDirectory(oldJSPDir))
				oldJSPDir.renameTo(getJSPDir());			
                       if(FileUtils.safeIsDirectory(oldXMLDir))
                                oldXMLDir.renameTo(getXMLDir());

                       if(FileUtils.safeIsDirectory(oldJWSDir))
                                oldJWSDir.renameTo(getJWSDir());

			boolean success = false;
			File appdir = getAppDir();

			if(FileUtils.safeIsDirectory(appdir) && FileUtils.safeIsDirectory(oldAppDir))
			{
				FileUtils.whack(appdir);
				success = oldAppDir.renameTo(appdir);
			}

			if(!success)
			{
				FileUtils.whack(oldAppDir);
				logger.log( Level.SEVERE, "enterprise.deployment.backend.directoryRenameFailure",
                                   new Object[] {oldAppDir.getAbsolutePath() , appdir.getAbsolutePath()});
			}

		}

                getManager().registerDescriptor(request.getName(), request.getDescriptor());

	}
	
	///////////////////////////////////////////////////////////////////////////
	
	private final void rollbackDirectory() //  throws IASDeploymentException, ConfigException
	{
		// bnevins -- On 9/8/2003 it was decided by the deploy team to NOT attempt
		// to do a rollback to a previous working version.  So we just fail everything,
		// undeploy, and keep the files right where they are.
		
		request.setReRegisterOnFailure(false);
	}

	///////////////////////////////////////////////////////////////////////////
	
	private final File setAppDirDirectory() throws IASDeploymentException
	{
		FileSource fileSource = request.getFileSource(); 

		if(!fileSource.exists()) 
		{
			String msg = localStrings.getString("enterprise.deployment.backend.file_source_does_not_exist", fileSource );
			throw new IASDeploymentException( msg );
		}

		assert fileSource.isDirectory();
		File appDirectory = fileSource.getFile();

		return appDirectory;
	}

	///////////////////////////////////////////////////////////////////////////

	private final File setAppDirArchive() throws IASDeploymentException
	{
		assert originalAppDir != null;
		File newAppDirectory = originalAppDir;
		oldAppDir = new File(newAppDirectory.getAbsolutePath() + "_old");

		
		// wipe it out if it happens to already exist
		FileUtils.whack(oldAppDir);

		// note -- java does NOT change the path of newAppDirectory, it just renames the file on disk!
		// i.e. "newAppDirectory" will still be pointing at the same file after the call as before the call.
		appdirWasRenamed = FileUtils.renameFile(newAppDirectory, oldAppDir);	

		if(!appdirWasRenamed)
		{
			String msg = localStrings.getString("enterprise.deployment.backend.directory_rename_error",
				newAppDirectory.getAbsolutePath(), oldAppDir.getAbsolutePath());
			throw new IASDeploymentException( msg );
		}

		return newAppDirectory;
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	private final void setOldDirs() throws IASDeploymentException
	{
		File stubsDirectory = getStubsDir();
		File jspDirectory	= getJSPDir();
		File xmlDirectory	= getXMLDir();
		File jwsDirectory	= getJWSDir();
		
		if(FileUtils.safeIsDirectory(stubsDirectory))
		{
			oldStubsDir = new File(stubsDirectory.getPath() + "_old");
			
			if(oldStubsDir.exists())
			{
				// what if there is a regular file with the name we need?  Let's check...
				
				if(oldStubsDir.isDirectory())
					FileUtils.whack(oldStubsDir);
				else
					oldStubsDir.delete();
			}
			
			if(!stubsDirectory.renameTo(oldStubsDir)) {
				String msg = localStrings.getString(
						"enterprise.deployment.backend.directory_rename_error",
						stubsDirectory.getPath(),
						oldStubsDir.getPath() );
				throw new IASDeploymentException( msg );
			}
		}

		// WBN 4-18-2002 I don't know when JSP files are generated. To be safe I'm
		// renaming the old ones -- just for rollback purposes...
		if(FileUtils.safeIsDirectory(jspDirectory))
		{
			oldJSPDir = new File(jspDirectory.getPath() + "_old");
			
			if(oldJSPDir.exists())
			{
				// what if there is a regular file with the name we need?  Let's check...
				
				if(FileUtils.safeIsDirectory(oldJSPDir))
					FileUtils.whack(oldJSPDir);
				else
					oldJSPDir.delete();
			}
			
			if(!jspDirectory.renameTo(oldJSPDir)) {
				String msg = localStrings.getString(
						"enterprise.deployment.backend.directory_rename_error",
						jspDirectory.getPath(),
						oldJSPDir.getPath() );
				throw new IASDeploymentException( msg );
			}
		}
                if(FileUtils.safeIsDirectory(xmlDirectory))
                {
                        oldXMLDir = new File(xmlDirectory.getPath() + "_old");

                        if(oldXMLDir.exists())
                        {
                                // what if there is a regular file with the name we need?  Let's check...

                                if(FileUtils.safeIsDirectory(oldXMLDir))
                                        FileUtils.whack(oldXMLDir);
                                else
                                        oldXMLDir.delete();
                        }

                        if( ! FileUtils.renameFile(xmlDirectory, oldXMLDir)) {
                                String msg = localStrings.getString(
                                                "enterprise.deployment.backend.directory_rename_error",
                                                xmlDirectory.getPath(),
                                                oldXMLDir.getPath() );
                                throw new IASDeploymentException( msg );
                        }
                }

                if (FileUtils.safeIsDirectory(jwsDirectory)) { 
                    if (jwsDirectory.list().length > 0) {
                        // warning if the directory exist and has some contents
                        DeploymentStatus jwsStatus = new DeploymentStatus(
                            request.getCurrentDeploymentStatus());
                        jwsStatus.setStageStatus(DeploymentStatus.WARNING);
                        jwsStatus.setStageStatusMessage(localStrings.getString(
                            "enterprise.deployment.backend.jws_redeploy", 
                            jwsDirectory.getPath())); 

                    }
                    oldJWSDir = new File(jwsDirectory.getPath() + "_old");

                    if(oldJWSDir.exists())
                    {
                        // what if there is a regular file with the name we need?                          // Let's check...

                        if(FileUtils.safeIsDirectory(oldJWSDir)) {
                            FileUtils.whack(oldJWSDir);
                        }
                        else {
                            oldJWSDir.delete();
                        }
                    }

                    if( ! FileUtils.renameFile(jwsDirectory, oldJWSDir)) {
                        String msg = localStrings.getString(
                            "enterprise.deployment.backend.directory_rename_error",
                            jwsDirectory.getPath(),
                            oldJWSDir.getPath() );
                            throw new IASDeploymentException( msg );
                    }
                }
	}

	///////////////////////////////////////////////////////////////////////////

	
	private String				failureMessage	= "\n*********************\n****Redeployment Failed -- rolled back redeployment";
	private String				successMessage	= "\n*********************\n****Redeployment Successful for ";
	private	File				oldAppDir						= null;
	private	File				oldStubsDir						= null;
	private	File				oldJSPDir						= null;
	private	File				oldXMLDir						= null;
	private	File				oldJWSDir						= null;
	private	File				originalAppDir					= null;
	//private String				oldRegisteredLocation			= null;
	private boolean				appdirWasRenamed				= false;
	private boolean				appWasUnregistered				= false;
	private static StringManager localStrings =
        StringManager.getManager( AppReDeployer.class );
}
