/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.backend;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.jar.Attributes;
import java.util.jar.Attributes.Name;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.List;
import java.util.logging.Level;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;
import java.util.zip.ZipException;
import javax.enterprise.deploy.shared.ModuleType;

import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.archivist.AppClientArchivist;
import com.sun.enterprise.deployment.archivist.Archivist;
import com.sun.enterprise.deployment.archivist.ArchivistFactory;
import com.sun.enterprise.deployment.deploy.shared.AbstractArchive;
import com.sun.enterprise.deployment.interfaces.ClientJarMaker;
import com.sun.enterprise.deployment.RootDeploymentDescriptor;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.util.shared.ArchivistUtils;
import com.sun.enterprise.util.zip.ZipItem;

/**
 * This class is responsible for creating an appclient jar file that
 * will be used by the appclient container to run the appclients for
 * the deployed application.
 *
 * @author deployment dev team
 */
class ClientJarMakerUtils {

    static void populateStubs(AbstractArchive target, ZipItem[] stubs) 
        throws IOException {
        Set elements = new HashSet();
        for (ZipItem item : stubs) {
            if (elements.contains(item.getName())) {
                continue;
            }
            elements.add(item.getName());
            OutputStream os = null;
            InputStream is = null;
            try {
                os = target.putNextEntry(item.getName());
                is = new BufferedInputStream(new FileInputStream(item.getFile()));
                ArchivistUtils.copyWithoutClose(is, os);
            } finally {
                if (is != null) {
                    is.close();
                }
                if (os != null) {
                    target.closeEntry();
                }
            }
        }
    }

    static void populateModuleJar(AbstractArchive original, 
            AbstractArchive generated, AbstractArchive target) 
            throws IOException {

        // Exclude the client jar file we are trying to generate
        String excludeFileName = target.getArchiveUri().substring(
            target.getArchiveUri().lastIndexOf(File.separatorChar)+1);
        Set excludeList = new HashSet();
        excludeList.add(excludeFileName); 

        // Copy all the generated content first.  Note that this
        // needs to be done before copying the original content
        // as any duplicates are otherwise ignored.
        // Also the ClientJarMaker does not have special knowledge
        // on what's modified.  that burden is on the deployment process to
        // make sure all generated content are available in the generated/xml
        if (generated != null) {
            copyArchive(generated, target, excludeList);
        }

        // preserve all entries in the original appclient jar
        copyArchive(original, target, excludeList);

        // copy manifest file since it does not appear in the list of files
        copy(original, target, JarFile.MANIFEST_NAME);
    }

    static void copyDeploymentDescriptors(
        Archivist archivist, AbstractArchive original, 
        AbstractArchive generated, AbstractArchive target) 
        throws IOException {

        AbstractArchive source = (generated == null) ? original : generated;
        //standard dd
        copy(source, target, 
             archivist.getStandardDDFile().getDeploymentDescriptorPath());
        //runtime dd
        copy(source, target, 
             archivist.getConfigurationDDFile().getDeploymentDescriptorPath());
    }

    /**
     * This method finds all the library jars needed by the appclient.
     * Note that instead of inspecting the relevant manifest files and retrieve
     * the jars recursively, we will include all jar files found in the module
     * sans anything that is included in non-ear Java EE module.  This is done
     * to avoid slowing down deployment performance.
     */
    static List<String> getLibraryEntries(
        Application app, AbstractArchive appSource) 
            throws IOException {

        File appArchive = new File(appSource.getArchiveUri());
        Vector<String> excludeList = new Vector();
        for (Iterator modules = app.getModules(); modules.hasNext();) {
            ModuleDescriptor md = (ModuleDescriptor) modules.next();
            String moduleRoot = appArchive.getAbsolutePath() + File.separator +
                            FileUtils.makeFriendlyFilename(md.getArchiveUri());
            excludeList.add(moduleRoot);
        }
    
        Vector<String> libraries = new Vector();
        //start with the top level directory.
        File[] files = appArchive.listFiles();
        for (File file : files) {
            if (!file.isDirectory() && file.getName().endsWith(".jar")) {
                //for backward compatibility, keep all top level jars
                libraries.add(file.getName());
            } else if (file.isDirectory()) {
                //exlude any sub-module directory (so we are not including
                //any jar files from the web module)
                if (!isExcluded(file.getAbsolutePath(), excludeList)) {
                    getLibraryEntries(
                        appArchive.getAbsolutePath(),file, libraries);
                }
            } else {
                //ignore the rest
            }
        }
        
        if (DeploymentLogger.get().isLoggable(Level.FINEST)) {
            for (String lib : libraries) {
                DeploymentLogger.get().fine(
                    "Adding to the appclient jar, library [" + lib + "]");
            }
        }
        return libraries;
    }

    private static boolean isExcluded(String path, List excludeList) {
        for (Iterator it = excludeList.iterator(); it.hasNext();) {
            String exclude = (String) it.next();
            if (path.startsWith(exclude)) {
                return true;
            }
        }
        return false;
    }

    private static void getLibraryEntries(
        String topDir, File directory, List libraries) {
        File[] files = directory.listFiles();
        for (File file : files) {
            if (!file.isDirectory() && file.getName().endsWith(".jar")) {
                String entryName = 
                    file.getAbsolutePath().substring(topDir.length()+1);
                libraries.add(entryName);
            } else if (file.isDirectory()) {
                getLibraryEntries(topDir, file, libraries);
            } else {
                //ignore the test
            }
        }
    }

    /**
     * copy the entryName element from the source abstract archive into
     * the target abstract archive
     */
    static void copy(
        AbstractArchive source, AbstractArchive target, String entryName)
        throws IOException {
            
        InputStream is=null;
        OutputStream os=null;
        try {
            is = source.getEntry(entryName);
            if (is != null) {
                try {
                    os = target.putNextEntry(entryName);
                } catch(ZipException ze) {
                    // this is a duplicate...
                    return;
                }
                ArchivistUtils.copyWithoutClose(is, os);
            }
        } catch (IOException ioe) {
            throw ioe;
        } finally {
            IOException closeEntryIOException = null;
            if (os!=null) {
                try {
                    target.closeEntry();
                } catch (IOException ioe) { 
                    closeEntryIOException = ioe;
                }
            }
            if (is!=null) {
                is.close();
            }
 
            if (closeEntryIOException != null) {
                throw closeEntryIOException;
            } 
        }
    }

    private static void copyArchive (
        AbstractArchive source, AbstractArchive target, Set excludeList) {
        for (Enumeration e = source.entries();e.hasMoreElements();) {
            String entryName = String.class.cast(e.nextElement());
            if (excludeList.contains(entryName)) {
                continue;
            }
            try {
                copy(source, target, entryName);
            } catch(IOException ioe) {
                // duplicate, we ignore
            }
        }
    }
}
