/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.resource;

import java.util.HashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.resource.ResourceException;
import javax.resource.spi.ConnectionRequestInfo;
import javax.resource.spi.ManagedConnection;
import javax.resource.spi.ManagedConnectionFactory;
import javax.resource.spi.ValidatingManagedConnectionFactory;
import javax.security.auth.Subject;

import com.sun.enterprise.PoolManager;
import com.sun.enterprise.deployment.ConnectorDescriptor;
import com.sun.logging.LogDomains;


/**
 * An abstract implementation of the <code>ResourceAllocator</code> interface
 * that houses all the common implementation(s) of the various connector allocators.
 * All resource allocators except <code>BasicResourceAllocator</code> extend this 
 * abstract implementation
 * @author Sivakumar Thyagarajan
 */
public abstract class AbstractConnectorAllocator 
                            implements ResourceAllocator {

    protected PoolManager poolMgr;
    protected ResourceSpec spec;
    protected ConnectionRequestInfo reqInfo;
    protected Subject subject;
    protected ManagedConnectionFactory mcf;
    protected ConnectorDescriptor desc;
    protected ClientSecurityInfo info;
    
    protected static Logger _logger = LogDomains.getLogger(LogDomains.RSR_LOGGER);

    public AbstractConnectorAllocator() {
    }

    public AbstractConnectorAllocator(PoolManager poolMgr,
                    ManagedConnectionFactory mcf,
                    ResourceSpec spec,
                    Subject subject,
                    ConnectionRequestInfo reqInfo,
                    ClientSecurityInfo info,
                    ConnectorDescriptor desc) {
        this.poolMgr = poolMgr;
        this.mcf = mcf;
        this.spec = spec;
        this.subject = subject;
        this.reqInfo = reqInfo;
        this.info = info;
        this.desc = desc;
        
    }

    public Set getInvalidConnections(Set connectionSet)
                                throws ResourceException {
        if(mcf instanceof ValidatingManagedConnectionFactory){
            return ((ValidatingManagedConnectionFactory)this.mcf).
                                    getInvalidConnections(connectionSet);
        }
        return null;
    }

    public boolean isConnectionValid( ResourceHandle h ) 
    {
         HashSet conn = new HashSet();
         conn.add( h.getResource() );
         Set invalids = null;
         try {
             invalids = getInvalidConnections( conn );
         } catch( ResourceException re ) {
             //ignore and continue??
         }
         
	 if ( (invalids != null && invalids.size() > 0)  ||
	         h.hasConnectionErrorOccurred() ) {
	     return false;
	 } 

	 return true;
    }

    public void destroyResource(ResourceHandle resourceHandle)
        throws PoolingException {
        throw new UnsupportedOperationException();
    }

    public void fillInResourceObjects(ResourceHandle resourceHandle)
        throws PoolingException {
        throw new UnsupportedOperationException();
    }



    public boolean supportsReauthentication() {
        return this.desc.supportsReauthentication();
    }

    public boolean isTransactional() {
        return true;
    }

    public void cleanup(ResourceHandle h) throws PoolingException {
        try {
            ManagedConnection mc = (ManagedConnection) h.getResource();
            mc.cleanup();
        } catch (Exception ex) {
            _logger.log(Level.WARNING, "managed_con.cleanup-failed", ex);
            throw new PoolingException(ex.toString(), ex);
        }
    }

    public boolean matchConnection(ResourceHandle h) {
        Set set = new HashSet();
        set.add(h.getResource());
        try {
            ManagedConnection mc = 
                mcf.matchManagedConnections(set, subject, reqInfo);
            return (mc != null);
        } catch (ResourceException ex) {
            return false;
        }
    }

    public void closeUserConnection(ResourceHandle resource) throws PoolingException {
    
        try {
            ManagedConnection mc = (ManagedConnection) resource.getResource();
            mc.cleanup();
        } catch (ResourceException ex) {
            throw new PoolingException(ex);
        }
    }

    public boolean shareableWithinComponent() {
        return false;
    }

    public Object getSharedConnection(ResourceHandle h) 
                                    throws PoolingException {
        throw new UnsupportedOperationException();
    }

}
