
/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.connector.grizzly;

import java.io.InputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;

/**
 * Stream wrapper around a <code>ByteBuffer</code>
 */
public class ByteBufferInputStream extends InputStream {

    /**
     * The wrapped <code>ByteBuffer</code<
     */
    private ByteBuffer byteBuffer;

    
    /**
     * The <code>SelectionKey</code> used by this stream.
     */
    private SelectionKey key = null;
    
    
    /**
     * The time to wait before timing out when reading bytes
     */
    private int readTimeout = 15000;
    
    
    /**
     * Number of times to retry before return EOF
     */
    protected int readTry = 2;
    
    // ------------------------------------------------- Constructor -------//
    
    
    public ByteBufferInputStream () {
    }

    
    public ByteBufferInputStream (final ByteBuffer byteBuffer) {
        this.byteBuffer = byteBuffer;
    }

    // ---------------------------------------------------------------------//
    
    
    /**
     * Set the wrapped <code>ByteBuffer</code>
     * @param byteBuffer The wrapped byteBuffer
     */
    public void setByteBuffer (final ByteBuffer byteBuffer) {
        this.byteBuffer = byteBuffer;
    }

    
    /**
     * Return the available bytes 
     * @return the wrapped byteBuffer.remaining()
     */
    public int available () {
        return (byteBuffer.remaining());
    }

    
    /**
     * Close this stream. 
     */
    public void close () {
    }

    
    /**
     * Return true if mark is supported.
     */
    public boolean markSupported () {
        return (true);
    }

    
    /**
     * Read the first byte from the wrapped <code>ByteBuffer</code>.
     */
    public int read () {
        if (!byteBuffer.hasRemaining()){
            int eof = 0;
            for (int i=0; i < readTry; i++) {
                eof = doRead();
                if ( eof != 0 ){
                    break;
                } 
            }
        }
        return (byteBuffer.hasRemaining() ? (byteBuffer.get () & 0xff): -1);
     }

    
    /**
     * Read the bytes from the wrapped <code>ByteBuffer</code>.
     */
    public int read (byte[] b) {
        return (read (b, 0, b.length));
    }

    
    /**
     * Read the first byte of the wrapped <code>ByteBuffer</code>.
     */
    public int read (byte[] b, int offset, int length) {
        if (!byteBuffer.hasRemaining()) {
            int eof = 0;
            for (int i=0; i < readTry; i++) {
                eof = doRead();
                
                if ( eof != 0 ){
                    break;
                } 
            }
            
            if (eof <= 0){
                return -1;
            }
        }
 
        if (length > byteBuffer.remaining()) {
            length = byteBuffer.remaining();
        }
        byteBuffer.get(b, offset, length);
         
        return (length);
    }
    
    
    /**
     * Recycle this object.
     */ 
    public void recycle(){
        byteBuffer = null;   
    }
    
    
    /**
     * Set the <code>SelectionKey</code> used to reads bytes.
     */
    public void setSelectionKey(SelectionKey key){
        this.key = key;
    }
    
    
    /**
     * Read bytes using the read <code>ReadSelector</code>
     */
    private int doRead(){        
        if ( key == null ) return -1;
        
        byteBuffer.clear();
        int count = 1;
        int byteRead = 0;
        Selector readSelector = null;
        SelectionKey tmpKey = null;
        
        try{
            SocketChannel socketChannel = (SocketChannel)key.channel();
            while (count > 0){
                count = socketChannel.read(byteBuffer);
                byteRead += count;
            }            
            
            if ( byteRead == 0 ){
                readSelector = SelectorFactory.getSelector();

                if ( readSelector == null ){
                    return 0;
                }
                count = 1;
                tmpKey = socketChannel
                        .register(readSelector,SelectionKey.OP_READ);               
                tmpKey.interestOps(tmpKey.interestOps() | SelectionKey.OP_READ);
                int code = readSelector.select(readTimeout);
                tmpKey.interestOps(
                    tmpKey.interestOps() & (~SelectionKey.OP_READ));

                if ( code == 0 ){
                    return 0; // Return on the main Selector and try again.
                }

                while (count > 0){
                    count = socketChannel.read(byteBuffer);
                    byteRead += count;
                }
            }
        } catch (Throwable t){
            return -1;
        } finally {
            if (tmpKey != null)
                tmpKey.cancel();

            if ( readSelector != null){
                // Bug 6403933
                try{
                    readSelector.selectNow();
                } catch (IOException ex){
                    ;
                }
                SelectorFactory.returnSelector(readSelector);
            }
        }
        byteBuffer.flip();
        return byteRead;
    } 
}

