/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 *  Copyright 1999-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package com.sun.enterprise.web.connector.grizzly;

import com.sun.enterprise.web.connector.grizzly.async.DefaultAsyncHandler;
import com.sun.enterprise.web.connector.grizzly.async.DefaultAsyncExecutor;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;

import java.net.InetAddress;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Logger;
import java.util.logging.Level;

import org.apache.catalina.util.ServerInfo;

import org.apache.coyote.ActionCode;
import org.apache.coyote.ActionHook;
import org.apache.coyote.Adapter;
import org.apache.coyote.Processor;
import org.apache.coyote.Request;
import org.apache.coyote.RequestInfo;
import org.apache.coyote.Response;
import org.apache.coyote.http11.InternalInputBuffer;
import org.apache.coyote.http11.InternalOutputBuffer;
import org.apache.coyote.http11.InputFilter;
import org.apache.coyote.http11.OutputFilter;
import org.apache.coyote.http11.filters.ChunkedInputFilter;
import org.apache.coyote.http11.filters.ChunkedOutputFilter;
import org.apache.coyote.http11.filters.GzipOutputFilter;
import org.apache.coyote.http11.filters.IdentityInputFilter;
import org.apache.coyote.http11.filters.IdentityOutputFilter;
import org.apache.coyote.http11.filters.VoidInputFilter;
import org.apache.coyote.http11.filters.VoidOutputFilter;
import org.apache.coyote.http11.filters.BufferedInputFilter;
import org.apache.coyote.tomcat5.CoyoteAdapter;
import org.apache.coyote.tomcat5.CoyoteRequest;  

import org.apache.tomcat.util.buf.Ascii;
import org.apache.tomcat.util.buf.ByteChunk;
import org.apache.tomcat.util.buf.HexUtils;
import org.apache.tomcat.util.buf.MessageBytes;
import org.apache.tomcat.util.http.FastHttpDateFormat;
import org.apache.tomcat.util.http.MimeHeaders;
import org.apache.tomcat.util.net.SSLImplementation;
import org.apache.tomcat.util.net.SSLSupport;
import com.sun.enterprise.web.connector.grizzly.handlers.NoParsingHandler;

import com.sun.org.apache.commons.modeler.Registry;

import javax.management.ObjectName;

import java.nio.channels.Selector;

/**
 * Process HTTP request. This class is based on
 * <code>org.apache.coyote.http11.Http11Processor</code>
 *
 * @author Jean-Francois Arcand
 */
public class ProcessorTask extends TaskBase implements Processor, ActionHook {

    /**
     * Associated adapter.
     */
    protected Adapter adapter = null;


    /**
     * Request object.
     */
    protected Request request = null;


    /**
     * Response object.
     */
    protected Response response = null;


    /**
     * Input.
     */
    protected InternalInputBuffer inputBuffer = null;


    /**
     * Output.
     */
    protected InternalOutputBuffer outputBuffer = null;


    /**
     * State flag.
     */
    protected boolean started = false;


    /**
     * Error flag.
     */
    protected boolean error = false;


    /**
     * Keep-alive.
     */
    protected boolean keepAlive = true;
    
    
    /**
     * Connection: value
     */
    protected boolean connectionHeaderValue = true;

    /**
     * HTTP/1.1 flag.
     */
    protected boolean http11 = true;


    /**
     * HTTP/0.9 flag.
     */
    protected boolean http09 = false;


    /**
     * Content delimitator for the request (if false, the connection will
     * be closed at the end of the request).
     */
    protected boolean contentDelimitation = true;


    /**
     * SSL information.
     */
    protected SSLSupport sslSupport;


    /**
     * Socket associated with the current connection.
     */
    protected Socket socket;


    /**
     * Remote Address associated with the current connection.
     */
    protected String remoteAddr = null;


    /**
     * Remote Host associated with the current connection.
     */
    protected String remoteHost = null;
    
    
    /**
     * Local Host associated with the current connection.
     */
    protected String localName = null;
        
    
    /**
     * Local port to which the socket is connected
     */
    protected int localPort = -1;
    
    
    /**
     * Remote port to which the socket is connected
     */
    protected int remotePort = -1;
    
    
    /**
     * The local Host address.
     */
    protected String localAddr = null; 


    /**
     * Maximum timeout on uploads. 5 minutes as in Apache HTTPD server.
     */
    protected int timeout = 300000;


    /**
     * Max post size.
     */
    protected int maxPostSize = 2 * 1024 * 1024;


    /**
     * Host name (used to avoid useless B2C conversion on the host name).
     */
    protected char[] hostNameC = new char[0];


    /**
     * The <code>taskContext</code> associated with this object. The 
     * <code>taskContext</code> contains information about the current
     * connection.
     */
    protected TaskContext taskContext;    
    
    
    /**
     * The <code>TaskEvent</code> associated with this task.
     */
    protected TaskEvent<TaskContext> taskEvent;

    
    /**
     * The underlying SSL implementation used.
     */
    protected SSLImplementation sslImplementation = null;
    
    
    /**
     * Are we using NIO non blocking mode?
     */
    protected boolean usingNioNonBlocking = true;
    
    
    /**
     * Has the request associated with this <code>ProcessorTask</code> been
     * registered with the <code>RequestGroupInfo</code>
     */
    private boolean hasRequestInfoRegistered = false;
    
    
    /**
     * Default HTTP header buffer size.
     */
    protected int maxHttpHeaderSize = Constants.DEFAULT_HEADER_SIZE;

    
    /**
     * The number of requests <code>ProcessorTask</code> has proceeded.
     */
    protected static int requestCount;

    
    /**
     * The input request buffer size.
     */
    private int requestBufferSize = Constants.DEFAULT_REQUEST_BUFFER_SIZE;


    /**
     * ObjectName under which this <code>ProcessorTask</code> will be
     * JMX-registered if monitoring has been turned on
     */
    private ObjectName oname;

    
    /**
     * Disable all keep-Alive functionality. Usualy used with NIO Mode 3
     */
    protected boolean useAlternateKeepAlive = false;
    
    
    /**
     * Allow client of this class to force connection closing.
     */
    protected boolean dropConnection = false;
    
    
    /**
     * Maximum number of Keep-Alive requests to honor. This value is used 
     * only when NIO blocking mode is enabled.
     */
    protected int maxKeepAliveRequests = Constants.DEFAULT_MAX_KEEP_ALIVE;
    
    
    /**
     * The current keep-alive count left before closing the connection.
     */
    protected int keepAliveLeft;
  
     
    /**
     * The handler used by this <code>Task</code> to manipulate the request.
     */
    protected Handler handler;
    
    
    /**
     * The default response-type
     */
    private String defaultResponseType = 
            org.apache.coyote.tomcat5.Constants.DEFAULT_RESPONSE_TYPE;
     
    
    /**
     * The forced response-type
     */
    private String forcedResponseType = 
            org.apache.coyote.tomcat5.Constants.FORCED_RESPONSE_TYPE;     
    
    
    /**
     * Is asynchronous mode enabled?
     */
    private boolean asyncExecution = false;
    
    
    /**
     * The code>RequestInfo</code> used to gather stats.
     */
    private RequestInfo requestInfo;
    
    
    /**
     * The <code>OutputBuffer</code> used when NIO non-blocking is enabled.
     */
    private SocketChannelOutputBuffer channelOutputBuffer;
    
    
    /**
     * When the asynchronous mode is enabled, the execution of this object
     * will be delegated to the <code>AsyncHandler</code>
     */
    private AsyncHandler asyncHandler;
    
    
    /**
     * Is SSL enabled.
     */
    private boolean sslEnabled = sslImplementation == null ? false: true;
    
        
    // ----------------------------------------------------- Constructor ---- //

    public ProcessorTask(){
        this(true, true);
    }
        
    public ProcessorTask(boolean useNio, boolean init){    
        
        usingNioNonBlocking = useNio;
        type = PROCESSOR_TASK;
        if (init) {
            initialize();
        }
    }

    
    /**
     * Initialize the stream and the buffer used to parse the request.
     */
    protected void initialize(){
        started = true;   
        request = new Request();

        response = new Response();
        response.setHook(this);
        
        inputBuffer = new InternalInputBuffer(request,requestBufferSize);
 
        if ( usingNioNonBlocking ){
            outputBuffer = new SocketChannelOutputBuffer(response,
                                                         maxHttpHeaderSize);
        } else { 
            outputBuffer = new InternalOutputBuffer(response,
                                                    maxHttpHeaderSize);
        }
        request.setInputBuffer(inputBuffer);
       
        response.setOutputBuffer(outputBuffer);
        request.setResponse(response);

        initializeFilters();

    }


    // ----------------------------------------------------- Thread run ---- //
    
    
     /**
     * Execute the HTTP request by parsing the header/body,
     * and then by delegating the process to the Catalina container.
     */
    public void doTask() throws IOException{
        boolean sslEnabled = sslImplementation == null ? false: true;
        usingNioNonBlocking = usingNioNonBlocking && !sslEnabled;
        try {
            if ( usingNioNonBlocking ) {          
                process(taskContext.getInputStream(),
                        taskContext.getOutputStream());
            } else {
                process(socket.getInputStream(),socket.getOutputStream());
            } 
        } catch(Throwable ex){
            SelectorThread.logger().log(Level.FINE,
                    "processorTask.errorProcessingRequest", ex);
        } finally {
            terminateProcess();        
        }
    }

 
     // --------------------------------------------------------- TaskEvent ---// 
        
     
    public void taskEvent(TaskEvent event){
        if ( event.getStatus() == TaskEvent.START) {
            taskContext = (TaskContext)event.attachement();
            if (  taskEvent == null ) {
                taskEvent = new TaskEvent<TaskContext>();
            }
            
            taskEvent.attach(taskContext);

            if ( !asyncExecution ) {
                execute();
            } else {
                asyncHandler.handle(this);
            }
        }
    }

    // -------------------------------------------------------------------- //

    
    /**
     * Pre process the request by decoding the request line and the header.
     */ 
    public void preProcess() throws Exception {
        if ( usingNioNonBlocking && !sslEnabled ) {
            preProcess(taskContext.getInputStream(),
                       taskContext.getOutputStream());
        } else {
            preProcess(socket.getInputStream(), 
                       socket.getOutputStream());            
        }
    }
    
    
    /**
     * Pre process the request by decoding the request line and the header.
     * @param input the InputStream to read bytes
     * @param output the OutputStream to write bytes
     */     
    public void preProcess(InputStream input, OutputStream output)
                                                            throws Exception {
        
        // Make sure this object has been initialized.
        if ( !started ){
            initialize();
        }
        
        if (isMonitoringEnabled()){
            adapter.
                fireAdapterEvent(Adapter.CONNECTION_PROCESSING_STARTED,
                    request.getRequestProcessor());
        }

      
        if (sslImplementation != null) {
            sslSupport = sslImplementation.getSSLSupport(socket);
        }
                              
        if( selectorThread.getDomain() != null 
                && isMonitoringEnabled() 
                && !hasRequestInfoRegistered ) {
            registerMonitoring();
        } else if (!isMonitoringEnabled() && hasRequestInfoRegistered) {
            unregisterMonitoring();
        } 
        
        if (isMonitoringEnabled()) {
            requestInfo = request.getRequestProcessor();
            requestInfo.setWorkerThreadID(Thread.currentThread().getId());
        }

        // Setting up the I/O
        inputBuffer.setInputStream(input);
        if ( usingNioNonBlocking ) {
            channelOutputBuffer = ((SocketChannelOutputBuffer)outputBuffer);
            channelOutputBuffer.setChannel((SocketChannel)key.channel());
        } else {
            outputBuffer.setOutputStream(output);  
        }
        
        // Set the remote address
        remoteAddr = null;
        remoteHost = null;
        localName = null;
        localAddr = null;
        remotePort = -1;
        localPort = -1;
        connectionHeaderValue = true;
        
        // Error flag
        error = false;
        keepAlive = true;

        keepAliveLeft = maxKeepAliveRequests;
        int soTimeout = socket.getSoTimeout();
        if (request.getServerPort() == 0) {
            request.setServerPort(selectorThread.getPort());
        }        
    }

    
    /**
     * Process an HTTP request using a non blocking <code>socket</code>
     */      
    public boolean processNonBlocked() throws Exception {
        if ( usingNioNonBlocking && !sslEnabled ) {
            return processNonBlocked(taskContext.getInputStream(),
                                     taskContext.getOutputStream());
        } else {
            return processNonBlocked(socket.getInputStream(), 
                                     socket.getOutputStream());            
        }
    }
    
    
    /**
     * Process an HTTP request using a non blocking <code>socket</code>
     * @param input the InputStream to read bytes
     * @param output the OutputStream to write bytes
     */         
    public boolean processNonBlocked(InputStream input, OutputStream output)
                                                            throws Exception{        
        boolean exitWhile = parseRequest(input,output,false);
        if ( exitWhile ) return exitWhile;
        invokeAdapter();
        postResponse();  
        return error;
    }
    
    
    /**
     * Process an HTTP request using a blocking <code>socket</code>
     */    
    public void processBlocked() throws Exception {
        if ( usingNioNonBlocking && !sslEnabled ) {
            processBlocked(taskContext.getInputStream(),
                              taskContext.getOutputStream());
        } else {
            processBlocked(socket.getInputStream(), 
                              socket.getOutputStream());            
        }
    }
    
    
    /**
     * Process an HTTP request using a blocking <code>socket</code>
     * @param input the InputStream to read bytes
     * @param output the OutputStream to write bytes
     */      
    public void processBlocked(InputStream input, OutputStream output)
                                                            throws Exception {
        boolean keptAlive = false;    
        while (started && !error && keepAlive) {                

            boolean exitWhile = parseRequest(input,output,keptAlive);
            if ( exitWhile ) break;
                        
            invokeAdapter();
            postResponse();
        }       
    }

    
    /**
     * Prepare and post the response.
     * @param input the InputStream to read bytes
     * @param output the OutputStream to write bytes
     */       
    public void postResponse() throws Exception{
        // Finish the handling of the request
        try {
             inputBuffer.endRequest();
        } catch (IOException e) {
            error = true;
        } catch (Throwable t) {
            SelectorThread.logger().log(Level.SEVERE,
                    "processorTask.errorFinishingRequest", t);
            // 500 - Internal Server Error
            response.setStatus(500);
            error = true;
        }
        try {
            outputBuffer.endRequest();
        } catch (IOException e) {
            error = true;
        } catch (Throwable t) {
            SelectorThread.logger().log(Level.SEVERE,
                    "processorTask.errorFinishingResponse", t);
            error = true;
        }

        // If there was an error, make sure the request is counted as
        // and error, and update the statistics counter
        if (error) {
            response.setStatus(500);
        }

        if (isMonitoringEnabled()) {
            request.updateCounters();
            adapter.fireAdapterEvent(Adapter.REQUEST_PROCESSING_COMPLETED, 
                    request.getRequestProcessor());              
        }

        // Next request
        inputBuffer.nextRequest();
        outputBuffer.nextRequest();
    }
    
    
    /**
     * Invoke the <code>Adapter</code>, which usualy invoke the Servlet
     * Container.
     */
    public void invokeAdapter(){
        // Process the request in the adapter
        if (!error) {
            try {
                adapter.service(request, response);
                // Handle when the response was committed before a serious
                // error occurred.  Throwing a ServletException should both
                // set the status to 500 and set the errorException.
                // If we fail here, then the response is likely already 
                // committed, so we can't try and set headers.
                if(keepAlive && !error) { // Avoid checking twice.
                    error = response.getErrorException() != null ||
                            statusDropsConnection(response.getStatus());
                }

            } catch (InterruptedIOException e) {
                error = true;
            } catch (Throwable t) {
                SelectorThread.logger().log(Level.SEVERE,
                        "processorTask.serviceError", t);
                // 500 - Internal Server Error
                response.setStatus(500);
                error = true;
            }
        }
    }
    
    
    /**
     * Parse the request line and the http header.
     */
    public void parseRequest() throws Exception {
        if ( usingNioNonBlocking && !sslEnabled ) {
            parseRequest(taskContext.getInputStream(),
                         taskContext.getOutputStream(), true);
        } else {
            parseRequest(socket.getInputStream(),socket.getOutputStream(),true);            
        }
    }    
    
    
    /**
     * Parse the request line and the http header.
     * @param input the InputStream to read bytes
     * @param output the OutputStream to write bytes
     */    
    public boolean parseRequest(InputStream input, OutputStream output,
            boolean keptAlive) throws Exception {
        
        // Parsing the request header
        try { 
            if (isMonitoringEnabled()){
                adapter.fireAdapterEvent(Adapter.REQUEST_PROCESSING_STARTED, 
                        request.getRequestProcessor());
            }

            inputBuffer.parseRequestLine();
            if (isMonitoringEnabled()) {
                request.getRequestProcessor().setRequestCompletionTime(0);
            }

            request.setStartTime(System.currentTimeMillis());
            if ( handler.handle(request,Handler.REQUEST_LINE_PARSED)
                    == Handler.BREAK){
                return true;
            }

            keptAlive = true;
            inputBuffer.parseHeaders();
        } catch (IOException e) {
            SelectorThread.logger().log(Level.FINEST,
                    "processorTask.nonBlockingError", e);
            error = true;
            keepAlive = false;
            return true;
        } catch (Throwable t) {
            SelectorThread.logger().log(Level.SEVERE,
                    "processorTask.nonBlockingError", t);
            // 400 - Bad Request
            response.setStatus(400);
            error = true;
        }

        // Setting up filters, and parse some request headers
        try {
            prepareRequest();
        } catch (Throwable t) {
            SelectorThread.logger().log(Level.FINE,
                    "processorTask.createRequestError", t);
            // 400 - Internal Server Error
            response.setStatus(400);
            error = true;
        }

        if (!usingNioNonBlocking){
            if (maxKeepAliveRequests > 0 && --keepAliveLeft == 0)
                keepAlive = false;
        }  
        return false;
    }
    
    
    /**
     * Post process the http request, after the response has been
     * commited.
     */      
    public void postProcess() throws Exception {
        if ( usingNioNonBlocking && !sslEnabled ) {
            postProcess(taskContext.getInputStream(),
                        taskContext.getOutputStream());
        } else {
            postProcess(socket.getInputStream(),socket.getOutputStream());            
        } 
    }      
    
    
    /**
     * Post process the http request, after the response has been
     * commited.
     */      
    public void postProcess(InputStream input, OutputStream output)
                                                            throws Exception {
        if (!recycle){
            started = false;
            inputBuffer = null;
            outputBuffer = null;
            response = null;
            if (isMonitoringEnabled()) {
                request.getRequestProcessor().setWorkerThreadID(0);
                adapter.fireAdapterEvent(Adapter.CONNECTION_PROCESSING_COMPLETED, 
                        request.getRequestProcessor());              
            }
            request = null;
        } else {
            inputBuffer.recycle();
            outputBuffer.recycle();
        }

        // Recycle ssl info
        sslSupport = null;

        if (error){
            keepAlive = false;
            connectionHeaderValue = false;
        }
    }
    

    /**
     * Notify the <code>TaskListener</code> that the request has been 
     * fully processed.
     */
    public void terminateProcess(){
        if ( error && usingNioNonBlocking) {
            taskEvent.setStatus(TaskEvent.ERROR);
        } else {
            taskEvent.setStatus(TaskEvent.COMPLETED);
        }
        fireTaskEvent(taskEvent); 
    }
    
    
    // -------------------------------------------------------------------- //
    
    
    /**
     * Process pipelined HTTP requests using the specified input and output
     * streams.
     * 
     * @param input stream from which the HTTP requests will be read
     * @param output stream which will be used to output the HTTP
     * responses
     * @return true is an error occured.
     * @throws Exception error during an I/O operation
     */
    public boolean process(InputStream input, OutputStream output)
            throws Exception {
        preProcess(input,output);      
        if ( !usingNioNonBlocking ) {
            processBlocked(input,output);
        } else {
            processNonBlocked(input,output);
        }
        postProcess(input,output);
        return keepAlive;
    }
    
    
    /** 
     * Get the request associated with this processor.
     */
    public Request getRequest() {
        return request;
    }
    

    // ----------------------------------------------------- ActionHook Methods


    /**
     * Send an action to the connector.
     * 
     * @param actionCode Type of the action
     * @param param Action parameter
     */
    public void action(ActionCode actionCode, Object param) {

        if (actionCode == ActionCode.ACTION_COMMIT) {
            // Commit current response

            if (response.isCommitted())
                return;
                        
            // Validate and write response headers
            prepareResponse();
            try {
                outputBuffer.commit();
            } catch (IOException ex) {
                 SelectorThread.logger().log(Level.FINEST,
                        "processorTask.nonBlockingError", ex);               
                // Set error flag
                error = true;
            }

        } else if (actionCode == ActionCode.ACTION_ACK) {

            // Acknowlege request

            // Send a 100 status back if it makes sense (response not committed
            // yet, and client specified an expectation for 100-continue)

            if ((response.isCommitted()) || (!http11))
                return;

            MessageBytes expectMB = request.getMimeHeaders().getValue("expect");
            if ((expectMB != null)
                && (expectMB.indexOfIgnoreCase("100-continue", 0) != -1)) {
                try {
                    outputBuffer.sendAck();
                } catch (IOException e) {
                    // Set error flag
                    error = true;
                }
            }

        } else if (actionCode == ActionCode.ACTION_CLOSE) {
            // Close

            // End the processing of the current request, and stop any further
            // transactions with the client

            try {
                outputBuffer.endRequest();
            } catch (IOException e) {
                SelectorThread.logger().log(Level.FINEST,
                        "processorTask.nonBlockingError", e);
                // Set error flag
                error = true;
            }

        } else if (actionCode == ActionCode.ACTION_RESET) {

            // Reset response

            // Note: This must be called before the response is committed

            outputBuffer.reset();

        } else if (actionCode == ActionCode.ACTION_CUSTOM) {

            // Do nothing

        } else if (actionCode == ActionCode.ACTION_START) {

            started = true;

        } else if (actionCode == ActionCode.ACTION_STOP) {

            started = false;

        } else if (actionCode == ActionCode.ACTION_REQ_SSL_ATTRIBUTE ) {

            try {
                if (sslSupport != null) {
                    Object sslO = sslSupport.getCipherSuite();
                    if (sslO != null)
                        request.setAttribute
                            (SSLSupport.CIPHER_SUITE_KEY, sslO);
                    sslO = sslSupport.getPeerCertificateChain(false);
                    if (sslO != null)
                        request.setAttribute
                            (SSLSupport.CERTIFICATE_KEY, sslO);
                    sslO = sslSupport.getKeySize();
                    if (sslO != null)
                        request.setAttribute
                            (SSLSupport.KEY_SIZE_KEY, sslO);
                    sslO = sslSupport.getSessionId();
                    if (sslO != null)
                        request.setAttribute
                            (SSLSupport.SESSION_ID_KEY, sslO);
                }
            } catch (Exception e) {
                SelectorThread.logger().log(Level.WARNING,
                        "processorTask.errorSSL" ,e);
            }

        } else if (actionCode == ActionCode.ACTION_REQ_HOST_ADDR_ATTRIBUTE) {

            if ((remoteAddr == null) && (socket != null)) {
                InetAddress inetAddr = socket.getInetAddress();
                if (inetAddr != null) {
                    remoteAddr = inetAddr.getHostAddress();
                }   
            }
            request.remoteAddr().setString(remoteAddr);

        } else if (actionCode == ActionCode.ACTION_REQ_LOCAL_NAME_ATTRIBUTE) {
            
            if ((localName == null) && (socket != null)) {
                InetAddress inetAddr = socket.getLocalAddress();
                if (inetAddr != null) {
                    localName = inetAddr.getHostName();
                }
            }
            request.localName().setString(localName);

        } else if (actionCode == ActionCode.ACTION_REQ_HOST_ATTRIBUTE) {
            
            if ((remoteHost == null) && (socket != null)) {
                InetAddress inetAddr = socket.getInetAddress();
                if (inetAddr != null) {
                    remoteHost = inetAddr.getHostName();
                }
                
                if(remoteHost == null) {
                    if(remoteAddr != null) {
                        remoteHost = remoteAddr;
                    } else { // all we can do is punt
                        request.remoteHost().recycle();
                    }
                }
            }
            request.remoteHost().setString(remoteHost);
            
        } else if (actionCode == ActionCode.ACTION_REQ_LOCAL_ADDR_ATTRIBUTE) {
                       
            if (localAddr == null)
               localAddr = socket.getLocalAddress().getHostAddress();

            request.localAddr().setString(localAddr);
            
        } else if (actionCode == ActionCode.ACTION_REQ_REMOTEPORT_ATTRIBUTE) {
            
            if ((remotePort == -1 ) && (socket !=null)) {
                remotePort = socket.getPort(); 
            }    
            request.setRemotePort(remotePort);

        } else if (actionCode == ActionCode.ACTION_REQ_LOCALPORT_ATTRIBUTE) {
            
            if ((localPort == -1 ) && (socket !=null)) {
                localPort = socket.getLocalPort(); 
            }            
            request.setLocalPort(localPort);
       
        } else if (actionCode == ActionCode.ACTION_REQ_SSL_CERTIFICATE) {
            if( sslSupport != null) {
                /*
                 * Consume and buffer the request body, so that it does not
                 * interfere with the client's handshake messages
                 */
                InputFilter[] inputFilters = inputBuffer.getFilters();
                ((BufferedInputFilter) inputFilters[Constants.BUFFERED_FILTER])
                    .setLimit(maxPostSize);
                inputBuffer.addActiveFilter
                    (inputFilters[Constants.BUFFERED_FILTER]);
                try {
                    Object sslO = sslSupport.getPeerCertificateChain(true);
                    if( sslO != null) {
                        request.setAttribute
                            (SSLSupport.CERTIFICATE_KEY, sslO);
                    }
                } catch (Exception e) {
                    SelectorThread.logger().log(Level.WARNING,
                            "processorTask.exceptionSSLcert",e);
                }
            }
        } else if ( actionCode == ActionCode.ACTION_POST_REQUEST ) { 
            if (response.getStatus() == 200){
                try{
                    handler.handle(request,Handler.RESPONSE_PROCEEDED);
                } catch(IOException ex){
                    ; //Ignore
                }
            }      
        }
    }

    // ------------------------------------------------------ Connector Methods


    /**
     * Set the associated adapter.
     * 
     * @param adapter the new adapter
     */
    public void setAdapter(Adapter adapter) {
        this.adapter = adapter;
    }


    /**
     * Get the associated adapter.
     * 
     * @return the associated adapter
     */
    public Adapter getAdapter() {
        return adapter;
    }


    // ------------------------------------------------------ Protected Methods


    /**
     * After reading the request headers, we have to setup the request filters.
     */
    protected void prepareRequest() {

        http11 = true;
        http09 = false;
        contentDelimitation = false;
        if (sslSupport != null) {
            request.scheme().setString("https");
        }
        MessageBytes protocolMB = request.protocol();
        if (protocolMB.equals(Constants.HTTP_11)) {
            http11 = true;
            protocolMB.setString(Constants.HTTP_11);
        } else if (protocolMB.equals(Constants.HTTP_10)) {
            http11 = false;
            keepAlive = false;
            protocolMB.setString(Constants.HTTP_10);
        } else if (protocolMB.equals("")) {
            // HTTP/0.9
            http09 = true;
            http11 = false;
            keepAlive = false;
        } else {
            // Unsupported protocol
            http11 = false;
            error = true;
            // Send 505; Unsupported HTTP version
            response.setStatus(505);
        }
        
        MessageBytes methodMB = request.method();
        if (methodMB.equals(Constants.GET)) {
            methodMB.setString(Constants.GET);
        } else if (methodMB.equals(Constants.POST)) {
            methodMB.setString(Constants.POST);
        }

        MimeHeaders headers = request.getMimeHeaders();

        // Check connection header
        MessageBytes connectionValueMB = headers.getValue("connection");
        if (connectionValueMB != null) {
            ByteChunk connectionValueBC = connectionValueMB.getByteChunk();
            if (findBytes(connectionValueBC, Constants.CLOSE_BYTES) != -1) {
                keepAlive = false;
                connectionHeaderValue = false;
            } else if (findBytes(connectionValueBC, 
                                 Constants.KEEPALIVE_BYTES) != -1) {
                keepAlive = true;
                connectionHeaderValue = true;
            }
        }


        // Check for a full URI (including protocol://host:port/)
        ByteChunk uriBC = request.requestURI().getByteChunk();
        if (uriBC.startsWithIgnoreCase("http", 0)) {

            int pos = uriBC.indexOf("://", 0, 3, 4);
            int uriBCStart = uriBC.getStart();
            int slashPos = -1;
            if (pos != -1) {
                byte[] uriB = uriBC.getBytes();
                slashPos = uriBC.indexOf('/', pos + 3);
                if (slashPos == -1) {
                    slashPos = uriBC.getLength();
                    // Set URI as "/"
                    request.requestURI().setBytes
                        (uriB, uriBCStart + pos + 1, 1);
                } else {
                    request.requestURI().setBytes
                        (uriB, uriBCStart + slashPos, 
                         uriBC.getLength() - slashPos);
                }
                MessageBytes hostMB = headers.setValue("host");
                hostMB.setBytes(uriB, uriBCStart + pos + 3, 
                                slashPos - pos - 3);
            }

        }

        // Input filter setup
        InputFilter[] inputFilters = inputBuffer.getFilters();

        // Parse content-length header
        long contentLength = request.getContentLengthLong();
        if (contentLength >= 0) {
            
            inputBuffer.addActiveFilter
                (inputFilters[Constants.IDENTITY_FILTER]);
            contentDelimitation = true;
        }

        // Parse transfer-encoding header
        MessageBytes transferEncodingValueMB = null;
        if (http11)
            transferEncodingValueMB = headers.getValue("transfer-encoding");
        if (transferEncodingValueMB != null) {
            String transferEncodingValue = transferEncodingValueMB.toString();
            // Parse the comma separated list. "identity" codings are ignored
            int startPos = 0;
            int commaPos = transferEncodingValue.indexOf(',');
            String encodingName = null;
            while (commaPos != -1) {
                encodingName = transferEncodingValue.substring
                    (startPos, commaPos).toLowerCase().trim();
                if (!addInputFilter(inputFilters, encodingName)) {
                    // Unsupported transfer encoding
                    error = true;
                    // 501 - Unimplemented
                    response.setStatus(501);
                }
                startPos = commaPos + 1;
                commaPos = transferEncodingValue.indexOf(',', startPos);
            }
            encodingName = transferEncodingValue.substring(startPos)
                .toLowerCase().trim();
            if (!addInputFilter(inputFilters, encodingName)) {
                // Unsupported transfer encoding
                error = true;
                // 501 - Unimplemented
                response.setStatus(501);
            }
        }

        MessageBytes valueMB = headers.getValue("host");

        // Check host header
        if (http11 && (valueMB == null)) {
            error = true;
            // 400 - Bad request
            response.setStatus(400);
        }

        parseHost(valueMB);

        if (!contentDelimitation) {
            // If there's no content length 
            // (broken HTTP/1.0 or HTTP/1.1), assume
            // the client is not broken and didn't send a body
            inputBuffer.addActiveFilter
                (inputFilters[Constants.VOID_FILTER]);
            contentDelimitation = true;
        }
    }


    /**
     * Parse host.
     */
    public void parseHost(MessageBytes valueMB) {

        if (valueMB == null || valueMB.isNull()) {
            // HTTP/1.0
            // Default is what the socket tells us. Overriden if a host is 
            // found/parsed
            request.setServerPort(socket.getLocalPort());
            InetAddress localAddress = socket.getLocalAddress();
            // Setting the socket-related fields. The adapter doesn't know 
            // about socket.
            request.setLocalHost(localAddress.getHostName());
            request.serverName().setString(localAddress.getHostName());
            return;
        }

        ByteChunk valueBC = valueMB.getByteChunk();
        byte[] valueB = valueBC.getBytes();
        int valueL = valueBC.getLength();
        int valueS = valueBC.getStart();
        int colonPos = -1;
        if (hostNameC.length < valueL) {
            hostNameC = new char[valueL];
        }

        boolean ipv6 = (valueB[valueS] == '[');
        boolean bracketClosed = false;
        for (int i = 0; i < valueL; i++) {
            char b = (char) valueB[i + valueS];
            hostNameC[i] = b;
            if (b == ']') {
                bracketClosed = true;
            } else if (b == ':') {
                if (!ipv6 || bracketClosed) {
                    colonPos = i;
                    break;
                }
            }
        }

        if (colonPos < 0) {
            if (sslSupport == null) {
                // 80 - Default HTTTP port
                request.setServerPort(80);
            } else {
                // 443 - Default HTTPS port
                request.setServerPort(443);
            }
            request.serverName().setChars(hostNameC, 0, valueL);
        } else {

            request.serverName().setChars(hostNameC, 0, colonPos);

            int port = 0;
            int mult = 1;
            for (int i = valueL - 1; i > colonPos; i--) {
                int charValue = HexUtils.DEC[(int) valueB[i + valueS]];
                if (charValue == -1) {
                    // Invalid character
                    error = true;
                    // 400 - Bad request
                    response.setStatus(400);
                    break;
                }
                port = port + (charValue * mult);
                mult = 10 * mult;
            }
            request.setServerPort(port);

        }

    }


    /**
     * When committing the response, we have to validate the set of headers, as
     * well as setup the response filters.
     */
    protected void prepareResponse() {

        boolean entityBody = true;
        contentDelimitation = false;

        OutputFilter[] outputFilters = outputBuffer.getFilters();

        if (http09 == true) {
            // HTTP/0.9
            outputBuffer.addActiveFilter
                (outputFilters[Constants.IDENTITY_FILTER]);
            return;
        }

        int statusCode = response.getStatus();
        if ((statusCode == 204) || (statusCode == 205) 
            || (statusCode == 304)) {
            // No entity body
            outputBuffer.addActiveFilter
                (outputFilters[Constants.VOID_FILTER]);
            entityBody = false;
            contentDelimitation = true;
        }

        MessageBytes methodMB = request.method();
        if (methodMB.equals("HEAD")) {
            // No entity body
            outputBuffer.addActiveFilter
                (outputFilters[Constants.VOID_FILTER]);
            contentDelimitation = true;
        }

        MimeHeaders headers = response.getMimeHeaders();
        if (!entityBody) {
            response.setContentLength(-1);
        } else {
            if ( forcedResponseType.equalsIgnoreCase(
                    org.apache.coyote.tomcat5.Constants.FORCED_RESPONSE_TYPE) ) {
                String contentType = response.getContentType();
                if (contentType != null) {
                    headers.setValue("Content-Type").setString(contentType);
                } else {
                    headers.setValue("Content-Type")
                        .setString(defaultResponseType);                
                }
            } else {
                headers.setValue("Content-Type").setString(forcedResponseType);
            }     
        
            String contentLanguage = response.getContentLanguage();
            if (contentLanguage != null) {
                headers.setValue("Content-Language")
                    .setString(contentLanguage);
            }
        }

        int contentLength = response.getContentLength();
        if (contentLength != -1) {
            headers.setValue("Content-Length").setInt(contentLength);
            outputBuffer.addActiveFilter
                (outputFilters[Constants.IDENTITY_FILTER]);
            contentDelimitation = true;
        } else {
            if (entityBody && http11 && keepAlive) {
                outputBuffer.addActiveFilter
                    (outputFilters[Constants.CHUNKED_FILTER]);
                contentDelimitation = true;
                response.addHeader("Transfer-Encoding", "chunked");
            } else {
                outputBuffer.addActiveFilter
                    (outputFilters[Constants.IDENTITY_FILTER]);
            }
        }

        // Add date header
        if (! response.containsHeader("Date")){
            String date = FastHttpDateFormat.getCurrentDate();
            response.addHeader("Date", date);
        }
         
        // Add server header
        headers.setValue("Server").setString(ServerInfo.getServerInfo());

        // Add transfer encoding header
        // FIXME

        if ((entityBody) && (!contentDelimitation)) {
            // Mark as close the connection after the request, and add the 
            // connection: close header
            keepAlive = false;
        }

        // If we know that the request is bad this early, add the
        // Connection: close header.
        keepAlive = keepAlive && !statusDropsConnection(statusCode)
            && !dropConnection;
        if (!keepAlive ) {
            headers.setValue("Connection").setString("close");
            connectionHeaderValue = false;
        } else if (!http11 && !error) {
            headers.setValue("Connection").setString("Keep-Alive");
        }

        // Build the response header
        outputBuffer.sendStatus();

        int size = headers.size();
        for (int i = 0; i < size; i++) {
            outputBuffer.sendHeader(headers.getName(i), headers.getValue(i));
        }
        outputBuffer.endHeaders();

    }


    /**
     * Initialize standard input and output filters.
     */
    protected void initializeFilters() {

        // Create and add the identity filters.
        inputBuffer.addFilter(new IdentityInputFilter());
        outputBuffer.addFilter(new IdentityOutputFilter());

        // Create and add the chunked filters.
        inputBuffer.addFilter(new ChunkedInputFilter());
        outputBuffer.addFilter(new ChunkedOutputFilter());

        // Create and add the void filters.
        inputBuffer.addFilter(new VoidInputFilter());
        outputBuffer.addFilter(new VoidOutputFilter());

        // Create and add buffered input filter
        inputBuffer.addFilter(new BufferedInputFilter());

        // Create and add the chunked filters.
        //inputBuffer.addFilter(new GzipInputFilter());
        outputBuffer.addFilter(new GzipOutputFilter());

    }


    /**
     * Add an input filter to the current request.
     * 
     * @return false if the encoding was not found (which would mean it is 
     * unsupported)
     */
    protected boolean addInputFilter(InputFilter[] inputFilters, 
                                     String encodingName) {
        if (encodingName.equals("identity")) {
            // Skip
        } else if (encodingName.equals("chunked")) {
            inputBuffer.addActiveFilter
                (inputFilters[Constants.CHUNKED_FILTER]);
            contentDelimitation = true;
        } else {
            for (int i = 2; i < inputFilters.length; i++) {
                if (inputFilters[i].getEncodingName()
                    .toString().equals(encodingName)) {
                    inputBuffer.addActiveFilter(inputFilters[i]);
                    return true;
                }
            }
            return false;
        }
        return true;
    }


    /**
     * Specialized utility method: find a sequence of lower case bytes inside
     * a ByteChunk.
     */
    protected int findBytes(ByteChunk bc, byte[] b) {

        byte first = b[0];
        byte[] buff = bc.getBuffer();
        int start = bc.getStart();
        int end = bc.getEnd();

        // Look for first char 
        int srcEnd = b.length;

        for (int i = start; i <= (end - srcEnd); i++) {
            if (Ascii.toLower(buff[i]) != first) continue;
            // found first char, now look for a match
            int myPos = i+1;
            for (int srcPos = 1; srcPos < srcEnd; ) {
                    if (Ascii.toLower(buff[myPos++]) != b[srcPos++])
                break;
                    if (srcPos == srcEnd) return i - start; // found it
            }
        }
        return -1;

    }

    /**
     * Determine if we must drop the connection because of the HTTP status
     * code.  Use the same list of codes as Apache/httpd.
     */
    protected boolean statusDropsConnection(int status) {
        return status == 400 /* SC_BAD_REQUEST */ ||
               status == 408 /* SC_REQUEST_TIMEOUT */ ||
               status == 411 /* SC_LENGTH_REQUIRED */ ||
               status == 413 /* SC_REQUEST_ENTITY_TOO_LARGE */ ||
               status == 414 /* SC_REQUEST_URI_TOO_LARGE */ ||
               status == 500 /* SC_INTERNAL_SERVER_ERROR */ ||
               status == 503 /* SC_SERVICE_UNAVAILABLE */ ||
               status == 501 /* SC_NOT_IMPLEMENTED */;
    }

     /**
     * Add input or output filter.
     * 
     * @param className class name of the filter
     */
    protected void addFilter(String className) {
        try {
            Class clazz = Class.forName(className);
            Object obj = clazz.newInstance();
            if (obj instanceof InputFilter) {
                inputBuffer.addFilter((InputFilter) obj);
            } else if (obj instanceof OutputFilter) {
                outputBuffer.addFilter((OutputFilter) obj);
            } else {
                SelectorThread.logger().log(Level.WARNING,
                        "processorTask.unknownFilter" ,className);
            }
        } catch (Exception e) {
            SelectorThread.logger().log(Level.SEVERE,"processorTask.errorFilter", 
                        new Object[]{className, e});
        }
    }


    /**
     * Set the maximum size of a POST which will be buffered in SSL mode.
     */
    public void setMaxPostSize(int mps) {
        maxPostSize = mps;
    }


    /**
     * Return the maximum size of a POST which will be buffered in SSL mode.
     */
    public int getMaxPostSize() {
        return maxPostSize;
    }


    /**
     * Set the socket associated with this HTTP connection.
     */
    public void setSocket(Socket socket){
        this.socket = socket;
    }

    
    /**
     * Set the upload timeout.
     */
    public void setTimeout( int timeouts ) {
        timeout = timeouts ;
    }

    /**
     * Get the upload timeout.
     */
    public int getTimeout() {
        return timeout;
    }
    
    /** 
     * Set the SSL implementation object we are using. This information will
     * be required when SSL is used.
     */
    public void setSSLImplementation( SSLImplementation sslImplementation){
        this.sslImplementation = sslImplementation;
    }
    
    
    /**
     * Register a new <code>RequestProcessor</code> instance.
     */
    private void registerMonitoring(){

        RequestInfo requestInfo = request.getRequestProcessor();
        // Add RequestInfo to RequestGroupInfo
        requestInfo.setGlobalProcessor(getRequestGroupInfo());

        try {
            oname = new ObjectName(selectorThread.getDomain()
                                   +  ":type=RequestProcessor,worker=http"
                                   + selectorThread.getPort()
                                   + ",name=HttpRequest" 
                                   + requestCount++ );
            Registry.getRegistry().registerComponent(requestInfo, oname,
                                                     null);
        } catch( Exception ex ) {
            SelectorThread.logger().log(Level.WARNING,
                       "processorTask.errorRegisteringRequest",
                       ex);
        }

        hasRequestInfoRegistered = true;
    }
    

    /**
     * Unregisters the MBean corresponding to this
     * <code>ProcessorTask</code>.
     */
    private void unregisterMonitoring() {

        RequestInfo requestInfo = request.getRequestProcessor();
        /*
         * Remove 'requestInfo' from 'requestGroupInfo'.
         * This will also update 'requestGroupInfo' with the current stats
         * of 'requestInfo', which is why we need to reset 'requestInfo' so
         * its current stats will not be considered when it is added back to
         * 'requestGroupInfo' next time registerMonitoring() is called.
         */
        requestInfo.setGlobalProcessor(null);
        requestInfo.reset();

        if (oname != null) {
            try {
                Registry.getRegistry().unregisterComponent(oname);
            } catch (Exception ex) {
                SelectorThread.logger().log(Level.WARNING,
                           "processorTask.errorUnregisteringRequest",
                           ex);
            }
        }

        hasRequestInfoRegistered = false;
    }


    public int getMaxHttpHeaderSize() {
        return maxHttpHeaderSize;
    }
    
    public void setMaxHttpHeaderSize(int maxHttpHeaderSize) {
        this.maxHttpHeaderSize = maxHttpHeaderSize;
    }


    /**
     * Set the request input buffer size
     */
    public void setBufferSize(int requestBufferSize){
        this.requestBufferSize = requestBufferSize;
    }
    

    /**
     * Return the request input buffer size
     */
    public int getBufferSize(){
        return requestBufferSize;
    }
    
    
    /**
     * Use alternate keep-alive mechanism by using the <code>Selector</code>
     * @param useAlternateKeepAlive true if an NIO based keep-alive algorithm
     */
    public void useAlternateKeepAlive(boolean useAlternateKeepAlive){
        this.useAlternateKeepAlive = useAlternateKeepAlive;
    }
    

    /**
     * Return the current <code>Socket</code> used by this instance
     * @return socket the current <code>Socket</code> used by this instance
     */
    protected Socket getSocket(){
        return socket;
    }   
    
    
    /**
     * Enable or disable the keep-alive mechanism. Setting this value
     * to <code>false</code> will automatically add the following header to the
     * response ' Connection: close '
     */
    public void setDropConnection(boolean dropConnection){
        this.dropConnection = dropConnection;
    }
    
    
    /**
     * Is the keep-alive mechanism enabled or disabled.
     */
    public boolean getDropConnection(){
        return dropConnection;
    }
    
    
    /**
     * Set the maximum number of Keep-Alive requests to honor.
     * This is to safeguard from DoS attacks.  Setting to a negative
     * value disables the check.
     */
    public void setMaxKeepAliveRequests(int maxKeepAliveRequests) {
        this.maxKeepAliveRequests = maxKeepAliveRequests;
    }


    /**
     * Return the number of Keep-Alive requests that we will honor.
     */
    public int getMaxKeepAliveRequests() {
        return maxKeepAliveRequests;
    }
    
    
    /**
     * Return the number of blocking keep-alive connection
     */
    public int countBlockingKeepAlive(){
        if (maxKeepAliveRequests == -1) return -1;
                
        return maxKeepAliveRequests - keepAliveLeft; 
    }
 
    
    /**
     * Set the <code>Handler</code> used by this class.
     */
    public void setHandler(Handler handler){
        this.handler = handler;
    } 
    
    
    /**
     * Set the default response type used. Specified as a semi-colon
     * delimited string consisting of content-type, encoding,
     * language, charset
     */
    public void setDefaultResponseType(String defaultResponseType){
         this.defaultResponseType = defaultResponseType;
    }


    /**
     * Return the default response type used
     */
    public String getDefaultResponseType(){
         return defaultResponseType;
    }
    
    
    /**
     * Set the forced response type used. The response type to be forced 
     * if the content served cannot be matched by any of the MIME mappings
     * for extensions. Specified as a semi-colon delimited string consisting of
     * content-type, encoding, language, charset
     */
    public void setForcedResponseType(String forcedResponseType){
        this.forcedResponseType = forcedResponseType;
    }  
    
        
    /**
     * Return the default response type used
     */
    public String getForcedResponseType(){
        return forcedResponseType;
    }   
    
    
    // ------------------------------------------------------- Asynch call ---//
    
    /**
     * Enable/disable asynchronous execution of this object.
     */
    public void setEnableAsyncExecution(boolean asyncExecution){
        this.asyncExecution = asyncExecution;
    }
    
    
    /**
     * Is asynchronous execution enabled?
     */    
    public boolean isAsyncExecutionEnabled(){
        return asyncExecution;
    }
    
    
    /**
     * Set the <code>AsyncHandler</code> used when asynchronous execution is 
     * enabled.
     */
    public void setAsyncHandler(AsyncHandler asyncHandler){
        this.asyncHandler = asyncHandler;     
    }
    
       
    /**
     * Return the <code>AsyncHandler</code> used when asynchronous execution is 
     * enabled.
     */    
    public AsyncHandler getAsyncHandler(){
        return asyncHandler;
    }
    
    
    /**
     * Recyle this object.
     */
    public void recycle(){
        clearTaskListeners();
        socket = null;
        dropConnection = false;
        key = null;
        channelOutputBuffer.setChannel(null);
    }
}

