/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.admin.gui.extensions.util;

import com.sun.enterprise.tools.admingui.util.GuiUtil;
import com.sun.enterprise.tools.admingui.util.JMXUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.ObjectName;


/**
 *
 * @author irfan
 */
public class SipUtil {
    public static final String RESOURCE_NAME = "sip";
    public static final String SIP_MODULE_TYPE = "org.jvnet.glassfish.comms.deployment.backend.SipArchiveDeployer";

    /**
     * This function returns the SipServiceMBean
     * @param sipConfigMBean ObjectName for SIP Config
     * @return ObjectName for the sip-service element
     */
    public static ObjectName getSipServiceMBean(ObjectName sipConfigMBean) {
        //Lets try to get the sip-service MBean
        ObjectName sipServiceMBean = null;

        try {
            sipServiceMBean = (ObjectName) JMXUtil.invoke(sipConfigMBean,
                    "getSipService", new Object[] { null },
                    new String[] { "java.lang.String" });
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }

        return sipServiceMBean;
    }

    /**
     * The function returns the properties for the element represented by the configMBean
     * @param configBean ObjectName representation of the element
     * @return HashMap of properties.
     */
    public static HashMap getProperties(ObjectName configBean) {
        HashMap props = new HashMap();

        try {
            AttributeList attributes = (AttributeList) JMXUtil.invoke(configBean,
                    "getProperties", null, null);
            Iterator<Attribute> it = attributes.iterator();

            while (it.hasNext()) {
                Attribute attribute = it.next();
                String name = attribute.getName();
                Object value = attribute.getValue();
                props.put(name, ((value == null) ? " " : value.toString()));
            }
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }

        return props;
    }

    /**
     * The function sets the properties on an element. The function takes the element bean, map of added
     *  properties and an list of names of properties that are to be removed
     * @param objectName ObjectName representation of the element
     * @param addProps Map of properties to be added
     * @param removeProps ArrayList of property names to be removed
     */
    public static void setProperties(ObjectName objectName, HashMap addProps,
        ArrayList removeProps) {
        if ((removeProps != null) && (removeProps.size() > 0)) {
            Iterator<String> it = removeProps.iterator();

            while (it.hasNext()) {
                Attribute attribute = new Attribute(it.next(), null);
                JMXUtil.invoke(objectName, "setProperty",
                    new Object[] { attribute },
                    new String[] { "javax.management.Attribute" });
            }
        }

        if ((addProps != null) && (addProps.size() > 0)) {
            Iterator<String> it = addProps.keySet().iterator();

            while (it.hasNext()) {
                String propName = it.next();
                Attribute attribute = new Attribute(propName,
                        addProps.get(propName));
                JMXUtil.invoke(objectName, "setProperty",
                    new Object[] { attribute },
                    new String[] { "javax.management.Attribute" });
            }
        }
    }

    /** The function sets the value of a single property on an element
     */
    public static void setProperty(ObjectName element, String propName,
        Object propValue) {
        Attribute attribute = new Attribute(propName, propValue);
        JMXUtil.invoke(element, "setProperty", new Object[] { attribute },
            new String[] { "javax.management.Attribute" });
    }

    /**
     * The function sets a single property on an element
     * @param objectName ObjectName representation of the element
     * @param propName
     * @param propValue
     */
    public static void setProperty(ObjectName objectName, String propName,
        String propValue) {
        Attribute attribute = new Attribute(propName, propValue);

        if (GuiUtil.isEmpty(propValue)) {
            //remove the property
            attribute = new Attribute(propName, null);
        }

        JMXUtil.invoke(objectName, "setProperty", new Object[] { attribute },
            new String[] { "javax.management.Attribute" });
    }

    /**
     * The function returns the attributes for an element. The function takes as input the ObjectName
     *  representation of the element and a string array of attribute names
     * @param objectName ObjectName representation of the element
     * @param propNames String array of attribute names.
     * @return
     */
    public static HashMap getAttributeValues(ObjectName objectName,
        String[] propNames) {
        HashMap props = new HashMap();

        try {
            for (String attributeName : propNames) {
                Object value = JMXUtil.getAttribute(objectName, attributeName);
                props.put(attributeName,
                    ((value == null) ? " " : value.toString()));
            }

            return props;
        } catch (Exception ex) {
            return null;
        }
    }

    /**
     * The function sets the attribute values for a given element
     * @param objectName ObjectName representation of the element
     * @param props Map of attribute names and values
     */
    public static void setAttributeValues(ObjectName objectName, HashMap props) {
        Iterator<String> it = props.keySet().iterator();

        while (it.hasNext()) {
            String propName = it.next();
            Object value = props.get(propName);
            Attribute attribute = new Attribute(propName, value);
            JMXUtil.setAttribute(objectName, attribute);
        }
    }

    /**
     * The function is called to get the default attribute values for an element
     *
     * @param objectName The ObjectName of the element
     * @param propNames Array of attribute names
     * @return Map of default attribute values for the specified attributes
     */
    public static HashMap getDefaultAttributeValues(ObjectName objectName,
        String[] propNames) {
        HashMap props = new HashMap();

        try {
            for (String attributeName : propNames) {
                Object value = JMXUtil.getDefaultAttributeValue(objectName.getCanonicalName(),
                        new String[] { attributeName });
                props.put(attributeName,
                    ((value == null) ? " " : value.toString()));
            }

            return props;
        } catch (Exception ex) {
            System.out.println("Exception :" + ex.getMessage());

            return null;
        }
    }

    /**
     * This function returns the sip-listener element ObjectName
     * @param sipConfigBeanName sip-service element
     * @param listenerName The listener id
     * @return ObjectName representing the sip-listener
     */
    public static ObjectName getSipListener(ObjectName sipConfigBeanName,
        String listenerName) {
        ObjectName sipListener = (ObjectName) JMXUtil.invoke(sipConfigBeanName,
                "getSipListener", new Object[] { listenerName, null },
                new String[] { "java.lang.String", "java.lang.String" });

        return sipListener;
    }

    /**
     * This function gets the SSL element under a sip-listener element
     * @param sipConfigBeanName ObjectName representing the sip-service element
     * @param listenerName The listener id for which ssl element is to be queried
     * @return ObjectName for the SSL element if ssl is configured: else it returns null.
     */
    public static ObjectName getSSL(ObjectName sipConfigBeanName,
        String listenerName) {
        ObjectName ssl = (ObjectName) JMXUtil.invoke(sipConfigBeanName,
                "getSSL", new Object[] { listenerName, null },
                new String[] { "java.lang.String", "java.lang.String" });

        return ssl;
    }

    /**
     *
     * @param objectValue
     * @return
     */
    public static boolean getBooleanValue(Object objectValue) {
        if (objectValue == null) {
            return false;
        }

        return new Boolean(objectValue.toString()).booleanValue();
    }

    /** The function updates the properties of an element. The skipList is used to skip over the properties
     *  that are defined in this list. These properties would be handled through the changeProperty method.
     *  Usually these are the properties that are always visible on the page and not in the properties
     *  table
     */
    public static void updateProperties(ObjectName element,
        HashMap<String, Object> newProps, List skipList) {
        HashMap oldProps = getProperties(element);

        //Remove any property that is no longer in the new list
        Iterator iter = oldProps.keySet().iterator();

        while (iter.hasNext()) {
            Object key = iter.next();

            if (skipList.contains(key)) {
                continue;
            }

            if (!newProps.containsKey(key)) {
                setProperty(element, (String) key, null);
            }
        }

        //update the value if the value is different or create a new property if it doesn't exist before
        for (String propName : newProps.keySet()) {
            String val = (String) newProps.get(propName);

            if (oldProps.containsKey(propName)) {
                //update
                String oldValue = (String) oldProps.get(propName);

                if ((!oldValue.equals(val)) && (!GuiUtil.isEmpty(val))) {
                    setProperty(element, propName, val);
                }
            } else {
                //new prop
                if (!GuiUtil.isEmpty(val)) {
                    setProperty(element, propName, val);
                }
            }
        }
    }

    /** The fucntion is called to change the value of a property or delete the same. This function is
     *  usually called for the properties that are always visible on the page
     */
    public static void changeProperty(ObjectName element, String propName,
        String propValue) {
        HashMap oldProps = getProperties(element);

        if (oldProps.containsKey(propName)) {
            if (GuiUtil.isEmpty(propValue)) {
                setProperty(element, propName, null);
            } else {
                setProperty(element, propName, propValue);
            }
        } else {
            if (!GuiUtil.isEmpty(propValue)) {
                setProperty(element, propName, propValue);
            }
        }
    }

    public static ObjectName getSipApplication(String objectName, String name) {
        try {
            ObjectName[] modules = (ObjectName[]) JMXUtil.invoke(objectName,
                    "getExtensionModule", null, null);

            for (ObjectName module : modules) {
                String moduleType = (String) JMXUtil.getAttribute(module,
                        "module-type");
                String moduleName = (String) JMXUtil.getAttribute(module, "name");

                if ((SIP_MODULE_TYPE.equals(moduleType)) &&
                        (name.equals(moduleName))) {
                    return module;
                }
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }

        return null;
    }
}
