/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core;

import com.sun.enterprise.tools.verifier.tests.app.AppAltDDAppClient;

import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.Collection;
import java.util.HashMap;
import java.util.logging.Level;


/**
 *
 * @author kshitiz
 */
public class Controller {
    private static final LogUtil _logger = new LogUtil(Controller.class);

    /* map containing name to cluster mapping */
    private HashMap<String, ServerCluster> clusterMap;

    /* map containing name to request group mapping for http request */
    private HashMap<String, RequestGroup> httpRequestGroupMap;

    /* map containing cluster to corresponding health monitor mapping */
    private HashMap<String, ClusterHealthMonitor> healthMonitorMap;

    /* the load-balancing policy */
    private String lbpolicy;

    /* name of Data Centric Rules file */
    private String dcrFileName;

    /* flag indicating which component(SIP/HTTP) of CLB */
    private int lbType;

    /* map containing instance to base 64 encoded key mapping */
    private GlobalInstanceMap globalInstanceMap;

    /* default http request group to handle http request */
    private RequestGroup defaultHttpRequestGroup;

    /* sip request group to handle sip requests */
    private RequestGroup sipRequestGroup;

    /* number of request being currently serviced */
    private long requestCount;

    /* lock to be acquired before modifying the request count */
    private Object requestCountLock;

    /* instance of dcr file update event notifier */
    private DCRFileUpdateEventNotifier dcrFileUpdateEventNotifier;
    private String controllerType;

    /** Creates a new instance of ControllerImpl */
    public Controller() {
        clusterMap = new HashMap<String, ServerCluster>();
        httpRequestGroupMap = new HashMap<String, RequestGroup>();
        requestCountLock = new Object();
        globalInstanceMap = new GlobalInstanceMap();
    }

    public static Controller createInstance() {
        return new Controller();
    }

    public boolean addCluster(ServerCluster cluster) {
        clusterMap.put(cluster.getName(), cluster);

        return true;
    }

    public boolean deleteCluster(ServerCluster cluster) {
        return (clusterMap.put(cluster.getName(), cluster) != null);
    }

    public boolean enableCluster(ServerCluster cluster) {
        return cluster.enableCluster();
    }

    public boolean disableCluster(ServerCluster cluster) {
        return cluster.enableCluster();
    }

    public void addCluster(String clusterName, boolean selfLoadbalance)
        throws CLBRuntimeException {
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType + " : Adding cluster " +
            clusterName + " with self-loadbalance as " + selfLoadbalance);

        if (getCluster(clusterName) != null) {
            throw new CLBRuntimeException("Controller : " + controllerType +
                " : Cluster with name : " + clusterName + " already exists." +
                "Cannot add duplicate.");
        }

        ServerCluster cluster = new ServerCluster(clusterName, selfLoadbalance);
        addCluster(cluster);
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType +
            " : Successfully added cluster " + clusterName);
    }

    public void addInstance(String clusterName, String instanceName,
        boolean enabled, int timeoutInMinutes, String[] listeners)
        throws CLBRuntimeException {
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType + " : Adding instance " +
            instanceName + " to cluster " + clusterName);

        ServerCluster cluster = clusterMap.get(clusterName);

        if (cluster == null) {
            throw new CLBRuntimeException("Controller : " + controllerType +
                " : Cluster with name " + clusterName + " does not exist");
        }

        cluster.addInstance(instanceName, enabled, timeoutInMinutes, listeners);
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType + " : Added instance " +
            instanceName + " to cluster " + clusterName);
    }

    public void addWebModule(String clusterName, String contextRoot)
        throws CLBRuntimeException {
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType + " : Adding web module " +
            contextRoot + " to cluster " + clusterName);

        if (lbType == CLBConstants.SIP_CLB) {
            throw new CLBRuntimeException("Controller : " + controllerType +
                " : No web module should be added in SIP CLB");
        }
        
        if (!contextRoot.startsWith("/")) {
            contextRoot = "/" + contextRoot;
        }

        RequestGroup requestGroup = null;
        
        if(contextRoot.equals(CLBConstants.DEFAULT_CONTEXT_ROOT)){
            if(defaultHttpRequestGroup == null){
                _logger.logMsg(Level.INFO,
                        "Controller : " + controllerType + " : Adding default web module " +
                        "to cluster " + clusterName);
                defaultHttpRequestGroup = RequestGroupFactory.createRequestGroup(this,
                    contextRoot, lbType);
            }                
            requestGroup = defaultHttpRequestGroup;
        }
        else{
            requestGroup = httpRequestGroupMap.get(contextRoot);
            if (requestGroup == null) {
                requestGroup = RequestGroupFactory.createRequestGroup(this,
                        contextRoot, lbType);
                httpRequestGroupMap.put(contextRoot, requestGroup);
            }
        }

        ServerCluster cluster = clusterMap.get(clusterName);

        if (cluster == null) {
            throw new CLBRuntimeException("Controller : " + controllerType +
                " : No such cluster - " + clusterName);
        }

        requestGroup.addCluster(cluster);
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType + " : Added web module " +
            contextRoot + " to cluster " + clusterName);
    }

    public ServerCluster getCluster(String clusterName) {
        return clusterMap.get(clusterName);
    }

    public Collection<ServerCluster> getAllClusters() {
        return clusterMap.values();
    }

    public ServerInstance getInstance(String clusterName, String instanceName) {
        ServerCluster cluster = clusterMap.get(clusterName);

        if (cluster == null) {
            return null;
        }

        return cluster.getInstance(instanceName);
    }

    public RequestGroup getRequestGroup(String appName) {
        if (lbType == CLBConstants.SIP_CLB) {
            return sipRequestGroup;
        }

        return httpRequestGroupMap.get(appName);
    }
    
    public RequestGroup getDefaultHttpRequestGroup() {
        return defaultHttpRequestGroup;
    }

    public void initialize() throws CLBRuntimeException {
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType +
            " : Intializing the controller ... ");

        healthMonitorMap = new HashMap<String, ClusterHealthMonitor>();

        dcrFileUpdateEventNotifier = new DCRFileUpdateEventNotifier();

        if (lbType == CLBConstants.SIP_CLB) {
            sipRequestGroup = RequestGroupFactory.createRequestGroup(this, "",
                    lbType);
        }

        for (ServerCluster cluster : clusterMap.values()) {
            cluster.initialize(lbType);
            createClusterHealthMonitor(cluster);
            cluster.initializeClusterRouter(lbpolicy, this);

            if (lbType == CLBConstants.SIP_CLB) {
                sipRequestGroup.addCluster(cluster);
            }
        }

        if (lbType == CLBConstants.SIP_CLB) {
            sipRequestGroup.initializeRouter(lbpolicy);
        } else {
            for (RequestGroup reqGroup : httpRequestGroupMap.values())
                reqGroup.initializeRouter(lbpolicy);
            if(defaultHttpRequestGroup != null)
                defaultHttpRequestGroup.initializeRouter(lbpolicy);
        }

        populateGlobalInstanceMap();

        dcrFileUpdateEventNotifier.registerWithAdminEventListener();

        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType +
            " : Completed intialization of the controller ... ");
    }

    public void setPolicy(String policy) {
        lbpolicy = policy;
    }

    public void setDCRFileName(String name) {
        dcrFileName = name;
    }

    public String getDCRFileName() {
        return dcrFileName;
    }

    public void setLBType(int lbType) {
        this.lbType = lbType;

        if (lbType == CLBConstants.HTTP_CLB) {
            controllerType = "HTTP CLB";
        } else {
            controllerType = "SIP CLB";
        }
    }

    public int getLBType() {
        return lbType;
    }

    public GlobalInstanceMap getGlobalInstanceMap() {
        return globalInstanceMap;
    }

    private void populateGlobalInstanceMap() throws CLBRuntimeException {
        globalInstanceMap = new GlobalInstanceMap();

        for (ServerCluster cluster : clusterMap.values()) {
            for (ServerInstance instance : cluster.getAllInstances()) {
                globalInstanceMap.putServerInstance(instance.getID(), instance);
            }
        }
    }

    /**
     * This method is called at the time of dynamic reconfig. It detects
     * affected server instances and takes care of quiescing them
     *
     */
    public void reconfig(Controller oldController) {
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType +
            " : Reconfiguring the controller ... ");

        Collection<ServerCluster> oldClusters = oldController.getAllClusters();

        for (ServerCluster oldCluster : oldClusters) {
            ServerCluster newCluster = getCluster(oldCluster.getName());

            if (newCluster != null) {
                newCluster.reconfig(oldCluster);
            }
        }

        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType +
            " : Completed reconfiguration of the controller ... ");
    }

    /**
     * This method is obsolete controller to perform the cleanup before being
     * garbage collected. This waits till all request being serviced by this controller
     * are done. Then it performs the cleanup
     */
    public void cleanUp() {
        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType +
            " : Performing cleanup of the controller ... ");

        boolean servingRequest = true;

        while (servingRequest) {
            synchronized (requestCountLock) {
                servingRequest = !(requestCount == 0);

                if (!servingRequest) {
                    break;
                }

                try {
                    requestCountLock.wait();
                } catch (InterruptedException ex) {
                    ex.printStackTrace();
                }
            }
        }

        for (ServerCluster cluster : clusterMap.values())
            cluster.cleanup();

        if (lbType == CLBConstants.HTTP_CLB) {
            for (RequestGroup reqGroup : httpRequestGroupMap.values())
                reqGroup.cleanup();
            if(defaultHttpRequestGroup != null)
                defaultHttpRequestGroup.cleanup();
        } else {
            sipRequestGroup.cleanup();
        }

        _logger.logMsg(Level.INFO,
            "Controller : " + controllerType +
            " : Completed cleanup of the controller ... ");
    }

    /**
     * Increment the count of request being serviced by this controller
     */
    public void incrementRequestCount() {
        synchronized (requestCountLock) {
            requestCount++;
        }
    }

    /**
     * Decrement the count of request being serviced by this controller
     */
    public void decrementRequestCount() {
        synchronized (requestCountLock) {
            requestCount--;

            if (requestCount == 0) {
                requestCountLock.notify();
            }
        }
    }

    private void createClusterHealthMonitor(ServerCluster cluster)
        throws CLBRuntimeException {
        ClusterHealthMonitor healthMonitor = new ClusterHealthMonitor(this,
                cluster);
        healthMonitorMap.put(cluster.getName(), healthMonitor);
        cluster.markInstancesHealthy(healthMonitor.getHealthyInstances());
    }

    /**
     * Returns an instance of DCRFileUpdateEventNotifier associated with this controller.
     *
     * @return instance of dcr file update event notifier
     */
    public DCRFileUpdateEventNotifier getDCRFileUpdateEventNotifier() {
        return dcrFileUpdateEventNotifier;
    }
}
