package org.jvnet.glassfish.comms.clb.core;

import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.AvailabilityService;
import com.sun.enterprise.config.serverbeans.Config;
import com.sun.enterprise.config.serverbeans.ConvergedLoadBalancer;
import com.sun.enterprise.config.serverbeans.ServerBeansFactory;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.server.ServerContext;

import org.jvnet.glassfish.comms.clb.admin.CLBConfigurator;
import org.jvnet.glassfish.comms.clb.core.sip.SipLoadBalancerManager;
import org.jvnet.glassfish.comms.httplayers.HttpLayer;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.File;
import java.io.FileInputStream;

import java.util.logging.Level;


public class ConvergedLoadBalancerFactory {
    private static SipLoadBalancerManager sipLBM;
    private static HttpLoadBalancingManager httpLBM;
    private static final LogUtil _logger = new LogUtil(ConvergedLoadBalancerFactory.class);
    private static String dcrFileName;
    private static boolean initialized = false;
    private static String clbConfigFileName;

    /**
     * Gets the SIP load balancing manager layer and also initializes. Used by the
     * {@link com.ericsson.ssa.config.LayerHandler} (via reflection) when
     * setting up the SIP and HTTP processing chains at start-up.
     *
     */
    public static SipLoadBalancerManager getInstance() {
        createLoadBalancingManagers();
        //Initiliazing the converged load balancer as both HttpLoadBalancingManager
        //and SipLoadBalancingManager must be initialized together
        //This is done here as SIP container will be available now
        //for SIP component to initialize
        initialize();

        return sipLBM;
    }

    private static void initialize() {
        if ((httpLBM != null) && (sipLBM != null)) {
            _logger.infoMsg("Initializing the Converged Load Balancer.");

            File configFile = new File(clbConfigFileName);
            CLBConfigurator clbConfigurator = CLBConfigurator.getInstance();
            clbConfigurator.init(configFile, sipLBM, httpLBM);
            dcrFileName = clbConfigurator.getDCRFileName();
            _logger.infoMsg(
                "Completed initialization of the Converged Load Balancer.");
        } else {
            _logger.infoMsg("Not initializing the Converged Load Balancer." +
                "This instance is not configured to act as Converged Load Balancer");
        }
    }

    private static synchronized void createLoadBalancingManagers() {
        if (initialized) {
            return;
        }

        initialized = true;

        _logger.infoMsg("Configuring the converged load balancer");

        ServerContext sc = com.sun.enterprise.server.ondemand.OnDemandServer.getServerContext();
        ConfigContext instanceConfigContext = sc.getConfigContext();

        //Added check whether this instance is DAS, as DAS cannot act as a converged load-balancer
        //This check may not be needed in case config of DAS will never have converged-load-balancer
        //under availability-service
        try {
            if (ServerHelper.isDAS(instanceConfigContext, sc.getInstanceName())) {
                initialized = true;
                _logger.infoMsg(
                    "This is Domain Administration Server. This cannot act as converged load" +
                    "balancer. Thus disabling it.");

                return;
            }
        } catch (ConfigException ex) {
            _logger.logMsg(Level.SEVERE, ex,
                "Unable to determine whether instance is DAS or not");
            _logger.logMsg(Level.SEVERE, ex,
                "Continuing with assumption that this instance is not DAS");
        }

        Config instanceConfig = null;

        try {
            instanceConfig = ServerBeansFactory.getConfigBean(instanceConfigContext);
        } catch (ConfigException ex) {
            _logger.logMsg(Level.SEVERE, ex,
                "ConfigException will getting instance config." +
                " Cannot initialize the converged load balancer");

            return;
        }

        AvailabilityService haService = instanceConfig.getAvailabilityService();

        if (haService == null) {
            _logger.infoMsg(
                "No Availability Service found when configuring the converged load balancer. CLB will not be activated");

            return;
        }

        ConvergedLoadBalancer clb = haService.getConvergedLoadBalancer();

        if (clb == null) {
            _logger.infoMsg(
                "No CLB entry found when configuring the converged load balancer. CLB will not be activated");

            return;
        }

        //Setting static variable with DCR file name
        clbConfigFileName = clb.getConfigFile();

        //Creating instance of HttpLoadBalancingManager
        //and SipLoadBalancerManager
        //They will be initialized later
        if (httpLBM == null) {
            httpLBM = HttpLoadBalancingManager.getInstance();
        }

        if (sipLBM == null) {
            sipLBM = SipLoadBalancerManager.getInstance();
        }

        _logger.infoMsg("The converged load balancer has been configured");
    }

    /**
     * This is the method used by the {@link org.jvnet.glassfish.comms.httplayers.HttpLayerHandler} when initializing from dispatcher.xml.
     * @return always null;
     */
    public static HttpLayer getHttpLayerInstance() {
        createLoadBalancingManagers();

        return httpLBM;
    }

    /**
     * Returns the DCR file name
     *
     * @return the name of dcr file
     */
    public static String getDCRFileName() {
        return dcrFileName;
    }
}
