/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core;

import com.sun.grizzly.tcp.Request;
import com.sun.grizzly.tcp.Response;

import org.apache.catalina.Valve;

import org.jvnet.glassfish.comms.clb.admin.CLBConfigurator;
import org.jvnet.glassfish.comms.clb.proxy.api.Endpoint;
import org.jvnet.glassfish.comms.clb.proxy.http.util.HttpRequest;
import org.jvnet.glassfish.comms.httplayers.HttpLayer;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.logging.Level;


/**
 *
 * @author kshitiz
 */
public class HttpLoadBalancingManager extends ControllerInitializer
    implements HttpLayer {
    private static HttpLoadBalancingManager instance = new HttpLoadBalancingManager();
    private static final LogUtil _logger = new LogUtil(HttpLoadBalancingManager.class);

    /** Creates a new instance of HttpLoadBalancingManager */
    private HttpLoadBalancingManager() {
    }

    public static HttpLoadBalancingManager getInstance() {
        return instance;
    }

    public String getInfo() {
        return "HTTP load-balancing valve";
    }

    public boolean invoke(Request request, Response response)
        throws Exception {
        _logger.logMsg(Level.INFO,
            "HttpLoadBalancingManager :  servicing request " +
            request.requestURI().toString());

        HttpRequest httpReq = (HttpRequest) request;

        if (ProxyKeyExtractor.isHttpRequestProxied(httpReq)) {
            _logger.logMsg(Level.INFO,
                "HttpLoadBalancingManager :  this request is already proxied." +
                " Sending it to local instance");
            setLocalEndPoint(httpReq);

            return true;
        }

        /* Case Controller is null
         * The possibility is that initialization is not done - This may happen in case 
         * if converged-load-balancer.xml is corrupt or not present at all
         */
        if (controller == null) {
            /* Only case it can be null is when it is not intialized for load-balancing
             * Cannot handle this request so setting error in response
             */
            _logger.logMsg(Level.SEVERE,
                    "HttpLoadBalancingManager :  controller is null." +
                    "Cannot handle this request, returning error");
            setErrorInResponse(response);
            return false;
        }

        String requestUri = httpReq.requestURI().getString();
        String appContextPath = null;
        int index = requestUri.indexOf("/", 1);

        if (index > -1) {
            appContextPath = requestUri.substring(0, index);
        } else {
            appContextPath = requestUri;
        }

        _logger.logMsg(Level.INFO,
            "HttpLoadBalancingManager : appContextPath " + appContextPath);

        RequestGroup reqGroup = controller.getRequestGroup(appContextPath);
        
        //if request group is not found lets fall back to default request group
        if(reqGroup == null){
            _logger.logMsg(Level.SEVERE,
                    "HttpLoadBalancingManager :  No request group found. Looking for " +
                    "default request group");
            reqGroup = controller.getDefaultHttpRequestGroup();
        }
            
        if (reqGroup == null) {
            /* No request group found for this request
             * let pass it to the same instance and see if it can handle it
             */
            _logger.logMsg(Level.SEVERE,
                    "HttpLoadBalancingManager :  No request group found to" +
                    "service the request. So passing it to the local instance itself");
            setLocalEndPoint(httpReq);
            return true;
        }

        _logger.logMsg(Level.SEVERE,
            "HttpLoadBalancingManager :  Request group found to service the request");

        reqGroup.serviceRequest(httpReq, (Response) response);

        if (httpReq.getConvergedLoadBalancerEndpoint().isLocal()) {
            _logger.logMsg(Level.SEVERE,
                "HttpLoadBalancingManager :  " + "Request " + requestUri +
                " will be serviced by local end-point");

            return true;
        } else {
            _logger.logMsg(Level.SEVERE,
                "HttpLoadBalancingManager :  " + "Request " + requestUri +
                " will be serviced by remote end-point");

            return false;
        }
    }

    private void setLocalEndPoint(final HttpRequest httpReq) {
        httpReq.setConvergedLoadBalancerEndpoint(EndPoint.getLocalEndPoint());
    }

    public Controller createController() {
        _logger.logMsg(Level.INFO,
            "Create controller called in HttpLoadBalancingManager");

        Controller newController = super.createController();
        newController.setLBType(CLBConstants.HTTP_CLB);

        return newController;
    }

    private void setErrorInResponse(Response response) {
        response.setStatus(500);
    }
}
