/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core;

import com.sun.grizzly.tcp.Response;

import org.jvnet.glassfish.comms.clb.proxy.http.util.HttpRequest;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.ArrayList;
import java.util.logging.Level;


/**
 *
 * @author kshitiz
 */
public class HttpRequestGroup implements RequestGroup {
    private static final LogUtil _logger = new LogUtil(HttpRequestGroup.class);
    private String appName;
    private ArrayList<ServerCluster> clusters;
    Router router;
    String lbpolicy;
    private Controller controller;
    private boolean isConsistentHashPolicy;

    /** Creates a new instance of HttpRequestGroup */
    public HttpRequestGroup(Controller controller, String appName) {
        this.controller = controller;
        this.appName = appName;
        clusters = new ArrayList<ServerCluster>();
    }

    public void addCluster(ServerCluster cluster) {
        clusters.add(cluster);
    }

    public void deleteCluster(ServerCluster cluster) {
        clusters.remove(cluster);
    }

    public void initializeRouter(String lbpolicy) throws CLBRuntimeException {
        _logger.logMsg(Level.INFO,
            "Intializing router for request group " + appName);
        this.lbpolicy = lbpolicy;

        if (lbpolicy.equals(RouterFactory.LBPolicy_CONSISTENT_HASH)) {
            router = RouterFactory.createRouter(RouterFactory.LBPolicy_STICKY_CONSISTENT_HASH,
                    clusters, false);
            isConsistentHashPolicy = true;
        } else if (lbpolicy.equals(RouterFactory.LBPolicy_ROUND_ROBIN)) {
            router = RouterFactory.createRouter(RouterFactory.LBPolicy_STICKY_ROUND_ROBIN,
                    clusters, false);
        } else {
            router = RouterFactory.createRouter(lbpolicy, clusters, false);
        }

        router.setController(controller);
        router.initialize();
        _logger.logMsg(Level.INFO,
            "Intialized router " + router + " with load-balancing policy " +
            lbpolicy + " for request group " + appName);
    }

    public void reinitializeRouter() throws CLBRuntimeException {
        _logger.logMsg(Level.INFO,
            "Re-intializing router for request group " + appName);
        router = RouterFactory.createRouter(lbpolicy, clusters, false);
        router.setController(controller);
        _logger.logMsg(Level.INFO,
            "Re-intialized router " + router + " with load-balancing policy " +
            lbpolicy + " for request group " + appName);
    }

    public void serviceRequest(HttpRequest req, Response res) {
        //increment counter of the number of request being currently serviced
        controller.incrementRequestCount();

        _logger.logMsg(Level.INFO,
            "Request group " + appName + " to service request " +
            req.requestURI().toString());

        EndPoint endPoint = null;

        ServerInstance instance = null;

        ConsistentHashRequest chrRequest = null;

        if (isConsistentHashPolicy) {
            chrRequest = new ConsistentHashRequest(req, null);
        }

        do {
            if (isConsistentHashPolicy) {
                instance = router.selectInstance(chrRequest);
            } else {
                instance = router.selectInstance(req);
            }
            assert instance != null : "The instance should not be null";

            _logger.logMsg(Level.SEVERE,
                "Selected server instance : " + instance.getName());

            boolean secure = req.isSecure();

            if (secure) {
                /* TBD
                 * As of now, ssl-offloading is done by default. May be it can be enabled/disabled
                 * using a property later
                 */

                //endPoint = instance.getEndPoint(CLBConstants.HTTPS_PROTOCOL);
                _logger.logMsg(Level.INFO,
                    "Offloading the secure request to non-secure end-point");
                endPoint = instance.getEndPoint(CLBConstants.HTTP_PROTOCOL);
            } else {
                endPoint = instance.getEndPoint(CLBConstants.HTTP_PROTOCOL);
            }

            if (endPoint != null) {
                break;
            }

            _logger.logMsg(Level.SEVERE,
                "Selected server instance : " + instance.getName() +
                " does not have suitable end-point");
        } while (true);

        _logger.logMsg(Level.INFO,
            "Selected endpoint : " + endPoint.getHost() + ":" +
            endPoint.getPort() + " from instance : " + instance.getName() +
            " to service the request : " + req.requestURI().toString());

        req.setConvergedLoadBalancerEndpoint(endPoint);

        if (lbpolicy.equals(RouterFactory.LBPolicy_ROUND_ROBIN)) {
            req.addHeader(CLBConstants.PROXY_HEADER_BEROUTE_KEY,
                instance.getID());
        } else if (lbpolicy.equals(RouterFactory.LBPolicy_CONSISTENT_HASH)) {
            req.addHeader(CLBConstants.PROXY_HEADER_BEKEY_KEY,
                chrRequest.getHashKey());
        } else {
            _logger.logMsg(Level.SEVERE, "Unknown load-balancing policy.");
        }

        //decrement counter of the number of request being currently serviced
        controller.decrementRequestCount();
    }

    public void cleanup() {
        _logger.logMsg(Level.INFO, "Cleaning up request group " + appName);
        clusters = null;
        controller = null;
        _logger.logMsg(Level.INFO, "Cleaned up request group " + appName);
    }
}
