/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core;

import org.jvnet.glassfish.comms.clb.core.common.chr.ConsistentHashRouter;
import org.jvnet.glassfish.comms.clb.core.common.chr.StickyConsistentHashRouter;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import java.util.HashMap;
import java.util.List;
import java.util.logging.Level;


/**
 * Factory class for creating router. The router are created based on load-balancing policy
 * as well as clusters part of this router.
 *
 * @author kshitiz
 */
public class RouterFactory {
    private static final LogUtil _logger = new LogUtil(RouterFactory.class);

    //consistent hash load-balancing policy name
    public static final String LBPolicy_CONSISTENT_HASH = "consistent-hash";

    //consistent hash load-balancing policy name
    public static final String LBPolicy_STICKY_CONSISTENT_HASH = "sticky-consistent-hash";

    //round robin load-balancing policy name
    public static final String LBPolicy_ROUND_ROBIN = "round-robin";

    //round robin load-balancing policy name
    public static final String LBPolicy_STICKY_ROUND_ROBIN = "sticky-round-robin";

    //map maintaining a router for each load balancing policy
    private static final HashMap<String, String> lbPolicyRouterMap = new HashMap<String, String>();

    static {
        //adding consistent hash routers to the list of router
        lbPolicyRouterMap.put(LBPolicy_CONSISTENT_HASH,
            ConsistentHashRouter.class.getName());
        lbPolicyRouterMap.put(LBPolicy_STICKY_CONSISTENT_HASH,
            StickyConsistentHashRouter.class.getName());

        //adding round-robin routers to the list of router
        lbPolicyRouterMap.put(LBPolicy_ROUND_ROBIN,
            RoundRobinRouter.class.getName());
        lbPolicyRouterMap.put(LBPolicy_STICKY_ROUND_ROBIN,
            StickyRoundRobinRouter.class.getName());
    }

    /**
     * Creates the router based on load-balancing policy and list of cluster provided. It can
     * create a new router at every call, or it can share a router created with given load-balancing
     * policy and list of clusters. Also there is a flag to indicate whether router need to route
     * to only active instances
     *
     * In general when router is created for RequestGroup, it need to cater to failover scenarios as well,
     * thus this flag is false.
     * However for a ServerCluster, it need to only route request to active instances within cluster. Thus
     * this flag is true in this case.
     *
     * @param lbpolicy the load-balancing policy
     * @param clusters a list of clusters
     * @param activeRouting flag to indicate whether router should only route to active instances only
     *
     * @return the router
     */
    public static Router createRouter(String lbpolicy,
        List<ServerCluster> clusters, boolean activeRouting)
        throws CLBRuntimeException {
        String className = lbPolicyRouterMap.get(lbpolicy);

        if (className == null) {
            //Log severe error message
            return null;
        }

        try {
            Class routerClass = Class.forName(className);
            Constructor routerConstructor = routerClass.getConstructor(List.class,
                    Boolean.TYPE);
            Router router = (Router) routerConstructor.newInstance(clusters,
                    activeRouting);

            return router;
        } catch (IllegalArgumentException ex) {
            _logger.logMsg(Level.SEVERE,
                "Unable to instantiate router." +
                " It will be due to error : " + ex.getMessage());
            throw new CLBRuntimeException("Unable to instantiate router", ex);
        } catch (InvocationTargetException ex) {
            _logger.logMsg(Level.SEVERE,
                "Unable to instantiate router." +
                " It will be due to error : " + ex.getMessage());
            throw new CLBRuntimeException("Unable to instantiate router", ex);
        } catch (IllegalAccessException ex) {
            _logger.logMsg(Level.SEVERE,
                "Unable to instantiate router." +
                " It will be due to error : " + ex.getMessage());
            throw new CLBRuntimeException("Unable to instantiate router", ex);
        } catch (InstantiationException ex) {
            _logger.logMsg(Level.SEVERE,
                "Unable to instantiate router." +
                " It will be due to error : " + ex.getMessage());
            throw new CLBRuntimeException("Unable to instantiate router", ex);
        } catch (SecurityException ex) {
            _logger.logMsg(Level.SEVERE,
                "Unable to instantiate router." +
                " It will be due to error : " + ex.getMessage());
            throw new CLBRuntimeException("Unable to instantiate router", ex);
        } catch (NoSuchMethodException ex) {
            _logger.logMsg(Level.SEVERE,
                "Unable to instantiate router." +
                " It will be due to error : " + ex.getMessage());
            throw new CLBRuntimeException("Unable to instantiate router", ex);
        } catch (ClassNotFoundException ex) {
            _logger.logMsg(Level.SEVERE,
                "Unable to instantiate router." +
                " It will be due to error : " + ex.getMessage());
            throw new CLBRuntimeException("Unable to instantiate router", ex);
        }
    }
}
