/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core;

import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.logging.Level;


/**
 * Implementation of interface ServerCluster
 * @author kshitiz
 */
public class ServerCluster {
    private static final LogUtil _logger = new LogUtil(ServerCluster.class);

    /* name of the cluster */
    private String name;

    /* map of server instance to server name */
    private HashMap<String, ServerInstance> instanceMap;

    /* router associated with this cluster */
    private Router router;

    /* flag indicating whether this is a self load-balancing cluster */
    private boolean selfLoadbalance;

    /** Creates a new instance of ServerClusterImple */
    public ServerCluster(String clusterName, boolean selfLoadbalance) {
        this.name = clusterName;
        this.selfLoadbalance = selfLoadbalance;
        instanceMap = new HashMap<String, ServerInstance>();
    }

    public boolean enableCluster() {
        throw new UnsupportedOperationException();
    }

    public boolean disableCluster() {
        throw new UnsupportedOperationException();
    }

    public boolean addInstance(ServerInstance instance) {
        instanceMap.put(instance.getName(), instance);

        return true;
    }

    public boolean deleteInstance(ServerInstance instance) {
        return (instanceMap.remove(instance) != null);
    }

    public boolean enableInstance(ServerInstance instance) {
        return instance.enableInstance();
    }

    public boolean disableInstance(ServerInstance instance) {
        return instance.disableInstance();
    }

    public Router getClusterRouter() {
        return router;
    }

    public void initializeClusterRouter(String lbpolicy, Controller controller)
        throws CLBRuntimeException {
        _logger.logMsg(Level.INFO,
            "Initializing the cluster router for cluster " + name);

        ArrayList<ServerCluster> clusterList = new ArrayList<ServerCluster>();
        clusterList.add(this);

        if (controller.getLBType() == CLBConstants.HTTP_CLB) {
            router = RouterFactory.createRouter(lbpolicy, clusterList, true);
            router.setController(controller);
        } else {
            router = RouterFactory.createRouter(RouterFactory.LBPolicy_CONSISTENT_HASH,
                    clusterList, true);
            router.setController(controller);
        }

        router.initialize();
        _logger.logMsg(Level.INFO,
            "Initialized the cluster router : " + router +
            " with load-balancing policy " + lbpolicy + " for cluster " + name);
    }

    public void initialize(int lbType) throws CLBRuntimeException {
        _logger.logMsg(Level.INFO, "Initializing the cluster " + name);

        for (ServerInstance instance : getAllInstances())
            instance.initialize(lbType);

        _logger.logMsg(Level.INFO, "Initialized the cluster " + name);
    }

    public void cleanup() {
        _logger.logMsg(Level.INFO, "Cleaning up the cluster " + name);

        for (ServerInstance instance : getAllInstances())
            instance.cleanup();

        instanceMap = null;
        router = null;
        _logger.logMsg(Level.INFO, "Cleaned up the cluster " + name);
    }

    public void reinitializeClusterRouter() {
        throw new UnsupportedOperationException();
    }

    public String getName() {
        return name;
    }

    public void addInstance(String instanceName, boolean enabled,
        int timeoutInMinutes, String[] listeners) throws CLBRuntimeException {
        _logger.logMsg(Level.INFO,
            "Adding instance " + instanceName + " to cluster " + name);

        if (getInstance(instanceName) != null) {
            throw new CLBRuntimeException("Instance with name : " +
                instanceName + " already exists in" + "cluster " + name + "." +
                "Cannot add duplicate.");
        }

        ServerInstance instance = new ServerInstance(this, instanceName,
                enabled, timeoutInMinutes);
        instance.addEndPoints(listeners);
        addInstance(instance);
    }

    public ServerInstance getInstance(String instanceName) {
        return instanceMap.get(instanceName);
    }

    public Collection<ServerInstance> getAllInstances() {
        return instanceMap.values();
    }

    public boolean isSelfLoadbalancing() {
        return selfLoadbalance;
    }

    public void markInstancesHealthy(List<String> instanceList) {
        for (String instanceName : instanceList) {
            ServerInstance instance = instanceMap.get(instanceName);
            _logger.logMsg(Level.INFO,
                "Marking instance " + instanceName + " of cluster " + name +
                " as healthy");
            instance.markAsHealthy();
        }
    }

    public void reconfig(ServerCluster oldCluster) {
        _logger.logMsg(Level.INFO, "Reconfiguring the cluster " + name);

        Collection<ServerInstance> oldInstances = oldCluster.getAllInstances();

        for (ServerInstance oldInstance : oldInstances) {
            ServerInstance newInstance = getInstance(oldInstance.getName());

            if (newInstance != null) {
                newInstance.reconfig(oldInstance);
            }
        }

        _logger.logMsg(Level.INFO, "Reconfigured the cluster " + name);
    }
}
