/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core;

import org.jvnet.glassfish.comms.clb.admin.CLBConfigurator;
import org.jvnet.glassfish.comms.clb.core.util.IDGenerator;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.logging.Level;


/**
 * Implementation of interface ServerInstance
 * @author kshitiz
 */
public class ServerInstance {
    private static final LogUtil _logger = new LogUtil(ServerInstance.class);

    /* name of the instance*/
    private String name;

    /* unique identity of this instance*/
    private String id;

    /* cluster with which this instance is associated with */
    private ServerCluster cluster;

    /* field indicating whether this instance is enabled or not */
    private boolean enabled;

    /* field indicating whether this instance is healthy or not */
    private boolean healthy;

    /* timeout in minutes */
    private int timeoutInMinutes;

    /* Map maintaining endpoints of the instance per protocol */
    private HashMap<String, ArrayList<EndPoint>> endPointProtocolMap;

    /* Index of endpoint returned on last call to getEndPoint(..)  */
    private HashMap<String, Integer> protocolEndPointIndex;

    /* Routers listening to event on this instance */
    private ArrayList<Router> routers;

    /* flag indicating whether instance is quiesced or not */
    private boolean quiesce;

    /* thread to take care of quiescing */
    private QuiesceThread quiesceThread;

    /* thread to take care of quiesce thread lock */
    private Object quiesceThreadLock;

    /* flag to mark that quiesce thread need to be started at the time of initialization */
    private boolean startQuiesceThread;

    /** Creates a new instance of ServerInstanceImpl */
    public ServerInstance(ServerCluster cluster, String instanceName,
        boolean enabled, int timeoutInMinutes) {
        this.cluster = cluster;
        this.name = instanceName;
        this.enabled = enabled;
        this.timeoutInMinutes = timeoutInMinutes;
        healthy = false;
        endPointProtocolMap = new HashMap<String, ArrayList<EndPoint>>();
        routers = new ArrayList<Router>();
    }

    public boolean enableInstance() {
        enabled = true;
        stopQuiesceThread();

        for (Router router : routers)
            router.handleEnableEvent(this);

        return true;
    }

    public boolean disableInstance() {
        enabled = false;

        for (Router router : routers)
            router.handleDisableEvent(this);

        startQuiesceThread();

        return true;
    }

    public void markAsUnhealthy() {
        healthy = false;

        for (Router router : routers) {
            _logger.logMsg(Level.INFO,
                "Sending failure event for instance " + name + " to router " +
                router);
            router.handleFailureEvent(this);
        }
    }

    public void markAsHealthy() {
        healthy = true;

        for (Router router : routers) {
            _logger.logMsg(Level.INFO,
                "Sending recovery event for instance " + name + " to router " +
                router);
            router.handleRecoveryEvent(this);
        }
    }

    public boolean addEndPoint(EndPoint endPoint) {
        ArrayList<EndPoint> endPointList = endPointProtocolMap.get(endPoint.getProtocol());

        if (endPointList == null) {
            endPointList = new ArrayList<EndPoint>();
            endPointProtocolMap.put(endPoint.getProtocol(), endPointList);
        }

        return endPointList.add(endPoint);
    }

    public boolean deleteEndPoint(EndPoint endPoint) {
        ArrayList<EndPoint> endPointList = endPointProtocolMap.get(endPoint.getProtocol());

        if (endPointList == null) {
            return false;
        }

        return endPointList.remove(endPoint);
    }

    public boolean enableEndPoint(EndPoint endPoint) {
        return endPoint.enableEndPoint();
    }

    public boolean disableEndPoint(EndPoint endPoint) {
        return endPoint.disableEndPoint();
    }

    public ServerCluster getServerCluster() {
        return cluster;
    }

    public EndPoint getEndPoint(String protocol) {
        ArrayList<EndPoint> endPointList = endPointProtocolMap.get(protocol);

        if (endPointList.size() < 1) {
            return null;
        }

        if (endPointList.size() == 1) {
            return endPointList.get(0);
        }

        Integer index = protocolEndPointIndex.get(protocol);

        if (index == null) {
            index = new Integer(0);
        } else {
            index = new Integer((index.intValue() + 1) % endPointList.size());
        }

        protocolEndPointIndex.put(protocol, index);

        return endPointList.get(index.intValue());
    }

    public boolean addRouter(Router router) {
        return routers.add(router);
    }

    public String getName() {
        return name;
    }

    public boolean isEnabled() {
        return enabled;
    }

    public boolean isDisabled() {
        return !enabled;
    }

    public boolean isHealthy() {
        return healthy;
    }

    public boolean isQuiesced() {
        return quiesce;
    }

    public void addEndPoints(String[] listeners) throws CLBRuntimeException {
        //flag to indicate whether this is a local instance
        boolean local = false;

        //It can be local instance only in case of self-loadbalancing cluster
        if (CLBConfigurator.getInstance().isSelfLoadbalanced()) {
            local = isLocalInstance();
        }

        for (int i = 0; i < listeners.length; i++) {
            String listener = listeners[i].trim();
            String protocol;

            if (listener.startsWith(CLBConstants.HTTP_PROTOCOL)) {
                protocol = CLBConstants.HTTP_PROTOCOL;
            } else if (listener.startsWith(CLBConstants.HTTPS_PROTOCOL)) {
                protocol = CLBConstants.HTTPS_PROTOCOL;
            } else if (listener.startsWith(CLBConstants.SIP_PROTOCOL)) {
                protocol = CLBConstants.SIP_PROTOCOL;
            } else if (listener.startsWith(CLBConstants.SIPS_PROTOCOL)) {
                protocol = CLBConstants.SIPS_PROTOCOL;
            } else if (listener.indexOf("://") == -1) {
                //protocol is not specified
                //mark it as http listener
                protocol = CLBConstants.HTTP_PROTOCOL;
            } else {
                throw new CLBRuntimeException("Provided protocol for listener " +
                    listeners[i] + " is unknown");
            }

            if (listener.indexOf(protocol) > -1) {
                listener = listener.substring(protocol.length());
            }

            if (listener.length() == 0) {
                throw new CLBRuntimeException(
                    "Host name and port number not specified for listener");
            }

            String hostName = null;
            int port = -1;
            int index;

            if ((index = listener.indexOf(":")) > -1) {
                hostName = listener.substring(0, index);

                try {
                    port = Integer.valueOf(listener.substring(index + 1));
                } catch (NumberFormatException ex) {
                    throw new CLBRuntimeException("The specified port is not a number",
                        ex);
                }
            } else {
                hostName = listener;

                //Since port number is not specified
                //take default port for that protocol
                if (protocol.equals(CLBConstants.HTTP_PROTOCOL)) {
                    port = CLBConstants.DEFAULT_HTTP_PORT;
                } else if (protocol.equals(CLBConstants.HTTPS_PROTOCOL)) {
                    port = CLBConstants.DEFAULT_HTTPS_PORT;
                } else if (protocol.equals(CLBConstants.SIP_PROTOCOL)) {
                    port = CLBConstants.DEFAULT_SIP_PORT;
                } else {
                    port = CLBConstants.DEFAULT_SIPS_PORT;
                }
            }

            EndPoint endPoint = new EndPoint(this, protocol, hostName, port,
                    local);
            addEndPoint(endPoint);
        }
    }

    public String getID() {
        return id;
    }

    private void generateID(int lbType) throws CLBRuntimeException {
        ArrayList<EndPoint> endPointList;
        EndPoint endPoint = null;
        boolean secure = false;

        if (lbType == CLBConstants.SIP_CLB) {
            endPointList = endPointProtocolMap.get(CLBConstants.SIP_PROTOCOL);

            if ((endPointList != null) && (endPointList.size() > 0)) {
                endPoint = endPointList.get(0);
            } else {
                endPointList = endPointProtocolMap.get(CLBConstants.SIPS_PROTOCOL);

                if ((endPointList != null) && (endPointList.size() > 0)) {
                    endPoint = endPointList.get(0);
                    secure = true;
                }
            }
        } else {
            endPointList = endPointProtocolMap.get(CLBConstants.HTTP_PROTOCOL);

            if ((endPointList != null) && (endPointList.size() > 0)) {
                endPoint = endPointList.get(0);
            } else {
                endPointList = endPointProtocolMap.get(CLBConstants.HTTPS_PROTOCOL);

                if ((endPointList != null) && (endPointList.size() > 0)) {
                    endPoint = endPointList.get(0);
                    secure = true;
                }
            }
        }

        if (endPoint == null) {
            throw new CLBRuntimeException(
                "No endpoints found for the server instance");
        }

        id = IDGenerator.getId(endPoint.getHost(), endPoint.getPort(), secure);
    }

    public void reconfig(ServerInstance oldInstance) {
        if ((this.isDisabled() && oldInstance.isEnabled()) ||
                (oldInstance.isDisabled() && !oldInstance.isQuiesced() &&
                this.isDisabled())) {
            startQuiesceThread = true;
        } else if (oldInstance.isDisabled() && oldInstance.isQuiesced() &&
                this.isDisabled()) {
            quiesce = true;
        }
    }

    public void initialize(int lbType) throws CLBRuntimeException {
        generateID(lbType);

        if (startQuiesceThread) {
            startQuiesceThread();
        }
    }

    public void cleanup() {
        stopQuiesceThread();
        cluster = null;
        endPointProtocolMap = null;
        protocolEndPointIndex = null;
        routers = null;
        quiesceThreadLock = null;
    }

    @Override
    public String toString() {
        // Must have a consistent toString() so that consistent hash works.  
        return cluster.getName()+"/"+name;
    }


    private void stopQuiesceThread() {
        if (quiesceThreadLock != null) {
            synchronized (quiesceThreadLock) {
                if (quiesceThread != null) {
                    quiesceThread.killThread();
                    quiesceThread.interrupt();
                    waitForQuiesceThreadCompletion();
                }
            }
        }
    }

    private void startQuiesceThread() {
        if (quiesceThreadLock == null) {
            quiesceThreadLock = new Object();
        }

        synchronized (quiesceThreadLock) {
            quiesceThread = new QuiesceThread(timeoutInMinutes);
            quiesceThread.start();
        }
    }

    private void cleanupQuiesceThread() {
        synchronized (quiesceThreadLock) {
            quiesceThread = null;
            quiesceThreadLock.notify();
        }
    }

    private void waitForQuiesceThreadCompletion() {
        synchronized (quiesceThreadLock) {
            while (quiesceThread != null) {
                try {
                    quiesceThreadLock.wait();
                } catch (InterruptedException ex) {
                    _logger.logMsg(Level.SEVERE,
                        "Got interrupted exception" + ex.getMessage());
                    _logger.logMsg(Level.FINE, ex, "Exception trace");
                }
            }
        }
    }

    /**
     * Method to check whether it is a local instance or a remote instance
     * Check is based on comparing the instance name
     */
    private boolean isLocalInstance() {
        String instanceName = com.sun.enterprise.server.ondemand.OnDemandServer.getServerContext()
                                                                               .getInstanceName();

        if (instanceName.equals(name)) {
            _logger.logMsg(Level.INFO, instanceName + " is a local instance");

            return true;
        } else {
            _logger.logMsg(Level.INFO,
                name + " is a remote instance to " + instanceName);

            return false;
        }
    }

    class QuiesceThread extends Thread {
        private long timeout;
        private long startTime;
        private boolean killThread;

        QuiesceThread(long timeout) {
            this.timeout = timeout;
        }

        public void run() {
            startTime = System.currentTimeMillis();

            long waitTime = timeout;

            while (true) {
                try {
                    Thread.sleep(waitTime);

                    break;
                } catch (InterruptedException ex) {
                    if (killThread) {
                        break;
                    }

                    _logger.logMsg(Level.SEVERE,
                        "Got interrupted exception" + ex.getMessage());
                    _logger.logMsg(Level.FINE, ex, "Exception trace");
                    waitTime = timeout -
                        (System.currentTimeMillis() - startTime);

                    if (waitTime <= 0) {
                        break;
                    }
                }
            }

            quiesce = true;
        }

        public long getTimeLeft() {
            return (timeout - (System.currentTimeMillis() - startTime));
        }

        private void killThread() {
            killThread = true;
        }
    }
}
