/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core.common.chr;

import com.ericsson.ssa.sip.SipFactoryImpl;
import com.ericsson.ssa.sip.dns.DnsResolver;

import org.jvnet.glassfish.comms.clb.core.ConsistentHashRequest;
import org.jvnet.glassfish.comms.clb.core.Controller;
import org.jvnet.glassfish.comms.clb.core.DCRFileUpdateEventListener;
import org.jvnet.glassfish.comms.clb.core.Router;
import org.jvnet.glassfish.comms.clb.core.ServerCluster;
import org.jvnet.glassfish.comms.clb.core.ServerInstance;
import org.jvnet.glassfish.comms.clb.core.common.chr.dcr.DcrConfigurableHashKeyExtractor;
import org.jvnet.glassfish.comms.clb.core.common.chr.dcr.DcrRulesException;
import org.jvnet.glassfish.comms.clb.core.common.chr.dcr.DcrUtils;
import org.jvnet.glassfish.comms.clb.core.util.ConsistentHash;
import org.jvnet.glassfish.comms.clb.proxy.http.util.HttpRequest;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.File;

import java.util.ArrayList;
import java.util.List;


/**
 * This is a router (common for HTTP and SIP) that implements the logic for
 * server instance selection based on DCR. It can be used both for cluster (then
 * it shall be set up by calling {@link #setClusterRouter(boolean)} with the
 * argument true) and for request groups (default).
 * <p>
 * Server selection is performed in several steps depending of configuration:
 *
 * <pre>
 * Extract hash key from request using configurable hash key extractor
 * IF NOT configured as a cluster router:
 *   First select server using a consistent hash which includes all servers in all
 *     clusters (and assumes that they are all healthy): the &quot;Consistent Hash for
 *     the Ideal Configuration&quot;
 *   Check that selected server is healthy, and if it is use it.
 *   Otherwise (if the selected server is faulty), select server using the &quot;Cluster
 *   Router&quot; of the cluster of the failed server
 * ENDIF
 * If no server was found above, select server using a router which contains only
 * the healthy servers of all clusters: the &quot;Consistent hash for the Actual
 * Configuration&quot;
 * Finally, if no servers were found at all return null.
 * </pre>
 */
public class ConsistentHashRouter implements Router, DCRFileUpdateEventListener {
    /** Parameter used to setup consistent hashes */
    private static final int POINTS_PER_NODE = 1024;
    private static LogUtil logger = new LogUtil(LogUtil.CLB_LOG_DOMAIN);

    /** Synchronization lock. */
    private final Object lock = new Object();

    /**
     * Consistent hash for the ideal configuration (not updated at server
     * failure). Contains all servers in all clusters.
     */
    private ConsistentHash<String, ServerInstance> idealConsistentHash;

    /**
     * Consistent hash for the ideal configuration (updated at server
     * failure/recovery). Contains all healthy servers in all clusters.
     */
    private ConsistentHash<String, ServerInstance> actualConsistentHash;

    /**
     * All clusters that are associated with this router and that are used for
     * routing at fail-over.
     */
    private List<ServerCluster> associatedClusters = new ArrayList<ServerCluster>();

    /** The composite hash key extractor. */
    private CompositeHashKeyExtractor hashKeyExtractor = new CompositeHashKeyExtractor();

    /** The XML file for DCR configuration. */
    private File dcrFile;
    private boolean ignoreIdealHash;
    private Controller controller;

    /**
     * @param associatedClusters
     */
    public ConsistentHashRouter(List<ServerCluster> associatedClusters,
        boolean activeRouting) {
        this.associatedClusters = associatedClusters;
        this.ignoreIdealHash = activeRouting;
    }

    /**
     * <b>Note</b>, this method assumes that the specified request is an
     * instance of {@link ConsistentHashRequest}.
     * <p>
     * Also note that the provided request will be updated, the hash key will be
     * inserted into it.
     *
     * @see org.jvnet.glassfish.comms.clb.core.Router#selectInstance(Request)
     */
    public ServerInstance selectInstance(ConsistentHashRequest request) {
        // Must be a ConsistentHashRequest
        if (request.getHashKey() == null) {
            // Extract the hash key...
            if (request.isHttp()) {
                request.setHashKey(hashKeyExtractor.getHashKey(
                        request.getHttpRequest()));
            } else {
                request.setHashKey(hashKeyExtractor.getHashKey(
                        request.getSipRequest()));
            }
        }

        // Then do routing based on the hash key...
        return getServerInstance(request);
    }

    /**
     * Gets the server instance for the specified request.
     *
     * @param req the request
     * @return the server instance for the given hash key
     */
    protected ServerInstance getServerInstance(ConsistentHashRequest req) {
        if (req.getHashKey() == null) {
            return null;
        }

        synchronized (lock) {
            ServerInstance server = null;

            if (!ignoreIdealHash) {
                // First try "ideal" router
                server = idealConsistentHash.get(req.getHashKey());
                assert server != null : "Should always be a value";

                /* Need to select another instance from cluster in below cases
                 * 1. Instance is unhealthy
                 * 2. Instance is disabled as well as quiesced
                 */
                if (!server.isHealthy() ||
                        (server.isDisabled() && server.isQuiesced())) {
                    //select another instance from cluster
                    server = ((ConsistentHashRouter) server.getServerCluster()
                                                           .getClusterRouter()).selectInstance(req);
                }
            }

            if (server == null) {
                // There where no servers available in the cluster (or this is a
                // cluster router); try all clusters
                server = actualConsistentHash.get(req.getHashKey());
            }

            return server;
        }
    }

    /**
     * Initializes this router.
     *
     * @see org.jvnet.glassfish.comms.clb.core.Router#initialize()
     */
    public boolean initialize() {
        synchronized (lock) {
            setupHashKeyExtractor();

            if (!ignoreIdealHash) {
                // Only create this if it is a request group router
                idealConsistentHash = new ConsistentHash<String, ServerInstance>(POINTS_PER_NODE);
            }

            actualConsistentHash = new ConsistentHash<String, ServerInstance>(POINTS_PER_NODE);

            for (ServerCluster cluster : associatedClusters) {
                for (ServerInstance server : cluster.getAllInstances()) {
                    server.addRouter(this);

                    if (idealConsistentHash != null) {
                        idealConsistentHash.addNode(server);
                    }

                    if (server.isHealthy() && server.isEnabled()) {
                        actualConsistentHash.addNode(server);
                    }
                }
            }

            if (idealConsistentHash != null) {
                idealConsistentHash.setup();
            }

            actualConsistentHash.setup();

            return true;
        }
    }

    public boolean reinitialize(ServerInstance serverInstance) {
        synchronized (lock) {
            actualConsistentHash.removeNode(serverInstance);

            if ((serverInstance != null) && serverInstance.isHealthy() &&
                    serverInstance.isEnabled()) {
                actualConsistentHash.addNode(serverInstance);
            }

            actualConsistentHash.setup();

            return true;
        }
    }

    /**
     * Sets the DCR file.
     *
     * @param dcrFile the DCR file
     */
    public void setDcrFile(File dcrFile) {
        this.dcrFile = dcrFile;
    }

    /**
     * Sets up the hash key extractor.
     * <p>
     * Pushes first a default hash key extractor and then a DCR extractor.
     */
    protected void setupHashKeyExtractor() {
        // First push the default hash key extractor...
        DefaultHashKeyExtractor defaultHashKeyExtractor = new DefaultHashKeyExtractor();
        pushHashKeyExtractor(defaultHashKeyExtractor);

        // Then DCR hash key extractor...
        DcrUtils.setup(DnsResolver.getInstance(), SipFactoryImpl.getInstance());

        // Only try to read DCR file if it exists; otherwise fall-back to Call-ID, from-tag (the default extractor)
        if (dcrFile != null && dcrFile.exists()) {
            try {
                DcrConfigurableHashKeyExtractor dcrHashKeyExtractor = new DcrConfigurableHashKeyExtractor(dcrFile.getAbsolutePath());
                pushHashKeyExtractor(dcrHashKeyExtractor);
            } catch (DcrRulesException e) {
                logger.severe("Failed to read DCR configuration: " + e +
                    " ; will only use default hash key extraction.");
            }
        }
    }

    /**
     * Pushes the specified hash key extractor on top of the others.
     *
     * @param hke the hash key extractor
     */
    protected void pushHashKeyExtractor(HashKeyExtractor hke) {
        hashKeyExtractor.push(hke);
    }

    public void handleDisableEvent(ServerInstance instance) {
        actualConsistentHash.removeNode(instance);
        actualConsistentHash.setup();
    }

    public void handleEnableEvent(ServerInstance instance) {
        actualConsistentHash.addNode(instance);
        actualConsistentHash.setup();
    }

    public void handleFailureEvent(ServerInstance instance) {
        actualConsistentHash.removeNode(instance);
        actualConsistentHash.setup();
    }

    public void handleRecoveryEvent(ServerInstance instance) {
        actualConsistentHash.addNode(instance);
        actualConsistentHash.setup();
    }

    public ServerInstance selectInstance(HttpRequest req) {
        throw new UnsupportedOperationException(
            "Not applicable for Consistent hash router.");
    }

    public void processDCRFileUpdateEvent() {
        setDcrFile(new File(controller.getDCRFileName()));
        setupHashKeyExtractor();
    }

    public void setController(Controller controller) {
        this.controller = controller;

        if (controller != null) {
            controller.getDCRFileUpdateEventNotifier()
                      .addDCRFileUpdateEventListener(this);
        }
    }
}
