/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.core.util;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;


/**
 * A consistent hash function. Maps arbitrary objects to one of a finite set of
 * nodes.
 * <p>
 * Usage details:
 * <p>
 * <ul>
 * <li>{@link #setup()} must be called after that nodes has been added or
 * removed.
 * <li>{@link #get(Object)} is used to get the node to which an arbitrary
 * object maps.
 * </ul>
 *
 * For more info, see, e.g.,
 * {@linkplain http://en.wikipedia.org/wiki/Consistent_hashing}
 */
public class ConsistentHash<K, V> {
    int pointsPerNode;
    HashMap<V, Integer> nodes = new HashMap<V, Integer>();
    int nPoints;
    Point<V>[] points;
    MessageDigest md5;

    /**
     * Creates the hash with the specified points per node.
     * @param pointsPerNode points per node
     */
    public ConsistentHash(int pointsPerNode) {
        this.pointsPerNode = pointsPerNode;

        try {
            md5 = MessageDigest.getInstance("md5");
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("can't get md5 instance");
        }
    }

    /**
     * Add a node to the mapped set.
     * <p>
     * <b>NOTE!</b> {@link #setup()} must be called before using the consistent
     * hash after one or more node(s) has been added.
     *
     * @param node the node
     * @param factor multiplication factor for points per node
     */
    public void addNode(V node, double factor) {
        nodes.put(node, new Integer((int) ((factor * pointsPerNode) + .5)));
        points = null; // crash and burn if user forgets setup()
    }

    /**
     * Add a node to the mapped set with factor = 1.0.
     * <p>
     * <b>NOTE!</b> {@link #setup()} must be called before using the consistent
     * hash after one or more node(s) has been added.
     *
     * @param node the node
     */
    public void addNode(V node) {
        addNode(node, 1.0);
    }

    /**
     * Remove node from the mapped set.
     * <p>
     * <b>NOTE!</b> {@link #setup()} must be called before using the consistent
     * hash after one or more node(s) has been removed.
     *
     * @param node the node to remove
     */
    public void removeNode(V node) {
        nodes.remove(node);
        points = null; // crash and burn if user forgets setup()
    }

    public void setup() {
        nPoints = 0;

        for (Iterator<Integer> it = nodes.values().iterator(); it.hasNext();)
            nPoints += (it.next()).intValue();

        points = new Point[nPoints];

        int point = 0;

        for (Iterator<V> keys = nodes.keySet().iterator(); keys.hasNext();) {
            Object node = keys.next();
            int n = nodes.get(node).intValue();

            for (int i = 0; i < n; i++)
                points[point++] = new Point(md5hash(node.toString() + "/" + i),
                        node);
        }

        Arrays.sort(points);
    }

    /**
     * Gets the node that the specified value maps to.
     *
     * @param value
     * @return the node that the specified value maps to
     */
    public V get(K value) {
        int hashValue = hash(value.toString());

        if (points.length == 0) {
            return null;
        }

        if (hashValue < points[0].start) {
            return points[nPoints - 1].node;
        }

        int lo = 0;
        int hi = nPoints;

        while (lo < (hi - 1)) {
            int mid = (lo + hi) / 2;

            if (points[mid].start > hashValue) {
                hi = mid;
            } else {
                lo = mid;
            }
        }

        return points[lo].node;
    }

    private int hash(String s) {
        int h = s.hashCode();

        // All the "entropy" is in the low bits; shift them up...
        return (h << 28) ^ (h << 24) ^ (h << 16) ^ h;
    }

    private int md5hash(String s) {
        md5.reset();

        byte[] digest = md5.digest(s.getBytes());

        return ((digest[12] & 0xff) << 24) | ((digest[13] & 0xff) << 16) |
        ((digest[14] & 0xff) << 8) | (digest[15] & 0xff);
    }

    private static class Point<V> implements Comparable<Point<V>> {
        int start;
        V node;

        Point(int start, V node) {
            this.start = start;
            this.node = node;
        }

        public int compareTo(Point<V> o) {
            int start2 = o.start;

            if (start < start2) {
                return -1;
            }

            if (start > start2) {
                return 1;
            }

            return 0;
        }

        @Override
        public String toString() {
            return "{" + node + "," + start + "}";
        }
    }
}
