/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.proxy;

import com.sun.enterprise.web.connector.grizzly.SelectorThread;

import com.sun.grizzly.ConnectorHandler;
import com.sun.grizzly.Controller;

import org.jvnet.glassfish.comms.clb.proxy.api.Endpoint;
import org.jvnet.glassfish.comms.clb.proxy.config.LoadBalancerProxyConstants;
import org.jvnet.glassfish.comms.clb.proxy.config.ProxyConfig;
import org.jvnet.glassfish.comms.clb.proxy.config.RoundRobinPolicy;
import org.jvnet.glassfish.comms.clb.proxy.outbound.ConnectionManager;
import org.jvnet.glassfish.comms.clb.proxy.util.Reader;
import org.jvnet.glassfish.comms.httplayers.HttpLayer;

import java.io.IOException;

import java.net.SocketAddress;

import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;

import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * This is a singleton class that is responsible for proxying a Http
 * request to a remote instances.
 *
 * @author rampsarathy
 */
public class HttpProxy {
    /**
     * Singleton object.
     */
    private static HttpProxy _thisInstance = null;

    /**
     * Manages the outbound connections
     */
    private ConnectionManager connectionManager;

    /**
     * Logger.
     */
    private Logger _logger;

    /**
     * The pluggable layer framework handler
     */
    private ProxyLayerHandler layerHandler = null;

    /**
     * Flag that indicates if proxy functionality
     * has to be enabled in the grzzly connector.
     */
    private boolean proxyDisabled = true;

    /**
     * Creates a new instance of HttpProxy
     */
    private HttpProxy() {
        connectionManager = new ConnectionManager();
        connectionManager.createConnectionHandlerPool();
        layerHandler = new ProxyLayerHandler();
        _logger = ProxyConfig.getInstance().getLogger();
        _logger.log(Level.INFO, "HttpProxy constructor called");
    }

    /**
     * Sets the logger
     */
    public void setLogger(Logger logger) {
        this._logger = logger;
    }

    /**
     * Exposed interface to add a layer.
     */
    public void addLayer(HttpLayer layer) {
        _logger.log(Level.INFO,
            "HttpProxy addLayer called" + layer.getClass().getName());

        if (proxyDisabled) {
            enableProxy();
        }

        this.layerHandler.addLayer(layer);
    }

    /**
     * Exported interface to add pluggable layers.
     */
    public void addLayers(List<HttpLayer> layers) {
        if (proxyDisabled) {
            enableProxy();
        }

        this.layerHandler.addLayers(layers);
    }

    /**
     * Getter for proxy status.
     */
    public boolean proxyDisabled() {
        return proxyDisabled;
    }

    /**
     * Enables the proxy functionality in the connector.
     */
    private void enableProxy() {
        try {
            System.setProperty(ProxyConfig.PIPELINE_CLASS_PROPERTY,
                ProxyConfig.PIPELINE_CLASS);
            proxyDisabled = false;
        } catch (Exception e) {
            _logger.log(Level.SEVERE,
                "Unable to set proxy system property," +
                "Proxy cannot be enabled");
        }
    }

    /**
     * Singleton method to get this instance.
     */
    public static synchronized HttpProxy getInstance() {
        if (_thisInstance == null) {
            _thisInstance = new HttpProxy();
        }

        return _thisInstance;
    }

    /**
     * Returns the pluggable framework invocation handler.
     */
    public ProxyLayerHandler getLayerHandler() {
        return this.layerHandler;
    }

    /**
     * Singleton object, so ensure nobody clones this.
     */
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();

        // you should not be able to clone a singleton
    }

    /**
     * Gets the connection manager for this proxy.
     */
    public ConnectionManager getConnectionManager() {
        return this.connectionManager;
    }

    /**
     * Method to proxy an Http request.
     * This method always keeps the connection alive.
     *
     * @param payload information about the payload that needs to be proxied.
     * @param buff byte buffer that contains the data.
     * @param key Selection key corresponding to the inbound channel.
     */
    public boolean doProxyHttp(ProxyTask task, ByteBuffer buff, SelectionKey key) {
        Endpoint backendendpoint = null;
        SocketAddress clientaddress = ((SocketChannel) key.channel()).socket()
                                       .getRemoteSocketAddress();
        boolean cachehandler = true;

        Endpoint clientendpoint = new DefaultEndpoint(clientaddress);
        int bytesRead = 0;
        _logger.log(Level.INFO, "In doProxyHttp , task is " + task);

        if (task != null) {
            /**
             * taskInfo contains the buffer, length of the task
             * and the bytesWritten field, with this and the key for the
             * inbound channel we should be ready to read from
             * the client channel  and write bytes into server channel
             */
            connectionManager.registerServerEndpoint(key, task);
            /*
             * We have the buff, start writing, if we want more we just
             * have to return to the selector and wait for data
             */
            bytesRead = buff.limit();
        } else {
            /**
             * If we dont have a task object it means that
             * we are receiving more data from an existing channel for
             * a known request that is being processed, for which
             * we had registered a ReadTask. Read the bytes from the channel and
             * send it.
             */
            task = connectionManager.getServerEndpoint(key);

            try {
                bytesRead = Reader.read(buff, (SocketChannel) key.channel(),
                        LoadBalancerProxyConstants.DEFAULT_READ_TIMEOUT);
            } catch (IOException ex) {
                ex.printStackTrace();
                task.setKeepAlive(false);

                return false;
            }
        }

        if (task == null) {
            _logger.log(Level.INFO,
                "In doProxyHttp , FATAL ERROR task is null " + task);

            // throw exception
            return false;
        }

        long bytesWritten = 0;

        if (bytesRead <= 0) {
            /* Go back to Grizzly 1.0 selector thread and
             * register the read key, return true will ensure keepalive.
             */
            return cachehandler;
        } else {
            /* Send the buffer through the handler.
             */
            ProxyTask backend = connectionManager.getServerEndpoint(key);

            try {
                bytesWritten = connectionManager.send(buff, backend, key,
                        task.getKeepAlive());
                _logger.log(Level.INFO,
                    "Bytes written to socket " + bytesWritten);
            } catch (Exception ex) {
                ex.printStackTrace();

                try {
                    bytesWritten = connectionManager.send(buff, backend, key,
                            task.getKeepAlive());
                    _logger.log(Level.FINE,
                        "Wrote bytes on retry " + bytesWritten);
                } catch (Exception ee) {
                    ee.printStackTrace();
                    task.setKeepAlive(false);
                }

                return false;
            }
        }

        task.setBytesWritten(task.getBytesWritten() + bytesWritten);

        if (!task.hasRemaining()) {
            _logger.log(Level.FINE, "Setting remove handler to false");
            cachehandler = false;
        }

        return cachehandler;
    }

    /**
     * Selector thread that would help the connection manager
     * to park the requests if the response from the remote host takes
     * a long time
     */
    public void setSelectorThread(SelectorThread selthread) {
        connectionManager.setSelectorThread(selthread);
    }

    /**
     * This is called by the handler when the selection key expires.
     * We have to handle this by removing the connection reference
     * and also cancelling the selection key corresponding to the
     * backend channel.
     */
    public void expireKey(SelectionKey key) {
        //do something , because we got a notification saying
        // the client channel key has expired
        connectionManager.removeClientEndpoint(key);
    }
}
