/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.proxy.http;

import com.sun.enterprise.web.connector.grizzly.ssl.SSLOutputBuffer;
import com.sun.enterprise.web.portunif.*;
import com.sun.enterprise.web.portunif.util.ProtocolInfo;

import com.sun.grizzly.http.SocketChannelOutputBuffer;
import com.sun.grizzly.tcp.Request;
import com.sun.grizzly.tcp.Response;
import com.sun.grizzly.util.http.MimeHeaders;

import org.jvnet.glassfish.comms.clb.proxy.HttpProxy;
import org.jvnet.glassfish.comms.clb.proxy.ProxyTask;
import org.jvnet.glassfish.comms.clb.proxy.api.Endpoint;
import org.jvnet.glassfish.comms.clb.proxy.config.ProxyConfig;
import org.jvnet.glassfish.comms.clb.proxy.http.util.ObjectManager;
import org.jvnet.glassfish.comms.clb.proxy.portunif.SailfinClbProxyProtocolInfo;

import java.io.IOException;

import java.nio.channels.SelectionKey;

import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Redirect the request to the proper protocol, which can be http or https.
 *
 * @author
 */
public class LoadBalancerProxyHandler implements ProtocolHandler {
    /**
     * The protocols supported by this handler.
     */
    protected String[] protocols = { "lb/http", "lb/https" };
    private HttpProxy proxy;
    private Logger _logger = null;
    private ObjectManager objManager;

    public LoadBalancerProxyHandler() {
        _logger = ProxyConfig.getInstance().getLogger();
        proxy = HttpProxy.getInstance();
        objManager = ObjectManager.getInstance();
    }

    /**
     * Redirect the request to the protocol defined in the
     * <code>protocolInfo</code>. Protocols supported are http and https.
     *
     * @param protocolInfo The protocol that needs to be redirected.
     */
    public void handle(ProtocolInfo protocolInfo) throws IOException {
        try {
            _handle(protocolInfo);
        } catch (Throwable t) {
            t.printStackTrace();
            throw new IOException(t.getMessage());
        }
    }

    private void _handle(ProtocolInfo protocolInfo) throws IOException {
        Object ptask = ((SailfinClbProxyProtocolInfo) protocolInfo).object;
        _logger.log(Level.FINE, "Inside Protocol Handler " + protocolInfo.key);

        ProxyTask task = null;
        Endpoint endpoint = null;
        Request request = null;
        Response response = null;
        proxy.setSelectorThread(((SailfinClbProxyProtocolInfo) protocolInfo).selectorThread);

        if (ptask != null) {
            task = (ProxyTask) ptask;
            response = task.getResponse();
            request = task.getRequest();

            if (response.getStatus() != 200) {
                _logger.log(Level.SEVERE, "Sending error response in Handler");
                /** Set
                 * Connection close here
                 */
                response.setHeader("connection", "close");

                if (task.isSecure()) {
                    sendSecureResponse(request, response);
                } else {
                    sendResponse(request, response);
                }

                _logger.log(Level.SEVERE, "Sent error response in Handler");
                task.recycle();
                objManager.offerTask(task, protocolInfo.isSecure);
                _logger.log(Level.SEVERE, "Released reources");
                protocolInfo.keepAlive = false;

                return;
            }

            endpoint = task.getEndpoint();
            _logger.log(Level.FINE,
                "LoadBalancer protocol handler invoked with " + "  endpoint " +
                endpoint);
        } else {
            /*
             * Means its not a new request but there is more data to
             * read from an older socket
             */
            _logger.log(Level.FINE,
                "LoadBalancer protocol handler invoked because" +
                "  there is more data to read");
        }

        /** Invoke the proxy API
         */

        /**
         * We have
         * the buffer where the headers are read completely
         * the key
         * the remote address is available through a ThreadLocal
         * object from the HA LB. This invocation is just like whats done
         * in WSTCPProtocolHandler
         */
        ((SailfinClbProxyProtocolInfo) protocolInfo).cacheHandler = proxy.doProxyHttp(task,
                protocolInfo.byteBuffer, protocolInfo.key);
        _logger.log(Level.INFO,
            "doProxyHttp returned " +
            ((SailfinClbProxyProtocolInfo) protocolInfo).cacheHandler);

        /* If we return false here the grizzly 1.0 controller will cancel key and close
         * the channel. Its a weird scenario because we will have to wait until a
         * response is received from the backend, whateven async processing we
         * do with the backend is of little use becase this thread cannot return.
         * If ret is true it means that the current requets has been read completely and
         * any more bytes on the chanenl is a new request which has to go thro
         * the finder
         *
         */
        if (!((SailfinClbProxyProtocolInfo) protocolInfo).cacheHandler) {
            protocolInfo.mappedProtocols.remove(protocolInfo.key);

            if (task == null) {
                task = proxy.getConnectionManager()
                            .getServerEndpoint(protocolInfo.key);
            }

            if (task != null) {
                task.recycle();
                objManager.offerTask(task, protocolInfo.isSecure);
            } else {
                _logger.log(Level.SEVERE, " Cannot release task");
            }
        }

        // need to see how we should handle keepAlive here
        // always keep alive for now, because we will cancel key through the
        // call back
        protocolInfo.keepAlive = true;

        // unless we know that we can close the channel park request should be true
        // we have to close the channel once the response is written back
        ((SailfinClbProxyProtocolInfo) protocolInfo).parkRequest = true;
        _logger.log(Level.INFO,
            "Load Balancer Handler returning keep alive " +
            protocolInfo.keepAlive);
    }

    //=========================================================

    /**
     * Returns an array of supported protocols.
     * @return an array of supported protocols.
     */
    public String[] getProtocols() {
        return protocols;
    }

    /**
     * Invoked when the SelectorThread is about to expire a SelectionKey.
     * @return true if the SelectorThread should expire the SelectionKey, false
     *              if not.
     */
    public boolean expireKey(SelectionKey key) {
        _logger.log(Level.INFO, "Expire calles on key " + key);
        proxy.expireKey(key);

        return true;
    }

    protected void sendResponse(Request request, Response response) {
        SocketChannelOutputBuffer outputBuffer = (SocketChannelOutputBuffer) response.getOutputBuffer();

        MimeHeaders headers = response.getMimeHeaders();
        headers.setValue("Connection").setString("close");

        // Build the response header
        outputBuffer.sendStatus();

        int size = headers.size();

        for (int i = 0; i < size; i++) {
            outputBuffer.sendHeader(headers.getName(i), headers.getValue(i));
        }

        outputBuffer.endHeaders();

        try {
            outputBuffer.endRequest();
            outputBuffer.flush();
            outputBuffer.commit();
        } catch (IOException ex) {
            ex.printStackTrace();
        }
    }

    protected void sendSecureResponse(Request request, Response response) {
        /**
         * Cannot use this outputBuffer here because this will cause a class
         * cast exception when flushing the buffer, when the worker thread
         * is casted. The thread on which we will run on is a 1.5 worker thread
         * but what the SSLOutputWriter looks for is a 1.0 worker thread.
         * So we have to send the bytes on the channel ourselves here.
         */
        SSLOutputBuffer outputBuffer = (SSLOutputBuffer) response.getOutputBuffer();

        MimeHeaders headers = response.getMimeHeaders();
        headers.setValue("Connection").setString("close");

        // Build the response header
        outputBuffer.sendStatus();

        int size = headers.size();
        // for (int i = 0; i < size; i++) {
        //   outputBuffer.sendHeader(headers.getName(i), headers.getValue(i));        }
        outputBuffer.endHeaders();

        try {
            outputBuffer.endRequest();
            outputBuffer.flush();
            outputBuffer.commit();
        } catch (IOException ex) {
            ex.printStackTrace();
        }
    }
}
