/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.proxy.outbound;


/**
 * The contentlength algo of GRIZZLY 1.5 has not been stanged yet
 * this should be updated to 1.5 when the staging is done.
 */
import com.sun.grizzly.CallbackHandler;
import com.sun.grizzly.ConnectorHandler;
import com.sun.grizzly.Context;
import com.sun.grizzly.Controller;
import com.sun.grizzly.IOEvent;
import com.sun.grizzly.filter.ReadFilter;
import com.sun.grizzly.http.algorithms.ContentLengthAlgorithm;
import com.sun.grizzly.util.OutputWriter;
import com.sun.grizzly.util.WorkerThread;

import org.jvnet.glassfish.comms.clb.proxy.HttpProxy;
import org.jvnet.glassfish.comms.clb.proxy.api.Endpoint;
import org.jvnet.glassfish.comms.clb.proxy.api.Protocols;
import org.jvnet.glassfish.comms.clb.proxy.config.ProxyConfig;

import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;

import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Handler that will be invoked by the Grizzly 1.5 pipeline code when one
 * of the events READ,WRITE or CONNECT occur for the seclection key corresponding
 * to this handler.
 *
 * @author rampsarathy
 */
public class AsyncResponseCallBackHandler implements CallbackHandler<Context> {
    private ConnectorHandler connectorHandler;
    private Logger _logger = null;
    private SelectionKey clientKey;
    private ReadFilter readFilter;
    private ContentLengthAlgorithm algorithm;
    private Endpoint ep;
    private boolean keepAlive = false;
    private ConnectionManager connectionManager;

    /**
     * This handler requires the connectorhandler that is handing this connection
     * the client socket channel to send the response back to and
     * the endpoint.
     */
    public AsyncResponseCallBackHandler(Endpoint ep,
        ConnectorHandler connectorHandler, SelectionKey clientkey,
        boolean keepalive) {
        _logger = ProxyConfig.getInstance().getLogger();
        this.connectorHandler = connectorHandler;
        this.ep = ep;
        this.clientKey = clientkey;
        this.keepAlive = keepalive;
        this.connectionManager = HttpProxy.getInstance().getConnectionManager();
        algorithm = new ContentLengthAlgorithm();
    }

    public void setKeepAlive(boolean keepalive) {
        this.keepAlive = keepalive;
    }

    /**
     * Called when the connection has been established.
     */
    public void onConnect(IOEvent<Context> ioEvent) {
        SelectionKey key = ioEvent.attachment().getSelectionKey();
        _logger.log(Level.FINE, "OnConnect Key " + key);

        if (key != null) {
            _logger.log(Level.FINE, "OnConnect chanel " + key.channel());
        }

        /**
         * Sometimes the connect is called with a NULL key
         * BUG ??
         */
        try {
            connectorHandler.finishConnect(key);
        } catch (Exception e) {
            e.printStackTrace();

            //clean
            return;
        }

        _logger.log(Level.FINE, "Connection created with " + key);

        ioEvent.attachment().getController()
               .registerKey(key, SelectionKey.OP_READ, Controller.Protocol.TCP);
    }

    public void onRead(IOEvent<Context> ioEvent) {
        Context ctx = ioEvent.attachment();

        try {
            SocketChannel clientChannel = (SocketChannel) clientKey.channel();
            readFilter = new ReadFilter();
            readFilter.execute(ctx);

            WorkerThread workerThread = (WorkerThread) Thread.currentThread();
            ByteBuffer byteBuffer = workerThread.getByteBuffer();
            algorithm.setSocketChannel((SocketChannel) ctx.getSelectionKey()
                                                          .channel());
            algorithm.preParse(byteBuffer);

            /**
             * Improve the logic below to write as much as we have and
             * then continue parsing in the next read, we should not
             * wait for the entire message before we start writing.
             */
            if (algorithm.parse(byteBuffer)) {
                _logger.log(Level.FINE,
                    "Algorithm says finished : " +
                    "Writing data into channel " +
                    clientChannel.socket().getRemoteSocketAddress());

                if ((clientChannel != null) && (clientChannel.isConnected())) {
                    OutputWriter.flushChannel(clientChannel, byteBuffer);
                } else {
                    _logger.log(Level.FINE,
                        "Client closed channel, " + "cannot write response");
                    throw new Exception("Socket channel closed");
                }

                /**
                 * There are problems in re-using the same client channel
                 * for further requests , so closing client
                 * and server channels for now, Need to improve this..
                 */

                // check for keep-alive and then close the channel,
                // check the connection header
                connectionManager.removeClientEndpoint(clientKey);
                connectionManager.releaseConnection(this.connectorHandler);

                //clientChannel.close();
                if (!keepAlive) {
                    connectionManager.cancelClientKey(clientKey);
                } else {
                    connectionManager.registerClientKey(clientKey);
                }

                //   ctx.getController().cancelKey(ctx.getSelectionKey());
            } else {
                SelectionKey key = ctx.getSelectionKey();

                if ((clientChannel != null) && (clientChannel.isConnected())) {
                    OutputWriter.flushChannel(clientChannel, byteBuffer);
                } else {
                    _logger.log(Level.FINE,
                        "Client closed channel, " + "cannot write response");
                    throw new Exception("Socket channel closed");
                }

                _logger.log(Level.FINE,
                    "Algorithm says there is more data to write to " +
                    clientChannel.socket().getRemoteSocketAddress());
                ctx.getController().registerKey(key, SelectionKey.OP_READ);
            }

            algorithm.postParse(byteBuffer);
            byteBuffer.clear();
        } catch (Throwable e) {
            ctx.getController().cancelKey(ctx.getSelectionKey());
            HttpProxy.getInstance().getConnectionManager()
                     .releaseConnection(this.connectorHandler);
            connectionManager.removeClientEndpoint(clientKey);
            e.printStackTrace();
        }
    }

    public void onWrite(IOEvent<Context> ioEvent) {
        throw new IllegalStateException("Should not be here!");
    }
}
