/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.clb.proxy.outbound;


// Grizzly 1.0 APIs
import com.sun.enterprise.web.connector.grizzly.SelectorThread;

// Grillzy 1.x.x APIs
import com.sun.grizzly.CallbackHandler;
import com.sun.grizzly.ConnectorHandler;
import com.sun.grizzly.ConnectorHandlerPool;
import com.sun.grizzly.Controller;
import com.sun.grizzly.SelectorHandler;
import com.sun.grizzly.TCPSelectorHandler;

import org.jvnet.glassfish.comms.clb.proxy.ProxyTask;
import org.jvnet.glassfish.comms.clb.proxy.api.Endpoint;
import org.jvnet.glassfish.comms.clb.proxy.config.ProxyConfig;
import org.jvnet.glassfish.comms.clb.proxy.outbound.ssl.SSLResponseCallBackHandler;

import java.io.IOException;

import java.net.SocketAddress;

import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;

import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Manages the outbound connections.
 * @author
 */
public class ConnectionManager {
    /**
     * Table for holding the proxy to backend instance mapprings.
     */
    private ConcurrentHashMap<Endpoint, ConnectorHandler> proxyTobackend = new ConcurrentHashMap<Endpoint, ConnectorHandler>();

    /**
     * Table for holding the client to proxy channel mappring.
     */
    private ConcurrentHashMap<SelectionKey, ProxyTask> clientToproxy = new ConcurrentHashMap<SelectionKey, ProxyTask>();

    /**
     * Cacheable pool of handlers.
     */
    protected ConnectorHandlerPool cacheableHandlerPool;

    /**
     * Grizzly 15 controller.
     */
    protected Controller controller;

    /**
     * Selector handler for this controller.
     */
    private SelectorHandler selectorHandler;

    /**
     * Logger
     */
    private Logger _logger = null;
    private SelectorThread selectorThread = null;

    /** Creates a new instance of ConnectionManager */
    public ConnectionManager() {
        _logger = ProxyConfig.getInstance().getLogger();
    }

    /**
     * Creates the handler pool and registers the controller.
     */
    public void createConnectionHandlerPool() {
        controller = new Controller();
        selectorHandler = new TCPSelectorHandler(true);
        controller.setSelectorHandler(selectorHandler);

        /**
         * Have to move this thread creation part to the
         * ConvergedProxy.
         */
        try {
            Thread t1 = new Thread(controller);
            t1.start();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void destroyConnectionHandlerPool() {
        // stop the controller 
    }

    /**
     * Creater a TCP client connection handler.
     */
    private ConnectorHandler createHandlerTCP(ProxyTask task,
        SelectionKey clientkey, boolean keepalive) throws IOException {
        Endpoint ep = task.getEndpoint();
        SocketAddress remote = ep.getSocketAddress();
        final ConnectorHandler connectorHandler = controller.acquireConnectorHandler(Controller.Protocol.TCP);
        CallbackHandler callbackHandler = null;

        if (task.isSecure()) {
            callbackHandler = new SSLResponseCallBackHandler(task.getEndpoint(),
                    connectorHandler, clientkey, keepalive, task.getSSLEngine());
        } else {
            callbackHandler = new AsyncResponseCallBackHandler(task.getEndpoint(),
                    connectorHandler, clientkey, keepalive);
        }

        try {
            connectorHandler.connect(remote, callbackHandler);
        } catch (Exception ex) {
            //ex.printStackTrace();
        }

        _logger.log(Level.FINE,
            "Created TCP handler, Socket address " +
            connectorHandler.getUnderlyingChannel() + "Local : " +
            ((SocketChannel) connectorHandler.getUnderlyingChannel()).socket()
             .getLocalAddress() + "Remote : " +
            ((SocketChannel) connectorHandler.getUnderlyingChannel()).socket()
             .getRemoteSocketAddress() + " for  " + task.getEndpoint());

        return connectorHandler;
    }

    public void putHandler(Endpoint ep, ConnectorHandler handler) {
        ConnectorHandler cached = proxyTobackend.putIfAbsent(ep, handler);
        _logger.log(Level.FINE,
            "Putt Endpoint " + ep + " and handler " + handler);

        if (cached != null) {
            try {
                handler.close(); //Race cond, remove if multiple are ok.
            } catch (Exception e) {
                ;
            }
        }
    }

    /**
     * Utility method for sending a byte buffer over a channel.
     */
    public long send(ByteBuffer bb, ProxyTask task, SelectionKey key,
        boolean keepalive) throws IOException {
        long bytesWritten = 0;
        int retrycount = 0;
        Endpoint ep = task.getEndpoint();

        if (!ep.isSecure()) { //TODO move Protocols
            _logger.log(Level.FINE, "Get Endpoint " + ep);

            ConnectorHandler handler = proxyTobackend.get(ep);
            _logger.log(Level.FINE, "Handler for " + ep + " is " + handler);

            if (handler == null) {
                handler = createHandlerTCP(task, key, keepalive);
            }

            while (!handler.getUnderlyingChannel().isOpen() &&
                    (retrycount < ProxyConfig.MAX_SEND_RETRY)) {
                retrycount++;
                handler = createHandlerTCP(task, key, keepalive);
            }

            bb.flip();

            try {
                bytesWritten = handler.write(bb, true);
            } catch (IOException ex) {
                proxyTobackend.remove(ep);
                _logger.log(Level.SEVERE,
                    "Write failed: " + handler.getUnderlyingChannel());
                throw ex;
            }
        } else {
            throw new UnsupportedOperationException();
        }

        return bytesWritten;
    }

    public void setSelectorThread(SelectorThread thread) {
        this.selectorThread = thread;
    }

    public void registerServerEndpoint(SelectionKey key, ProxyTask beEndpoint) {
        clientToproxy.put(key, beEndpoint);
    }

    public ProxyTask getServerEndpoint(SelectionKey key) {
        return clientToproxy.get(key);
    }

    public ProxyTask removeClientEndpoint(SelectionKey key) {
        return clientToproxy.remove(key);
    }

    public void releaseConnection(ConnectorHandler handler) {
        if (handler != null) {
            try {
                handler.close();
            } catch (IOException ex) {
                ex.printStackTrace();
            }

            controller.releaseConnectorHandler(handler);
        }
    }

    public void cancelClientKey(SelectionKey key) {
        if (key.isValid() && (selectorThread != null)) {
            selectorThread.cancelKey(key);
        }
    }

    public void registerClientKey(SelectionKey key) {
        if (key.isValid() && (selectorThread != null)) {
            selectorThread.registerKey(key);
        }
    }
}
