/*
* The contents of this file are subject to the terms
* of the Common Development and Distribution License
* (the License).  You may not use this file except in
* compliance with the License.
*
* You can obtain a copy of the license at
* https://glassfish.dev.java.net/public/CDDLv1.0.html or
* glassfish/bootstrap/legal/CDDLv1.0.txt.
* See the License for the specific language governing
* permissions and limitations under the License.
*
* When distributing Covered Code, include this CDDL
* Header Notice in each file and include the License file
* at glassfish/bootstrap/legal/CDDLv1.0.txt.
* If applicable, add the following below the CDDL Header,
* with the fields enclosed by brackets [] replaced by
* you own identifying information:
* "Portions Copyrighted [year] [name of copyright owner]"
*
* Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
* Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
*/
package org.jvnet.glassfish.comms.admin.config;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.Constants;
import com.ericsson.ssa.config.event.ConfigAddEvent;
import com.ericsson.ssa.config.event.ConfigRemoveEvent;
import com.ericsson.ssa.config.event.ConfigUpdateEvent;

import com.sun.enterprise.admin.event.AdminEventListener;
import com.sun.enterprise.admin.event.AdminEventListenerException;
import com.sun.enterprise.admin.event.AdminEventListenerRegistry;
import com.sun.enterprise.admin.event.http.HSRequestProcessingEvent;
import com.sun.enterprise.admin.event.http.HSRequestProcessingEventListener;
import com.sun.enterprise.admin.event.http.HSSslEvent;
import com.sun.enterprise.admin.event.http.HSSslEventListener;
import com.sun.enterprise.config.ConfigBean;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.server.ApplicationServer;

import org.jvnet.glassfish.comms.util.LogUtil;

import org.netbeans.modules.schema2beans.BaseBean;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipLinkEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipLinkEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipListenerEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipListenerEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipProtocolEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipProtocolEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipServiceEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipServiceEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipTimersEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipTimersEventListener;
import org.jvnet.glassfish.comms.util.LogUtil;
import org.netbeans.modules.schema2beans.BaseBean;
import java.util.logging.Level;


/**
 * Maps the ConfigBean model as defined by the domain.dtd with the sip container
 * internal model. Note that the mapping is historically not 121.
 * The intention is to grow to a 121 mapping. The intention is to keep
 * an internal model to allow proper dependencies between the modules.
 */
public class ConfigAdapter implements ConfigStartup {
    private static final String DEFAULT_MAPPER = "DEFAULT_MAPPER";
    private Map<String, ConfigBean2InternalMapper> mappers;
    private LogUtil log = new LogUtil(getClass());
    private SimpleConfig simpleConfig;

    public static final ConfigAdapter INSTANCE = new ConfigAdapter();
    private boolean initialized;

    
    /**
     * Creates a new instance of ConfigAdapter
     */
    private ConfigAdapter() {
    }

    public static final ConfigAdapter instance() {
        return INSTANCE;
    }

//Lifecycle support
    public void init() {
        if (!initialized) {
            mappers = discoverConfigBean2InternalMappers();
            registerEventListeners(mappers);
            simpleConfig = initInternalConfig();
            initialized=true;
        }
    }
    
    public void startup() { 
        init();
        configureSipContainer(mappers);
    }

    public void configureSpecificContext(ConfigBean bean, String ctx) {
        //This is not ideal yet, need to now how ctx is mapped to internal, not wanted ofcourse!
        //eventMask = ctx + one child
        simpleConfig.captureEvents("/" + ctx + "(([^/]+)([/]?))?",
            new String[] { ConfigAddEvent.TYPE });
        configureSipContainer(bean, ctx, ConfigFactory.getConfig(), mappers);
        simpleConfig.processEvents();

        if (log.logInfo()) {
            log.infoMsg("Updated internal Sip Container config: " +
                ConfigFactory.getConfig());
        }
    }

    void reconfigureSpecificContext(ConfigBean bean, String ctx) {
        //This is not ideal yet, need to now how ctx is mapped to internal, not wanted ofcourse!
        //eventMask = ctx + any decending child
        simpleConfig.captureEvents("/" + ctx + "(([^/]+)([/]?))*",
            new String[] { ConfigAddEvent.TYPE, ConfigUpdateEvent.TYPE });
        reconfigureSipContainer(bean, ctx, ConfigFactory.getConfig(), mappers);
        simpleConfig.processEvents();

        if (log.logInfo()) {
            log.infoMsg("Updated internal Sip Container config: " +
                ConfigFactory.getConfig());
        }
    }

    void unconfigureSpecificContext(ConfigBean bean, String ctx) {
        //This is not ideal yet, need to now how ctx is mapped to internal, not wanted ofcourse!
        //eventMask = ctx + one child
        simpleConfig.captureEvents("/" + ctx + "(([^/]+)([/]?))?",
            new String[] { ConfigRemoveEvent.TYPE });
        unconfigureSipContainer(bean, ctx, ConfigFactory.getConfig(), mappers);
        simpleConfig.processEvents();

        if (log.logInfo()) {
            log.infoMsg("Updated internal Sip Container config: " +
                ConfigFactory.getConfig());
        }
    }

    /**
     * Discovers the mappers that are active in the current setup.
     * Mappers can be added, i.e. to deal with changes in the domain.dtd
     */
    protected Map<String, ConfigBean2InternalMapper> discoverConfigBean2InternalMappers() {
        Map<String, ConfigBean2InternalMapper> mappers = new HashMap<String, ConfigBean2InternalMapper>();

        //No proper dynamic extention mechanism yet.
        //Hardcode here your mappers.
        //Key is the element.name of the ConfigBeans the mapper is to be applicable for.
        mappers.put("SipContainer", new SipContainerMapper()); 
        mappers.put("ElementProperty", new ElementPropertyMapper());
        mappers.put("SipService", new ConfigBean2InternalMapperImpl(SipServiceEvent.eventType, SipServiceEventListener.class, false));
        //mappers.put("RequestProcessing", new ConfigBean2InternalMapperImpl(HSRequestProcessingEvent.eventType, HSRequestProcessingEventListener.class, false));        
        mappers.put("SipListener", new ConfigBean2InternalMapperImpl(SipListenerEvent.eventType, SipListenerEventListener.class, true));        
        //mappers.put("Ssl", new ConfigBean2InternalMapperImpl(HSSslEvent.eventType, HSSslEventListener.class, true));        
        mappers.put("SipProtocol", new ConfigBean2InternalMapperImpl(SipProtocolEvent.eventType, SipProtocolEventListener.class, false));       
        mappers.put("SipLink", new ConfigBean2InternalMapperImpl(SipLinkEvent.eventType, SipLinkEventListener.class, false));
        mappers.put("SipTimers", new ConfigBean2InternalMapperImpl(SipTimersEvent.eventType, SipTimersEventListener.class, false));
        mappers.put(DEFAULT_MAPPER, new DefaultMapper());        
        
        return mappers;
    }

    protected void registerEventListeners(
        Map<String, ConfigBean2InternalMapper> mappers) {
        for (ConfigBean2InternalMapper mapper : mappers.values()) {
            if (mapper.requiresEventListerenerRegistration()) {

                if (log.isLoggable(Level.FINEST)) {
                    log.logMsg(Level.FINEST, "Registrering EventListener "+mapper.getEventListenerClass()+" for "+mapper.getEventType());
                }
                
                try {
                AdminEventListenerRegistry.registerEventListener(mapper.getEventType(),
                    mapper.getEventListenerImpl(this),
                    mapper.getEventListenerClass());
                } catch (IllegalArgumentException e) {
		// It is still doubtful whether this should be done.
		// Disregardly the behaviour of AdminEventListenerRegistry 
		// overriding the previous listener with the newly added or
		// allowing multple, it might not be ideal.
                   if (log.isLoggable(Level.FINEST)) { 
			   log.logMsg(Level.FINEST, "Type already registered Already, just adding listener.");
		}
                    AdminEventListenerRegistry.addEventListener(mapper.getEventType(),
                        mapper.getEventListenerImpl(this));
                }
            }
        }
    }
    
    protected void configureSipContainer(
        Map<String, ConfigBean2InternalMapper> mappers) {
        //Approach is to recursivly travers the complete tree and map it to the internal config starting from the
        // for the sip container root nodes.

        try {
            configureSipContainer(SipBeansFactory.getSipContainerBean(), "",
                simpleConfig, mappers);
        } catch (ConfigException ex) {
            if (log.logWarning()) {
                log.warning(ex, "config_adaptation_failure_continue",
                    "SipContainer");
            }
        }

        try {
            for (Iterator i = SipBeansFactory.getSipServices().iterator();
                    i.hasNext();) {
                configureSipContainer((ConfigBean) i.next(), "", simpleConfig,
                    mappers);
            }
        } catch (ConfigException ex) {
            if (log.logWarning()) {
                log.warning(ex, "config_adaptation_failure_continue",
                    "SipService");
            }
        }
        if (log.logInfo()) {
            log.infoMsg("Using internal Sip Container config: " + simpleConfig);
        }
    }

    protected void configureSipContainer(ConfigBean bean, String accumCtx,
        Config config, Map<String, ConfigBean2InternalMapper> mappers) {
        if (log.logInfo()) {
            log.infoMsg("Configuring bean " + bean.name() + " to context " +
                accumCtx);
            dumpBean(bean);
        }

        if (log.isLoggable(Level.FINEST)) {
            log.logMsg(Level.FINEST,
                "Configuring bean " + bean.name() + " to context " + accumCtx);
            dumpBean(bean);
        }

        String ctx = accumCtx + bean.name();

        ConfigBean2InternalMapper mapper = mappers.get(bean.name());

        if (mapper == null) {
            mapper = mappers.get(DEFAULT_MAPPER);
        }

        String[] ctxHolder = new String[] { ctx };
        BaseBean[] childeren2Traverse = mapper.map2Internal(bean, ctxHolder,
                config);

        if (childeren2Traverse != null) {
            for (BaseBean baseBean : childeren2Traverse) {
                configureSipContainer((ConfigBean) baseBean,
                    ctxHolder[0] + "/", config, mappers);
            }
        }
    }

    protected void reconfigureSipContainer(ConfigBean bean, String accumCtx,
        Config config, Map<String, ConfigBean2InternalMapper> mappers) {
        if (log.logInfo()) {
            log.infoMsg("Reconfiguring bean " + bean.name() + " to context " +
                accumCtx);
            dumpBean(bean);
        }

        if (log.isLoggable(Level.FINEST)) {
            log.logMsg(Level.FINEST,
                "Reconfiguring bean " + bean.name() + " to context " +
                accumCtx);
            dumpBean(bean);
        }

        String ctx = accumCtx + bean.name();

        ConfigBean2InternalMapper mapper = mappers.get(bean.name());

        if (mapper == null) {
            mapper = mappers.get(DEFAULT_MAPPER);
        }

        String[] ctxHolder = new String[] { ctx };
        BaseBean[] childeren2Traverse = mapper.remap2Internal(bean, ctxHolder,
                config);

        if (childeren2Traverse != null) {
            for (BaseBean baseBean : childeren2Traverse) {
                reconfigureSipContainer((ConfigBean) baseBean,
                    ctxHolder[0] + "/", config, mappers);
            }
        }
    }

    protected void unconfigureSipContainer(ConfigBean bean, String accumCtx,
        Config config, Map<String, ConfigBean2InternalMapper> mappers) {
        if (log.logInfo()) {
            log.infoMsg("Unconfiguring bean " + bean.name() + " from context " +
                accumCtx);
            dumpBean(bean);
        }

        if (log.isLoggable(Level.FINEST)) {
            log.logMsg(Level.FINEST,
                "Unconfiguring bean " + bean.name() + " from context " +
                accumCtx);
            dumpBean(bean);
        }

        String ctx = accumCtx + bean.name();

        ConfigBean2InternalMapper mapper = mappers.get(bean.name());

        if (mapper == null) {
            mapper = mappers.get(DEFAULT_MAPPER);
        }

        String[] ctxHolder = new String[] { ctx };
        BaseBean[] childeren2Traverse = mapper.unmap2Internal(bean, ctxHolder,
                config);

        if (childeren2Traverse != null) {
            for (BaseBean baseBean : childeren2Traverse) {
                unconfigureSipContainer((ConfigBean) baseBean,
                    ctxHolder[0] + "/", config, mappers);
            }
        }
    }

    private SimpleConfig initInternalConfig() {
        String asInstanceName = ApplicationServer.getServerContext()
                                                 .getInstanceName();

        if (log.isLoggable(Level.FINEST)) {
            log.logMsg(Level.FINEST,
                "AS Instance Name used for default context: " + asInstanceName);
        }

        SimpleConfig config = new SimpleConfig(asInstanceName);
        ConfigFactory.instance().registerConfig(config);

        return config;
    }

    private void dumpBean(ConfigBean bean) {
        StringBuffer buf = new StringBuffer(bean.name());

        for (String attr : bean.getAttributeNames()) {
            buf.append("\n ");
            buf.append(attr);
            buf.append(" = ");
            buf.append(bean.getAttributeValue(attr));
        }

        log.logMsg(Level.FINEST, "Bean details: " + buf.toString());
    }

    class DefaultMapper extends BaseMapper {
        public DefaultMapper() {
            super();
        }

        public DefaultMapper(boolean isMultiple) {
            super(isMultiple);
        }

        public boolean requiresEventListerenerRegistration() {
            return false;
        }
    }

    class SipContainerMapper extends BaseMapper {
        protected void mapAttribute2Internal(String attribute, String value,
            ConfigBean bean, String ctx, Config config) {
            if ("ExternalAddress".equals(attribute)) {
                if (log.isLoggable(Level.FINEST)) {
                    log.logMsg(Level.FINEST,
                        " SipContainer mapping rule active for [node = " + ctx +
                        ", key = " + attribute + ", value = " + value + "]");
                }

                String mappedAttribute = Constants.SIP_PUBLIC_HOST;

                if (value != null) {
                    if (log.isLoggable(Level.FINEST)) {
                        log.logMsg(Level.FINEST,
                            "Mapped to [node = <server specific node>, key = " +
                            mappedAttribute + ", value = " + value + "]");
                    }

                    config.set(mappedAttribute, value);
                } else {
                    if (log.isLoggable(Level.FINEST)) {
                        log.logMsg(Level.FINEST,
                            "Not mapped, expected optional.");
                    }
                }
                
                //Temp 
                
                //Also do it the right way
                super.mapAttribute2Internal(attribute, value, bean, ctx, config);
            } else {
                super.mapAttribute2Internal(attribute, value, bean, ctx, config);
            }
        }        
    }
    
    class ElementPropertyMapper extends BaseMapper {
        protected void mapAttributes2Internal(ConfigBean bean, String ctx, Config config) {
            String name = bean.getAttributeValue("name");            
            String value = bean.getAttributeValue("value");
            
            if (name!=null && value!=null) {
                config.set(ctx, name, value);
            } else {
                //TODO log warning, not an element property!
            }
        }        
    }    
}
