/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package org.jvnet.glassfish.comms.startup;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.ConfigImpl;
import com.ericsson.ssa.config.LayerHandler;
import com.ericsson.ssa.config.RuntimeConfig;
import com.ericsson.ssa.container.SipBindingResolver;
import com.ericsson.ssa.container.mbeans.Constants;
import com.ericsson.ssa.container.mbeans.MBeanUtil;
import com.ericsson.ssa.container.sim.ApplicationDispatcher;
import com.ericsson.ssa.dd.ModuleDeployException;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.SipFactoryImpl;

import com.sun.enterprise.web.VirtualServer;

import org.apache.catalina.ContainerEvent;
import org.apache.catalina.Lifecycle;
import org.apache.catalina.LifecycleEvent;
import org.apache.catalina.LifecycleListener;
import org.apache.catalina.Valve;

import org.jvnet.glassfish.comms.admin.config.ConfigAdapter;
import org.jvnet.glassfish.comms.admin.monitor.MonitoringAdapter;
import org.jvnet.glassfish.comms.admin.monitor.MonitoringStartup;
import org.jvnet.glassfish.comms.startup.lifecycle.SipContainerLifecycle;
import org.jvnet.glassfish.comms.util.LogUtil;

import org.xml.sax.SAXException;

import java.io.IOException;
import java.io.InputStream;

import java.lang.reflect.Method;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;


public class SipServiceListener implements LifecycleListener {
    private LogUtil log = new LogUtil(getClass());

    /**
     * Configuration class
     */
    private String configClassName = "com.ericsson.ssa.config.ConfigManager";
    LayerHandler _layerHandler = LayerHandler.getInstance();
    ExecutorService es = Executors.newFixedThreadPool(1);

    public SipServiceListener() {
        if (log.logInfo()) {
            log.info("sip.integration.sip_service_listener_construction",
                getClass().getName());
        }
    }

    public void lifecycleEvent(LifecycleEvent event) {
        try {
            // if (event.getSource() instanceof StandardService &&
            // "start".equals(event.getType()))
            if (Lifecycle.BEFORE_START_EVENT.equals(event.getType())) {
                initializeServer(event);

                if (log.isLoggable(Level.FINE)) {
                    log.logMsg(Level.FINE, "Received Before Start Event");
                }

                // TODO handle Performance Listener
                // Class<?> perfListenerClass = Class
                // .forName("com.ericsson.ssa.container.startup.PerformanceMBeanListener");
                // perfListener = (LifecycleListener)
                // perfListenerClass.newInstance();
                // perfListener.lifecycleEvent(event);
            }
            else if (Lifecycle.AFTER_START_EVENT.equals(event.getType())) {
                if (log.isLoggable(Level.FINE)) {
                    log.logMsg(Level.FINE, "Received After Start Event");
                }

                startSipStack();

                // Fault Management. Try/catch moved here in order to catch any
                // exception in initializeServer(),startSipStack()

                // FmEventSender.clearAlarms();
            }
            // else if (event.getSource() instanceof StandardService &&
            // "stop".equals(event.getType()))
            else if (Lifecycle.STOP_EVENT.equals(event.getType())) {
                if (log.isLoggable(Level.FINE)) {
                    log.logMsg(Level.FINE, "Received Stop Event");
                }

                stopSipStack();

                // perfListener.lifecycleEvent(event);
            }
        } catch (Throwable e) {
            if (log.logSevere()) {
                log.severe(e, "sip.integration.lifecycle_failure_continue",
                    event.getType().toString());
            }
        }
    }

    private void initLayerHandler()
        throws IOException, ModuleDeployException, SAXException {
        if (log.isLoggable(Level.FINE)) {
            log.logMsg(Level.FINE, "Initializing the layer handler");
        }

        // Create SIP Stack
        InputStream is = SipContainerLifecycle.class.getResourceAsStream(
                "dispatcher.xml");
        _layerHandler.parse(is);
    }

    public void containerEvent(ContainerEvent event) {
        //Can be good to know perhaps!?
        if (log.isLoggable(Level.FINE)) {
            log.logMsg(Level.FINE, "Container event : " + event);
        }
    }

    private void initializeServer(LifecycleEvent event)
        throws Exception {
        MBeanUtil mbeanUtil = MBeanUtil.getInstance();
        mbeanUtil.createServer();

        //Temporary switch for using old or new config
        //default is old.
        if (Boolean.getBoolean("sip.module.use_new_config")) {
            if (log.logInfo()) {
                log.infoMsg("Using new config!");
            }

            ConfigAdapter.instance().startup();
        } else {
            if (log.logInfo()) {
                log.infoMsg("Using old config!");
            }

            initializeConfig();
        }

        Config config = ConfigFactory.getConfig();

        retrieveSipModuleVersion();

        if (log.logInfo()) {
            log.info("sip.integration.sip_module_version",
                System.getProperty("sip.module.version"));
        }

        String sipBindAddress = null;

        if (Boolean.getBoolean("sip.module.use_new_config")) {
            SipBindingResolver.instance().init();
        } else {
            sipBindAddress = determineBindings(config);
        }

        VirtualServer host = (VirtualServer) event.getSource();

        // Reconfigure the host with our own Context config class
        host.setConfigClass(com.ericsson.ssa.config.Constants.CONFIG_CLASS);

        mbeanUtil.loadDescriptors(this.getClass(), "mbeans-desciptor.xml");

        initLayerHandler();

        for (Layer layer : _layerHandler.getLayers()) {
            if (layer instanceof Valve) {
                if (layer != ApplicationDispatcher.getInstance()) {
                    host.addValve((Valve) layer); // TODO: check that it is correct to add the valves to 'host' (qbinjoe)
                }
            }
        }

        // 4.Create new ApplicationDispatcher to replace the Standard Host Valve
        ApplicationDispatcher appDispatcher = ApplicationDispatcher.getInstance();
        host.addValve(appDispatcher);
        host.addContainerListener(appDispatcher);

        // mbeanUtil.loadDescriptors(sipStack.getClass(),
        // "mbeans-desciptor.xml");
        mbeanUtil.loadDescriptors(mbeanUtil.getClass(), "mbeans-desciptor.xml");

        // TODO this should be done by LayerHandler
        // mbeanUtil.registerMBean( sipStack , Constants.NETWORK_MANAGER );
        List<Object> mbeanCollection = new ArrayList<Object>(LayerHandler.getInstance()
                                                                         .getLayers());
        mbeanCollection.add(SipFactoryImpl.getInstance());
        mbeanUtil.registerMBeanCollection(mbeanCollection,
            Constants.NETWORK_MANAGER);
        mbeanUtil.registerMBean(mbeanUtil, Constants.JMX_CONFIGURATION);

        // Load Runtime Parameters
        RuntimeConfig rc = loadRuntimeParameters();

        /// TODO also add the check if the LB HOST AND PORT ARE NOT CONFIGURED IN DOMAIN.XML
        if (!Boolean.getBoolean("sip.module.use_new_config") && (rc != null) &&
                (sipBindAddress == null)) {
            /// Add the best guess interface as the defaul LB IF
            config.set(com.ericsson.ssa.config.Constants.SIP_PUBLIC_HOST,
                rc.getLanAddress());
        }

        // Start Remote MBean Adapter
        //Fix for Issue 58
        //Don't publish the Mbeans using the RemoteAdapter
        //Note that the MBeans are candidates to be removed when config cleanup is performed.
        //mbeanUtil.startRemoteAdapter();

        //TODO Why was it initialized manually ekrigro!?
        //NetworkManager.getInstance().initialize();
        if (log.logInfo()) {
            if (Boolean.getBoolean("sip.module.use_new_config")) {
                log.infoMsg("Sip Module Bindings: " +
                    SipBindingResolver.instance().reportBindings());
            } else {
                log.info("sip.integration.used_hosts_and_ports",
                    config.get(com.ericsson.ssa.config.Constants.HOST),
                    config.get(com.ericsson.ssa.config.Constants.SIP_PORT),
                    config.get(
                        com.ericsson.ssa.config.Constants.SIP_PUBLIC_HOST),
                    config.get(
                        com.ericsson.ssa.config.Constants.SIP_PUBLIC_PORT));
            }
        }

        if (log.logInfo()) {
            log.infoMsg("configuring monitoring");
        }

        MonitoringStartup monitoringStartup = new MonitoringAdapter();
        monitoringStartup.startup();
    }

    private void initializeConfig() {
        // HostNumber is used as a unique indentifer for each host, usally set by
        // the startscript.
        if (System.getProperty("HostNumber", null) == null) {
            System.setProperty("HostNumber", "1");
        }

        ConfigFactory.instance().registerConfig(new ConfigImpl());
    }

    private String determineBindings(final Config config) {
        config.set(com.ericsson.ssa.config.Constants.SIP_PORT, "5060");
        config.set(com.ericsson.ssa.config.Constants.SIP_PUBLIC_PORT, "5060");

        //Check whether it is already configured.
        String publicSipBindAddress = null;
        String localSipBindAddress = null;

        try {
            publicSipBindAddress = config.get(com.ericsson.ssa.config.Constants.SIP_PUBLIC_HOST);
        } catch (RuntimeException e) { //IGNORE            
        }

        try {
            localSipBindAddress = config.get(com.ericsson.ssa.config.Constants.HOST);
        } catch (RuntimeException e) { //IGNORE          
        }

        if (publicSipBindAddress == null) {
            publicSipBindAddress = System.getProperty("SipBindAddress", null);

            if (publicSipBindAddress == null) {
                if (log.logInfo()) {
                    log.infoMsg(
                        "Using system property 'SipBindAddress' for public SipBindingAddress: " +
                        publicSipBindAddress);
                }
            }
        } else {
            if (log.logInfo()) {
                log.infoMsg(
                    "Using previously stored public SipBindingAddress: " +
                    publicSipBindAddress);
            }
        }

        if (localSipBindAddress == null) {
            localSipBindAddress = System.getProperty("SipBindAddress", null);

            if (localSipBindAddress == null) {
                if (log.logInfo()) {
                    log.infoMsg(
                        "Using system property 'SipBindAddress' for local SipBindingAddress: " +
                        localSipBindAddress);
                }
            }
        } else {
            if (log.logInfo()) {
                log.infoMsg("Using previously stored local SipBindingAddress: " +
                    localSipBindAddress);
            }
        }

        // SipBindAddress
        // if set, then use this address to bind te SIP_PORT to
        // else use whatever address getLocalHost returns
        if (System.getProperty("SipBindAddress", null) != null) {
            config.set(com.ericsson.ssa.config.Constants.HOST,
                localSipBindAddress);
            config.set(com.ericsson.ssa.config.Constants.SIP_PUBLIC_HOST,
                publicSipBindAddress);
        } else {
            if (log.logWarning()) {
                log.warning("sip.integration.using_any_host");
            }

            config.set(com.ericsson.ssa.config.Constants.HOST,
                com.ericsson.ssa.config.Constants.BIND_TO_ANY);
        }

        return System.getProperty("SipBindAddress", null);
    }

    private void retrieveSipModuleVersion() {
        // Sipmodule version is retreived from sip-module.version
        try {
            Properties p = new Properties();
            p.load(this.getClass().getClassLoader()
                       .getResourceAsStream("sip-module.version"));
            System.setProperty("sip.module.version",
                p.getProperty("SipModuleVersion"));
        } catch (IOException e) {
            if (log.logWarning()) {
                log.warning(e,
                    "sip.integration.sip_module_version_lookup_failure_continue",
                    "default");
            }

            System.setProperty("sip.module.version", "Glassfish_SIP_1.0.0");
        } catch (NullPointerException e) {
            if (log.logWarning()) {
                log.warning(e,
                    "sip.integration.sip_module_version_lookup_failure_continue",
                    "default");
            }

            System.setProperty("sip.module.version", "Glassfish_SIP_1.0.0");
        }
    }

    /**
     * Starts the SIP stack in a new Thread. This method is exposed for JMX
     * management
     */
    public void startSipStack() {
        Runnable firstRunnableLayer = null;
        Iterator<Layer> i = _layerHandler.getLayers().iterator();

        while (i.hasNext()) {
            Layer l = i.next();

            if ((firstRunnableLayer == null) && l instanceof Runnable) {
                firstRunnableLayer = (Runnable) l;
            }

            try {
                Method m = l.getClass().getMethod("start", new Class[0]);
                m.invoke(l, new Object[0]);
            } catch (NoSuchMethodException ignore) {
                // No start method defined for that layer
            } catch (Exception e) {
                if (log.logSevere()) {
                    log.severe(e,
                        "sip.integration.sip_stack_initialization_failure_abort",
                        "Failed to start layer: " + l);
                }

                throw new RuntimeException(e);
            }
        }

        if (firstRunnableLayer != null) {
            es.execute(firstRunnableLayer);
        } else {
            if (log.logSevere()) {
                log.severe("sip.integration.sip_stack_initialization_failure_abort",
                    "No startable layer found!");
            }

            stopSipStack();
        }
    }

    /**
     * Stops the SIP stack.
     */
    public void stopSipStack() {
        try {
            List<Object> objectsToStop = new ArrayList<Object>();

            for (Layer l : _layerHandler.getLayers()) {
                objectsToStop.add(l);
            }

            for (Object o : objectsToStop) {
                try {
                    Method m = o.getClass().getMethod("stop", new Class[0]);
                    m.invoke(o, new Object[0]);
                } catch (NoSuchMethodException ignore) {
                    // No start method defined for that layer
                } catch (Exception e) {
                    if (log.logWarning()) {
                        log.warning(e,
                            "sip.integration.sip_stack_teardown_failure_continue",
                            "Failed to stop layer: " + o);
                    }
                }
            }
        } catch (Exception e) {
            if (log.logWarning()) {
                log.warning(e,
                    "sip.integration.sip_stack_teardown_failure_continue",
                    "Reason unknown.");
            }
        } finally {
            if (es != null) {
                es.shutdown();
            }
        }
    }

    public void setConfigClassName(String configClassName) {
        this.configClassName = configClassName;
    }

    private RuntimeConfig loadRuntimeParameters() {
        try {
            Class clazz = Class.forName(configClassName);
            RuntimeConfig config = (RuntimeConfig) clazz.newInstance();
            config.loadRuntimeConfiguration();

            return config;
        } catch (Exception e) {
            //         logger.log(Level.SEVERE, "Cannot create a Runtime Config", e);
        }

        return null;
    }
}
