/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.config;

import com.ericsson.eas.telurl.TelUrlResolver;

import com.ericsson.ssa.config.ConvergedContext;
import com.ericsson.ssa.config.ConvergedContextRuleSet;
import com.ericsson.ssa.config.SARDirContext;
import com.ericsson.ssa.config.SipFactoryFacade;
import com.ericsson.ssa.container.auth.DescriptorProcessor;
import com.ericsson.ssa.container.sim.ApplicationDispatcher;
import com.ericsson.ssa.container.sim.ServletDispatcher;
import com.ericsson.ssa.dd.ConvergedDescriptor;
import com.ericsson.ssa.dd.ConvergedDescriptor;
import com.ericsson.ssa.dd.EjbLocalReference;
import com.ericsson.ssa.dd.EjbReference;
import com.ericsson.ssa.dd.EnvironmentEntry;
import com.ericsson.ssa.dd.Parameter;
import com.ericsson.ssa.dd.ResourceEnvironmentReference;
import com.ericsson.ssa.dd.ResourceReference;
import com.ericsson.ssa.dd.SecurityRole;
import com.ericsson.ssa.dd.SipApplication;
import com.ericsson.ssa.dd.SipApplicationListeners;
import com.ericsson.ssa.dd.SipParser;
import com.ericsson.ssa.sip.SipFactoryImpl;
import com.ericsson.ssa.sip.SipSessionsUtilImpl;
import com.ericsson.ssa.sip.dns.DnsResolver;
import com.ericsson.ssa.sip.timer.TimerServiceImpl;

import com.sun.enterprise.NamingManager;
import com.sun.enterprise.Switch;
import com.sun.enterprise.naming.NamingManagerImpl;
import com.sun.enterprise.web.WebModuleContextConfig;

import com.sun.org.apache.commons.digester.Digester;
import com.sun.org.apache.commons.digester.RuleSet;

import org.apache.catalina.Container;
import org.apache.catalina.Host;
import org.apache.catalina.LifecycleEvent;
import org.apache.catalina.core.StandardContext;
import org.apache.catalina.core.StandardEngine;
import org.apache.catalina.core.StandardHost;
import org.apache.catalina.deploy.ContextEjb;
import org.apache.catalina.deploy.ContextEnvironment;
import org.apache.catalina.deploy.ContextLocalEjb;
import org.apache.catalina.deploy.ContextResource;
import org.apache.catalina.startup.ExpandWar;
import org.apache.catalina.startup.NamingRuleSet;

import org.apache.naming.resources.FileDirContext;
import org.apache.naming.resources.WARDirContext;

import org.jvnet.glassfish.comms.security.util.PolicyBuilder;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import java.net.URL;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.*;

import javax.naming.Context;
import javax.naming.NamingException;
import javax.naming.Reference;
import javax.naming.StringRefAddr;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.sip.SipFactory;
import javax.servlet.sip.SipServlet;


public final class ConvergedContextConfig extends WebModuleContextConfig {
    private static Logger logger = Logger.getLogger("SipContainer");
    private static final String APPLICATION_SIP_XML = "/WEB-INF/sip.xml";
    protected NamingManager namingManager;
    Switch theSwitch;
    protected ConvergedContextImpl convergedContext;

    public ConvergedContextConfig() {
        theSwitch = Switch.getSwitch();
        namingManager = theSwitch.getNamingManager();
    }

    /**
     * Create (if necessary) and return a Digester configured to process the
     * context configuration descriptor for an application.
     */
    protected Digester createContextDigester() {
        Digester digester = new Digester();
        digester.setValidating(false);

        RuleSet contextRuleSet = new ConvergedContextRuleSet("", false);
        digester.addRuleSet(contextRuleSet);

        RuleSet namingRuleSet = new NamingRuleSet("Context/");
        digester.addRuleSet(namingRuleSet);

        return digester;
    }

    /**
     * Process events for an associated Context.
     *
     * @param event The lifecycle event that has occurred
     */
    public void lifecycleEvent(LifecycleEvent event) {
        // Identify the context we are associated with
        try {
            convergedContext = (ConvergedContextImpl) event.getLifecycle();
        } catch (ClassCastException e) {
            logger.log(Level.FINE,
                "Lifecycle event data object {0} is not a ConvergedContext", e);

            return;
        }

        super.lifecycleEvent(event);
    }

    /**
     * Process a "before start" event for this Context.
     */
    protected void beforeStart() {
        //super.beforeStart();
        setContextResources();
    }

    /**
     * Process a "start" event for this Context.
     */
    protected synchronized void start() {
        super.start();
        ok = true;

        // Set properties based on DefaultContext
        Container container = context.getParent();

        if (!context.getOverride()) {
            if (container instanceof Host) {
                container = container.getParent();
            }
        }

        ServletContext servletContext = context.getServletContext();
        InputStream sipXml = servletContext.getResourceAsStream(APPLICATION_SIP_XML);

        if (sipXml != null) {
            // Do the SIP servlet specific part.
            try {
                /*
                 * The sequence to start a context 1. Parse the deployment descriptor into a
                 * SipApplication 2. Copy Context Parameters to the Standard Context of
                 * Tomcat 3. bind resources to the context 4. Initialize the servlet
                 * dispatcher
                 */
                getSipApplication();
                initServletContextAttributes();
                copyContextProperties();

                PolicyBuilder builder = new PolicyBuilder();
                String pContextId = convergedContext.getSipApplication()
                                                    .getAppName();
                if (pContextId == null) {
		    pContextId =  context.getName();
                     if(pContextId.length() > 0 && pContextId.startsWith("/")){
                 pContextId = pContextId.substring(1);
            }
                }

                pContextId = pContextId + "/_" + pContextId;
                builder.createRoleMapper(super.getDescriptor(), pContextId);
                new DescriptorProcessor().process(context.getName(),convergedContext.getSipApplication());
                initializeServletDispatcher();
            } catch (Exception e) {
                ok = false;
            }
        }

        // Make our application available if no problems were encountered
        if (ok) {
            context.setConfigured(true);
        } else {
            logger.log(Level.WARNING, "contextConfig.unavailable");
            context.setAvailable(false);
        }
    }

    /**
     * Upon stop of context, all allocated resources and references are reset
     */
    protected synchronized void stop() {
        super.stop();

        // Notify ServletContextListener.contextDestroyed here.
        if (convergedContext.getSipApplicationListeners() != null) {
            ServletContextEvent servletContextEvent = new ServletContextEvent(context.getServletContext());
            Iterator<ServletContextListener> contextListenerIterator = convergedContext.getSipApplicationListeners()
                                                                                       .getServletContextListener()
                                                                                       .iterator();

            while (contextListenerIterator.hasNext()) {
                ServletContextListener listener = contextListenerIterator.next();
                listener.contextDestroyed(servletContextEvent);
            }
        }

        // Calling the un-initialize method to call the
        // destroy method on the servlet(s).
        if (convergedContext.getDispatcher() != null) {
            convergedContext.getDispatcher().unInitialize();
        }

        if (convergedContext.getSipApplicationListeners() != null) {
            convergedContext.getSipApplicationListeners().clean();
        }

        convergedContext.setSipApplication(null);
        convergedContext.setDispatcher(null);
        convergedContext.setSipFactory(null);

        // Unpublish the SipFactory from the JNDI namespace
        try {
            namingManager.unpublishObject("sip" + context.getName());

            //remove the SipFactory from the Map
            SipFactory sf = SipFactoryMap.getInstance()
                                         .removeSipFactory(context.getName());

            if (sf != null) {
                logger.log(Level.INFO,
                    "SipFactory object corresponding to" + context.getName() +
                    "removed");
            }
        } catch (NamingException ne) {
            ne.printStackTrace();
        }
    }

    /**
     * Process a "destroy" event for this Context.
     */
    protected synchronized void destroy() {
        String workDir = ((StandardContext) context).getWorkPath();

        if (workDir != null) {
            ExpandWar.delete(new File(workDir));
        }
    }

    /**
     * Parses the SIP deployment descriptor.
     *
     * @param sipXml The SIP servlet XML descriptor input stream.
     * @throws Exception
     */
    private void parseSipDD(InputStream sipXml) throws Exception {
        try {
            convergedContext.setSipApplication(SipParser.parse(sipXml));
        } catch (Exception e) {
            throw e; // Just rethrow.
        } finally {
            if (sipXml != null) {
                sipXml.close();
            }
        }
    }

    /**
     * Upon initialization of a context the SIP resources
     * are bound to the context
     */
    private void initServletContextAttributes() {
        // Add the sip factory and the registrar to the attributes
        ServletContext servletContext = context.getServletContext();

        // SSA 1.0, section 4.2, here we can add new extensions.
        // The only extension we support right now is 100rel.
        List<String> supportedExt = new ArrayList<String>();
        supportedExt.add("100rel");
        servletContext.setAttribute("javax.servlet.sip.supported", supportedExt);

        SipApplication sipApplication = convergedContext.getSipApplication();
        SipFactoryImpl.getInstance()
                      .setServiceHandler(ApplicationDispatcher.getInstance());

        SipApplicationListeners sipApplicationListeners = new SipApplicationListeners(sipApplication.getListeners(),
                context.getLoader().getClassLoader());
        convergedContext.setSipApplicationListeners(sipApplicationListeners);

        /*
         * SipFactory
         */
        SipFactoryFacade sipFactory = new SipFactoryFacade(convergedContext,
                SipFactoryImpl.getInstance(), sipApplication);
        convergedContext.setSipFactory(sipFactory);
        servletContext.setAttribute(SipServlet.SIP_FACTORY, sipFactory);
 
        /*
         * SipSessionsUtil
         */
        SipSessionsUtilImpl ssu = new SipSessionsUtilImpl(convergedContext);
        convergedContext.setSipSessionsUtil(ssu);
        servletContext.setAttribute(SipServlet.SIP_SESSIONS_UTIL, ssu);

        try {
            String sipFactoryMappedName = "sip" + context.getName();

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "sipFactoryMappedName = " + sipFactoryMappedName);
            }

            //Create a Reference object to hold the SipFactory here
            Reference sipFactoryRef = new Reference("javax.servlet.sip.SipFactory",
                    new StringRefAddr("url", sipFactoryMappedName),
                    "org.jvnet.glassfish.comms.deployment.backend.SipFactoryObjectFactory",
                    null);

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "SipFactoryRef = " + sipFactoryRef);
            }

            namingManager.publishObject(sipFactoryMappedName, sipFactoryRef,
                false);

            // adding the SipFactory to the map
            (SipFactoryMap.getInstance()).addSipFactory(sipFactoryMappedName,
                (SipFactory) sipFactory);
        } catch (NamingException ne) {
            ne.printStackTrace();
        }

        // Bind the timer service
        servletContext.setAttribute(SipServlet.TIMER_SERVICE,
            TimerServiceImpl.getInstance());

        //Needed by applications which needs direct TelUrl lookup support
        servletContext.setAttribute(TelUrlResolver.CONTEXT_ATTRIBUTE_NAME,
            DnsResolver.getInstance());

        // Notify the ServletContextListener.contextInitialized here.
        ServletContextEvent servletContextEvent = new ServletContextEvent(servletContext);
        Iterator<ServletContextListener> contextListenerIterator = sipApplicationListeners.getServletContextListener()
                                                                                          .iterator();

        while (contextListenerIterator.hasNext()) {
            ServletContextListener listener = contextListenerIterator.next();
            listener.contextInitialized(servletContextEvent);
        }
    }

    /**
     * Attaches a <code>SipServletInvokerValve</code> to a <code>Context</code>
     * pipeline and registers the valve with the SIM. The initialization steps
     * of the valve are also executed.
     */
    private void initializeServletDispatcher() {
        ServletDispatcher dispatcher = new ServletDispatcher(convergedContext.getSipSessionManager());
        dispatcher.setSipApplicationModel(convergedContext.getSipApplication());
        context.getPipeline().addValve(dispatcher);

        // Initialize all the servlet wrappers upon initialization of the context.
        // Must do this after adding the valve to the pipeline, so that the parent
        // container of the valve and it's Loader are initialized.
        dispatcher.setSipFactory(convergedContext.getSipFactory());
        dispatcher.initialize();
        convergedContext.setDispatcher(dispatcher);
    }

    /**
     * Adjust docBase.
     */
    protected void fixDocBase() throws IOException {
        Host host = (Host) context.getParent();
        String appBase = host.getAppBase();
        boolean unpackWARs = true;

        if (host instanceof StandardHost) {
            unpackWARs = ((StandardHost) host).isUnpackWARs() &&
                ((StandardContext) context).getUnpackWAR();
        }

        File canonicalAppBase = new File(appBase);

        if (canonicalAppBase.isAbsolute()) {
            canonicalAppBase = canonicalAppBase.getCanonicalFile();
        } else {
            canonicalAppBase = new File(System.getProperty("catalina.base"),
                    appBase).getCanonicalFile();
        }

        String docBase = context.getDocBase();

        if (docBase == null) {
            // Trying to guess the docBase according to the path
            String path = context.getPath();

            if (path == null) {
                return;
            }

            if (path.equals("")) {
                docBase = "ROOT";
            } else {
                if (path.startsWith("/")) {
                    docBase = path.substring(1);
                } else {
                    docBase = path;
                }
            }
        }

        File file = new File(docBase);

        if (!file.isAbsolute()) {
            docBase = (new File(canonicalAppBase, docBase)).getPath();
        } else {
            docBase = file.getCanonicalPath();
        }

        file = new File(docBase);

        if ((docBase.toLowerCase().endsWith(".sar") ||
                docBase.toLowerCase().endsWith(".war")) && !file.isDirectory() &&
                unpackWARs) {
            URL war = new URL("jar:" + (new File(docBase)).toURL() + "!/");
            String contextPath = context.getPath();

            if (contextPath.equals("")) {
                contextPath = "ROOT";
            }

            docBase = ExpandWar.expand(host, war, contextPath);
            file = new File(docBase);
            docBase = file.getCanonicalPath();
        } else {
            File docDir = new File(docBase);

            if (!docDir.exists()) {
                String[] extensions = new String[] { ".sar", ".war" };

                for (String extension : extensions) {
                    File archiveFile = new File(docBase + extension);

                    if (archiveFile.exists()) {
                        if (unpackWARs) {
                            URL war = new URL("jar:" + archiveFile.toURL() +
                                    "!/");
                            docBase = ExpandWar.expand(host, war,
                                    context.getPath());
                            file = new File(docBase);
                            docBase = file.getCanonicalPath();
                        } else {
                            docBase = archiveFile.getCanonicalPath();
                        }

                        break;
                    }
                }
            }
        }

        if (docBase.startsWith(canonicalAppBase.getPath())) {
            docBase = docBase.substring(canonicalAppBase.getPath().length());
            docBase = docBase.replace(File.separatorChar, '/');

            if (docBase.startsWith("/")) {
                docBase = docBase.substring(1);
            }
        } else {
            docBase = docBase.replace(File.separatorChar, '/');
        }

        context.setDocBase(docBase);
    }

    // =================================================================
    // Copy parameters from the SipApplication Model to the Context
    //
    //
    private void copyContextProperties() {
        SipApplication sipApplication = convergedContext.getSipApplication();
        context.setDisplayName(sipApplication.getDisplayName());
        context.setDistributable(sipApplication.isDistributable());
        copyContextParameters();
        copyResourceEnvRefs();
        copyResourceRefs();
        copySecurityRoles();
        copyEnvironmentEntries();
        copyEjbReferences();
        copyEjbLocalReferences();
    }

    private void copyEjbLocalReferences() {
        Collection<EjbLocalReference> references = convergedContext.getSipApplication()
                                                                   .getEjbLocalReferences();

        if (references != null) {
            for (Iterator<EjbLocalReference> it = references.iterator();
                    it.hasNext();) {
                EjbLocalReference ref = it.next();
                ContextLocalEjb ejb = new ContextLocalEjb();
                ejb.setDescription(ref.getDescription());
                ejb.setHome(ref.getLocalHome());
                ejb.setLink(ref.getEjbLink());
                ejb.setLocal(ref.getLocal());
                ejb.setName(ref.getName());
                ejb.setType(ref.getType());

                // TODO Changed in Tomcat 5.5 context.addLocalEjb(ejb);
            }
        }
    }

    private void copyEjbReferences() {
        Collection<EjbReference> references = convergedContext.getSipApplication()
                                                              .getEjbReferences();

        if (references != null) {
            for (Iterator<EjbReference> it = references.iterator();
                    it.hasNext();) {
                EjbReference ref = it.next();
                ContextEjb ejb = new ContextEjb();
                ejb.setDescription(ref.getDescription());
                ejb.setHome(ref.getHome());
                ejb.setLink(ref.getEjbLink());
                ejb.setName(ref.getName());
                ejb.setRemote(ref.getRemote());
                ejb.setType(ref.getType());

                // TODO Changed in Tomcat 5.5 context.addEjb(ejb);
            }
        }
    }

    private void copyEnvironmentEntries() {
        Collection<EnvironmentEntry> entries = convergedContext.getSipApplication()
                                                               .getEnvironmentEntries();

        if (entries != null) {
            for (Iterator<EnvironmentEntry> it = entries.iterator();
                    it.hasNext();) {
                EnvironmentEntry entry = it.next();
                ContextEnvironment env = new ContextEnvironment();
                env.setDescription(entry.getDescription());
                env.setName(entry.getName());
                env.setType(entry.getType());
                env.setValue(entry.getValue());

                // TODO Changed in Tomcat 5.5 context.addEnvironment(env);
            }
        }
    }

    private void copySecurityRoles() {
        Map<String, SecurityRole> roles = convergedContext.getSipApplication()
                                                          .getSecurityRoles();

        if (roles != null) {
            for (Iterator<SecurityRole> it = roles.values().iterator();
                    it.hasNext();) {
                context.addSecurityRole(it.next().getName());
            }
        }
    }

    private void copyResourceEnvRefs() {
        Collection<ResourceEnvironmentReference> resources = convergedContext.getSipApplication()
                                                                             .getResourceEnvironmentReferences();

        if (resources != null) {
            for (Iterator<ResourceEnvironmentReference> it = resources.iterator();
                    it.hasNext();) {
                ResourceEnvironmentReference resource = it.next();

                // TODO Changed in Tomcat 5.5
                // context.addResourceEnvRef(resource.getName(),
                // resource.getType());
            }
        }
    }

    private void copyResourceRefs() {
        Collection<ResourceReference> resources = convergedContext.getSipApplication()
                                                                  .getResourceReferences();

        if (resources != null) {
            for (Iterator<ResourceReference> it = resources.iterator();
                    it.hasNext();) {
                ResourceReference resource = it.next();
                ContextResource contextResource = new ContextResource();
                contextResource.setAuth(resource.getAuthentication());
                contextResource.setDescription(resource.getDescription());
                contextResource.setName(resource.getName());
                contextResource.setScope(resource.getSharingScope());
                contextResource.setType(resource.getType());

                // TODO Changed in Tomcat 5.5 context.addResource(contextResource);
            }
        }
    }

    private void copyContextParameters() {
        Map<String, Parameter> parameters = convergedContext.getSipApplication()
                                                            .getContextParameters();

        if (parameters != null) {
            for (Iterator<Parameter> it = parameters.values().iterator();
                    it.hasNext();) {
                Parameter param = it.next();
                // Check if this is an internal session case
                context.addParameter(param.getName(), param.getValue());
            }
        }
    }

    /**
     * Get base path.
     */
    private String getBasePath(StandardContext context) {
        String docBase = null;
        Container container = context;

        while (container != null) {
            if (container instanceof Host) {
                break;
            }

            container = container.getParent();
        }

        File file = new File(context.getDocBase());

        if (!file.isAbsolute()) {
            if (container == null) {
                docBase = (new File(engineBase(context), context.getDocBase())).getPath();
            } else {
                // Use the "appBase" property of this container
                String appBase = ((Host) container).getAppBase();
                file = new File(appBase);

                if (!file.isAbsolute()) {
                    file = new File(engineBase(context), appBase);
                }

                docBase = (new File(file, context.getDocBase())).getPath();
            }
        } else {
            docBase = file.getPath();
        }

        return docBase;
    }

    /**
     * Return a File object representing the base directory for the
     * entire servlet container (i.e. the Engine container if present).
     *
     * @param context The StandardContext
     */
    protected File engineBase(StandardContext context) {
        String base = System.getProperty("catalina.base");

        if (base == null) {
            StandardEngine engine = (StandardEngine) context.getParent()
                                                            .getParent();

            base = engine.getBaseDir();
        }

        return new File(base);
    }

    /**
     * Set the resources DirContext object with which this
     * context is associated.
     */
    private void setContextResources() {
        try {
            if ((context.getDocBase() != null) &&
                    (context.getDocBase().endsWith(".war")) &&
                    (!(new File(getBasePath((StandardContext) context))).isDirectory())) {
                context.setResources(new WARDirContext());
            } else if ((context.getDocBase() != null) &&
                    (context.getDocBase().endsWith(".sar")) &&
                    (!(new File(getBasePath((StandardContext) context))).isDirectory())) {
                context.setResources(new SARDirContext());
            } else {
                context.setResources(new FileDirContext());
            }
        } catch (IllegalArgumentException e) {
            logger.log(Level.SEVERE,
                "Error initializing resources: " + e.getMessage());
        }
    }

    private void getSipApplication() throws Exception {
        SipApplication sipApplication = null;

        try {
            if ((super.getDescriptor() != null) &&
                    super.getDescriptor() instanceof ConvergedDescriptor) {
                sipApplication = ((ConvergedDescriptor) super.getDescriptor()).getSipApplication();
                convergedContext.setSipApplication(sipApplication);
            }
        } catch (Exception e) {
            throw e; // Just rethrow.
        }
    }
}
