/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.config;

import com.ericsson.ssa.container.sim.ServletDispatcher;
import com.ericsson.ssa.container.sim.SipServletFacade;
import com.ericsson.ssa.container.sim.SipServletWrapper;
import com.ericsson.ssa.dd.ConvergedDescriptor;
import com.ericsson.ssa.dd.SipApplication;
import com.ericsson.ssa.dd.SipApplicationListeners;
import com.ericsson.ssa.fm.FmEventSender;
import com.ericsson.ssa.sip.ConvergedHttpSessionFacade;
import com.ericsson.ssa.sip.PersistentSipSessionManagerBase;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipApplicationSessionStore;
import com.ericsson.ssa.sip.SipFactoryImpl;
import com.ericsson.ssa.sip.SipSessionManager;
import com.ericsson.ssa.sip.SipSessionManagerBase;
import com.ericsson.ssa.sip.SipSessionsUtilImpl;
import com.ericsson.ssa.sip.timer.TimerServiceImpl;

import com.sun.enterprise.config.serverbeans.SipContainerAvailability;
import com.sun.enterprise.deployment.WebBundleDescriptor;
import com.sun.enterprise.deployment.runtime.web.SessionConfig;
import com.sun.enterprise.deployment.runtime.web.SessionManager;
import com.sun.enterprise.web.PersistenceStrategyBuilder;
import com.sun.enterprise.web.PersistenceStrategyBuilderFactory;
import com.sun.enterprise.web.WebContainer;
import com.sun.enterprise.web.WebModule;
import com.sun.enterprise.web.WebModuleConfig;
import com.sun.enterprise.web.session.PersistenceType;

import org.apache.catalina.Container;
import org.apache.catalina.Globals;
import org.apache.catalina.Host;
import org.apache.catalina.Lifecycle;
import org.apache.catalina.LifecycleException;
import org.apache.catalina.Session;
import org.apache.catalina.Wrapper;
import org.apache.catalina.core.ApplicationContext;
import org.apache.catalina.core.StandardContext;
import org.apache.catalina.core.StandardWrapper;
import org.apache.catalina.session.StandardSession;
import org.apache.catalina.session.StandardSessionFacade;

import org.apache.coyote.tomcat5.CoyoteRequest;
import org.apache.coyote.tomcat5.CoyoteResponse;

import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.File;
import java.io.IOException;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.sip.SipServlet;


/**
 * Extension of the Tomcat StandardContext that implements the ConvergedContext
 * interface relevant for SIP applications. The ConvergedContextImpl class is
 * configured by the ConvegedContextConfig class.
 *
 * @author epiesan
 * @since Feb 20, 2006
 */
@SuppressWarnings("serial")
public class ConvergedContextImpl extends WebModule implements ConvergedContext {
    private static final LogUtil SIP_LOGGER =
        new LogUtil(LogUtil.SIP_LOG_DOMAIN);
    private SipApplication sipApplication;
    private ServletDispatcher dispatcher;
    private SipFactoryFacade sipFactory;
    private SipApplicationListeners sipApplicationListeners;
    private SipSessionsUtilImpl sipSessionsUtil;

    //private ConvergedApplicationContext context;
    private SipSessionManager sipSessionManager;

    public ConvergedContextImpl(WebContainer wc) {
        super(wc);
    }

    /**
     * Load and initialize all servlets marked "load on startup" in the web
     * application deployment descriptor. Also initializes all SIP servlets that
     * have been marked "load on startup".
     *
     * @param children
     *           Array of wrappers for all currently defined servlets (including
     *           those not declared load on startup)
     */
    @Override
    public void loadOnStartup(Container[] children) {
        loadOnStartupHttp(children);
        loadOnStartupSip();
    }

    private void loadOnStartupSip() {
        if (dispatcher != null) {
            if (SIP_LOGGER.isLoggable(Level.FINE)) {
                SIP_LOGGER.log(Level.FINE, "Loading SIP servlets on startup");
            }
            dispatcher.loadServletsMarkedOnStartup();
        }
    }

    private void loadOnStartupHttp(Container[] children) {
        // Collect "load on startup" servlets that need to be initialized
        TreeMap<Integer, ArrayList<Wrapper>> map = new TreeMap<Integer, ArrayList<Wrapper>>();

        for (int i = 0; i < children.length; i++) {
            Wrapper wrapper = (Wrapper) children[i];
            int loadOnStartup = wrapper.getLoadOnStartup();

            if (loadOnStartup < 0) {
                continue;
            }

            if (loadOnStartup == 0) { // Arbitrarily put them last
                loadOnStartup = Integer.MAX_VALUE;
            }

            Integer key = new Integer(loadOnStartup);
            ArrayList<Wrapper> list = map.get(key);

            if (list == null) {
                list = new ArrayList<Wrapper>();
                map.put(key, list);
            }

            list.add(wrapper);
        }

        // Load the collected "load on startup" servlets
        Iterator keys = map.keySet().iterator();

        while (keys.hasNext()) {
            Integer key = (Integer) keys.next();
            ArrayList list = (ArrayList) map.get(key);
            Iterator wrappers = list.iterator();

            while (wrappers.hasNext()) {
                Wrapper wrapper = (Wrapper) wrappers.next();

                try {
                    wrapper.load();
                } catch (ServletException e) {
                    // NOTE: load errors (including a servlet that throws
                    // UnavailableException from the init() method) are NOT
                    // fatal to application startup
                    SIP_LOGGER.severe(StandardWrapper.getRootCause(e),
                                      "sipstack.servlet_init_error",
                                      wrapper.getName());

                    // Generate a servlet initialization failure notification.
                    FmEventSender.servletInitializationFailed(getName(),
                        wrapper.getName(), e.getMessage());
                }
            }
        }
    }

    /**
     * The method will allow another object to add an attribute into the
     * ServletContext and should only be called in the handler for the
     * BEFORE_DEPLOYMENT_EVENT event.
     *
     * @param attributeName
     *           The name of the context attribute.
     * @param attributeValue
     *           The value of the context attribute.
     */
    public void addAttributeToServletContext(String attributeName,
        Object attributeValue) {
        if ((attributeName != null) && (attributeValue != null)) {
            ServletContext servletContext = getServletContext();
            servletContext.setAttribute(attributeName, attributeValue);
        }
    }

    public ServletDispatcher getDispatcher() {
        return dispatcher;
    }

    public void setDispatcher(ServletDispatcher dispatcher) {
        this.dispatcher = dispatcher;
    }

    public SipApplication getSipApplication() {
        return sipApplication;
    }

    public void setSipApplication(SipApplication sipApplication) {
        this.sipApplication = sipApplication;
    }

    public SipApplicationListeners getSipApplicationListeners() {
        return sipApplicationListeners;
    }

    public void setSipApplicationListeners(
        SipApplicationListeners sipApplicationListeners) {
        this.sipApplicationListeners = sipApplicationListeners;
    }

    public SipFactoryFacade getSipFactory() {
        return sipFactory;
    }

    public void setSipFactory(SipFactoryFacade sipFactory) {
        this.sipFactory = sipFactory;
    }

    /**
     * Sets the given SipSessionsUtil on this converged context.
     *
     * @param util The SipSessionsUtil
     */
    void setSipSessionsUtil(SipSessionsUtilImpl util) {
        this.sipSessionsUtil = util;
    }

    /**
     * Gets the SipSessionsUtil of this converged context.
     *
     * @return The SipSessionsUtil of this converged context
     */
    public SipSessionsUtilImpl getSipSessionsUtil() {
        return sipSessionsUtil;
    }

    /**
     * Get base path.
     */
    private String getBasePath() {
        String docBase = null;
        Container container = this;

        while (container != null) {
            if (container instanceof Host) {
                break;
            }

            container = container.getParent();
        }

        File file = new File(getDocBase());

        if (!file.isAbsolute()) {
            if (container == null) {
                docBase = (new File(engineBase(), getDocBase())).getPath();
            } else {
                // Use the "appBase" property of this container
                String appBase = ((Host) container).getAppBase();
                file = new File(appBase);

                if (!file.isAbsolute()) {
                    file = new File(engineBase(), appBase);
                }

                docBase = (new File(file, getDocBase())).getPath();
            }
        } else {
            docBase = file.getPath();
        }

        return docBase;
    }

    /**
     * Return the servlet context for which this Context is a facade.
     */
    public ServletContext getServletContext() {
        if (context == null) {
            context = new ConvergedApplicationContext(getBasePath(), this);

            if (getAltDDName() != null) {
                context.setAttribute(Globals.ALT_DD_ATTR, getAltDDName());
            }
        }

        return (((ConvergedApplicationContext) context).getConvergedFacade());
    }

    /**
     * Returns a facade for the given session.
     *
     * @return A facade for the given session
     */
    public StandardSessionFacade createSessionFacade(StandardSession session) {
        return new ConvergedHttpSessionFacade(session, sipSessionManager);
    }

    /**
     * Starts the HTTP session and SIP session managers of this converged
     * context.
     */
    protected void managerStart() throws LifecycleException {
        // Start the HTTP session manager
        super.managerStart();

        // Start the SIP session manager
        if ((sipSessionManager != null) &&
                (sipSessionManager instanceof Lifecycle)) {
            ((Lifecycle) sipSessionManager).start();
        }
    }

    /**
     * Stops the HTTP session and SIP session managers of this converged
     * context.
     */
    protected void managerStop() throws LifecycleException {
        // Stop the HTTP session manager
        super.managerStop();

        // Stop the SIP session manager
        if ((sipSessionManager != null) &&
                (sipSessionManager instanceof Lifecycle)) {
            ((Lifecycle) sipSessionManager).stop();
        }
    }

    /**
     * Gets the SIP session manager of this converged context.
     *
     * @return The SIP session manager of this converged context.
     */
    public SipSessionManager getSipSessionManager() {
        try {
            readLock.lock();

            if (sipSessionManager != null) {
                return sipSessionManager;
            }
        } finally {
            readLock.unlock();
        }

        return null;
    }

    /**
     * Sets the SIP session manager for this converged context
     *
     * @param sipSessionManager The new SIP session manager
     */
    public void setSipSessionManager(SipSessionManager sipSessionManager) {
        SipSessionManager oldSipSessionManager;

        try {
            writeLock.lock();
            oldSipSessionManager = this.sipSessionManager;

            if (oldSipSessionManager == sipSessionManager) {
                return;
            }

            this.sipSessionManager = sipSessionManager;

            // Stop the old SipSessionManager
            if (started && (oldSipSessionManager != null) &&
                    (oldSipSessionManager instanceof Lifecycle)) {
                try {
                    ((Lifecycle) oldSipSessionManager).stop();
                } catch (LifecycleException e) {
                    SIP_LOGGER.severe(e, "sipstack.session_manager_stop",
                                      getName());
                }
            }

            if (sipSessionManager != null) {
                sipSessionManager.setContext(this);

                // Start the new SipSessionManager
                if (started && (sipSessionManager instanceof Lifecycle)) {
                    try {
                        ((Lifecycle) sipSessionManager).start();
                    } catch (LifecycleException e) {
                        SIP_LOGGER.severe(e, "sipstack.session_manager_start",
                                          getName());
                    }
                }
            }
        } finally {
            writeLock.unlock();
        }

        // Report this property change to interested listeners
        support.firePropertyChange("sipSessionManager", oldSipSessionManager,
            this.sipSessionManager);
    }

    /**
     * Checks the given request for a specific header, and adds a
     * corresponding cookie to the given response.
     *
     * The exact request header to look for, and the cookie to add to the
     * response, are specific to, and determined by, a subclass (i.e., a
     * SIP ConvergedContext) of this class, as it overrides this method.
     *
     * @param request The request to examine
     * @param response The response to which to add a cookie
     */
    public void addResponseCookie(CoyoteRequest request, CoyoteResponse response) {
        if (!getCookies()) {
            // cookies have been disabled
            return;
        }

        String value = request.getHeader(Constants.PROXY_BEROUTE_HEADER);

        if (value != null) {
            Cookie cookie = new Cookie(Constants.BEROUTE, value);
            request.configureSessionCookie(cookie);
            response.addCookie(cookie);
        } else {
            value = request.getHeader(Constants.PROXY_BEKEY_HEADER);

            if (value != null) {
                Cookie cookie = new Cookie(Constants.BEKEY, value);
                request.configureSessionCookie(cookie);
                response.addCookie(cookie);
            }
        }
    }

    /**
     * Checks the given request for a specific header and adds it to the given
     * StringBuffer.
     *
     * @param request The request to examine
     * @param sb The StringBuffer to which to add the request header
     */
    public void encodeRequestHeader(CoyoteRequest request, StringBuffer sb) {
        String value = request.getHeader(Constants.PROXY_BEROUTE_HEADER);

        if (value != null) {
            sb.append(Constants.BEROUTE_URI_PARAMETER);
            sb.append(value);
        } else {
            value = request.getHeader(Constants.PROXY_BEKEY_HEADER);

            if (value != null) {
                sb.append(Constants.BEKEY_URI_PARAMETER);
                sb.append(value);
            }
        }
    }

    /**
     * Persists the unit of work that the given HTTP session has created or
     * modified.
     *
     * @param s The HTTP session whose associated unit of work to persist
     */
    public void saveUnitOfWork(Session s) throws IOException {
        if (!(sipSessionManager instanceof PersistentSipSessionManagerBase)) {
            return;
        }

        PersistentSipSessionManagerBase ps = (PersistentSipSessionManagerBase)
            sipSessionManager;
        StandardSession session = (StandardSession) s;

        // Does this HTTP session have any SAS associated with it?
        String sasId = session.getSipApplicationSessionId();
        if (sasId == null) {
            return;
        }

        ConvergedHttpSessionFacade facade = (ConvergedHttpSessionFacade)
            session.getSession();
        if (facade != null) {
            SipApplicationSessionImpl sas = (SipApplicationSessionImpl)
                facade.getApplicationSession();
            if (sas != null) {
                ps.saveSipApplicationSession(sas);
            }
        }
    }

    /**
     * Configures the HTTP and SIP session managers of this converged context,
     * by combining information from domain.xml as well as from the converged
     * context's web and SIP related deployment descriptors
     */
    protected void configureSessionManager(SessionManager httpSmBean,
        WebBundleDescriptor wbd, WebModuleConfig wmInfo) {

        SessionConfig sipSessionConfig =
            ((ConvergedDescriptor) wbd).getSunSipSessionConfig();
        SessionManager sipSmBean = null;
        if (sipSessionConfig != null) {
            sipSmBean = sipSessionConfig.getSessionManager();
        }

        ConvergedSessionManagerConfigurationHelper configHelper = new ConvergedSessionManagerConfigurationHelper(this,
                httpSmBean, sipSmBean, wbd, wmInfo);

        /*
         * Configure HTTP session manager
         */
        PersistenceType httpType = configHelper.getPersistenceType();
        String httpFrequency = configHelper.getPersistenceFrequency();
        String httpScope = configHelper.getPersistenceScope();

        PersistenceStrategyBuilderFactory httpFactory = new PersistenceStrategyBuilderFactory();
        PersistenceStrategyBuilder httpBuilder = httpFactory.createPersistenceStrategyBuilder(httpType.getType(),
                httpFrequency, httpScope, this);
        httpBuilder.setLogger(Logger.getLogger("SipContainer"));
        httpBuilder.initializePersistenceStrategy(this, httpSmBean);

        /*
         * Configure SIP session manager
         */
        PersistenceType sipType = configHelper.getSipPersistenceType();
        String sipFrequency = configHelper.getSipPersistenceFrequency();
        String sipScope = configHelper.getSipPersistenceScope();

        SipPersistenceStrategyBuilderFactory sipFactory = new SipPersistenceStrategyBuilderFactory();
        SipPersistenceStrategyBuilder sipBuilder = sipFactory.createPersistenceStrategyBuilder(sipType.getType(),
                sipFrequency, sipScope, this);
        sipBuilder.setLogger(Logger.getLogger("SipContainer"));
        sipBuilder.initializePersistenceStrategy(this,
            configHelper.getSipContainerAvailability());
    }

    public class ConvergedApplicationContext extends ApplicationContext {
        public ConvergedApplicationContext(String basePath,
            StandardContext context) {
            super(basePath, context);
        }

        /**
         * Return a <code>RequestDispatcher</code> object that acts as a wrapper
         * for the named servlet.
         *
         * @param name
         *           Name of the servlet for which a dispatcher is requested
         */
        public RequestDispatcher getNamedDispatcher(String name) {
            // Validate the name argument
            if (name == null) {
                return (null);
            }

            Container child = findChild(name);

            if (child instanceof SipServletWrapper) {
                // Handles SIP requests/responses
                return new SipRequestDispatcher(name,
                    SipFactoryImpl.getInstance().getServiceHandler());
            } else {
                // Handles HTTP requests
                return super.getNamedDispatcher(name);
            }
        }

        public ServletContext getConvergedFacade() {
            return getFacade();
        }
    }
}
