/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.config;

import java.util.HashMap;

import javax.servlet.sip.SipFactory;


/**
 * Th SipFactoryMap exposes a set of methods to hold the SipFactory instances
 * corresponding to an SipApplication or a ConvergedApplication. A HashMap is
 * used to map the SipFactory instances with the names of the corresponding
 * Sip/Converged Application
 *
 * This class is a singleton, and would have a single Map per instance of the
 * Application Server. As per JSR289, there would never be a need to retrive a
 * SipFactory of an Application which is deployed on another instance of the
 * Application Server.
 *
 * @author lmcpepe
 */
public class SipFactoryMap {
    /** Member variables */
    // SipFactoryMap singleton object
    protected static SipFactoryMap sipFactoryMap = new SipFactoryMap();

    // HashMap where the SipFactory instance for each app is stored.
    private HashMap<String, SipFactory> sfMap;

    /** Creates a new instance of SipFactoryMap */
    private SipFactoryMap() {
    }

    /** Public methods */
    public static SipFactoryMap getInstance() {
        return sipFactoryMap;
    }

    /**
     * Adds the instance of the SipFactory associated with an application
     * @param  appName The name of the application whose SipFactory instance is being
     *         stored
     * @param  sf The SipFactory object being added
     */
    public void addSipFactory(String appName, SipFactory sf) {
        if (sfMap == null) {
            sfMap = new HashMap();
            sfMap.put(appName, sf);
        } else {
            sfMap.put(appName, sf);
        }
    }

    /**
     * Retrieves the SipFactory instance for a specific application
     * @param the name of the application whose SipFactory instance is to
     *        be retrieved.
     * @return the SipFactory object
     */
    public SipFactory getSipFactory(String appName) {
        return (SipFactory) sfMap.get(appName);
    }

    /**
     * Removes the SipFactory instance for a particular
     * application
     * @param appName The name of the application whose SipFactory instance is to be
     *        removed from the HashMap
     * @return The SipFactory object that has been removed.
     */
    public SipFactory removeSipFactory(String appName) {
        return (SipFactory) sfMap.remove(appName);
    }

    /**
     * Checks if the application has a SipFactory mapped
     * @param appName the name of the application for which the check is being
     *                made
     */
    public boolean isSipFactoryMapped(String appName) {
        return sfMap.containsKey(appName);
    }
}
