/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.Constants;
import com.ericsson.ssa.config.lease.Lease;
import com.ericsson.ssa.config.lease.LeaseExpiredException;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;

import com.sun.grizzly.CallbackHandler;
import com.sun.grizzly.ConnectorHandler;
import com.sun.grizzly.Context;
import com.sun.grizzly.Controller;
import com.sun.grizzly.Controller.Protocol;
import com.sun.grizzly.DefaultPipeline;
import com.sun.grizzly.DefaultProtocolChain;
import com.sun.grizzly.DefaultSelectionKeyHandler;
import com.sun.grizzly.IOEvent;
import com.sun.grizzly.Pipeline;
import com.sun.grizzly.PipelineFullException;
import com.sun.grizzly.ProtocolChain;
import com.sun.grizzly.ProtocolChainInstanceHandler;
import com.sun.grizzly.SSLCallbackHandler;
import com.sun.grizzly.SSLConnectorHandler;
import com.sun.grizzly.SSLSelectorHandler;
import com.sun.grizzly.TCPConnectorHandler;
import com.sun.grizzly.TCPSelectorHandler;
import com.sun.grizzly.UDPConnectorHandler;
import com.sun.grizzly.UDPSelectorHandler;
import com.sun.grizzly.filter.ReadFilter;
import com.sun.grizzly.filter.SSLReadFilter;
import com.sun.grizzly.util.ByteBufferFactory;
import com.sun.grizzly.util.net.SSLImplementation;
import com.sun.grizzly.util.net.ServerSocketFactory;

import java.io.IOException;

import java.net.InetAddress;
import java.net.InetSocketAddress;

import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;


/**
 *
 * @author ekrigro
 */
public class CtxAwareGrizzlyNetworkManager extends NetworkManager
    implements Runnable {
    private static ConcurrentLinkedQueue<ByteBuffer> byteBuffersPool = new ConcurrentLinkedQueue<ByteBuffer>();
    private Logger logger = Logger.getLogger("SipContainer");

    //private static GrizzlyNetworkManager _singletonInstance = new GrizzlyNetworkManager();
    private Controller controller = new Controller();
    private Pipeline<Callable> ericssonThreadPoolWrapper = null;
    private Layer nextLayer = null;
    private SSLContext sslContext;
    private Config _config = ConfigFactory.getConfig();
    ConnectionManager connectionManager = null;

    /** Creates a new instance of Main */
    public CtxAwareGrizzlyNetworkManager() {
        initPipeline();
    }

    //  ================== Bean Properties =====================
    public synchronized void start() throws IOException {
        int keepAliveTimeoutInSeconds = 600;
        controller = new Controller();

        for (String ctx : SipBindingResolver.instance().getLocalContexts()) {
            startCtx(ctx); //bad name I guess
        }

        SipBindingResolver.instance().registerSipBindingListener(new SipBindingListener() {
                public void newSipBindingAvaliable(String context) {
                    startCtx(context);
                }
            });

        DefaultSelectionKeyHandler keyHandler = new DefaultSelectionKeyHandler() {
                public void expire(SelectionKey key) {
                } // The connections are never going to be closed
            };

        keyHandler.setLogger(logger);
        keyHandler.setTimeout(keepAliveTimeoutInSeconds * 1000); // Not in effect because of the overloaded expire()

        controller.setSelectionKeyHandler(keyHandler);

        final DefaultProtocolChain protocolChain = new DefaultProtocolChain();

        protocolChain.addFilter(new ReadFilter());
        protocolChain.addFilter(new MessageProcessorFilter(this));

        ProtocolChainInstanceHandler instanceHandler = new ProtocolChainInstanceHandler() {
                public ProtocolChain poll() {
                    return protocolChain;
                }

                public boolean offer(ProtocolChain instance) {
                    return true;
                }
            };

        controller.setProtocolChainInstanceHandler(instanceHandler);
        controller.setPipeline(ericssonThreadPoolWrapper);

        //Initiate the outbound connections pool
        connectionManager = new ConnectionManager(controller);
    }

    private ProtocolChainInstanceHandler setupTLSSupport()
        throws IOException {
        // TLS support
        ServerSocketFactory serverSF = null;

        try {
            SSLImplementation sslHelper = SSLImplementation.getInstance();
            serverSF = sslHelper.getServerSocketFactory();
            serverSF.setAttribute("keystoreType", "JKS");
            serverSF.setAttribute("keystore",
                System.getProperty("javax.net.ssl.keyStore"));
            serverSF.setAttribute("truststoreType", "JKS");
            serverSF.setAttribute("truststore",
                System.getProperty("javax.net.ssl.trustStore"));

            //TODO Admin team to make it configurable
            String keyAlias = System.getProperty("sip.network.tls.keyAlias",
                    "s1as");
            serverSF.setAttribute("keyAlias", keyAlias); //Default GF s1sa
            serverSF.init();
        } catch (ClassNotFoundException t) {
            throw new RuntimeException(t);
        }

        sslContext = serverSF.getSSLContext();

        SSLReadFilter readFilter = new SSLReadFilter();
        readFilter.setSSLContext(sslContext);

        //TODO Admin team to make it configurable
        Boolean clientAuth = new Boolean(System.getProperty(
                    "sip.network.tls.clientAuth", Boolean.FALSE.toString()));
        readFilter.setNeedClientAuth(clientAuth);

        final ProtocolChain tlsProtocolChain = new DefaultProtocolChain();
        tlsProtocolChain.addFilter(readFilter);
        tlsProtocolChain.addFilter(new MessageProcessorFilter(this));

        ProtocolChainInstanceHandler tlsProtocolChainHandler = new ProtocolChainInstanceHandler() {
                public ProtocolChain poll() {
                    return tlsProtocolChain;
                }

                public boolean offer(ProtocolChain instance) {
                    return true;
                }
            };

        return tlsProtocolChainHandler;
    }

    public synchronized void stop() throws IOException {
        //TODO do we need some more Grizzly cleanup?
        controller.stop();
    }

    public void startCtx(String ctx) {
        Lease<TargetTuple[]> lease = SipBindingResolver.instance().lease(ctx);

        //TODO lease needs to be kept close to ctx
        //TODO How to deal when things fail.
        try {
            for (TargetTuple targetTuple : lease.getResource()) {
                switch (targetTuple.getProtocol().ordinal()) {
                case SipTransports.UDP: {
                    UDPSelectorHandler udpSelector = createUdpEndpoint(targetTuple.getSocketAddress());
                    controller.addSelectorHandler(udpSelector);

                    break;
                }

                case SipTransports.TCP: {
                    TCPSelectorHandler tcpSelector = createTcpEndpoint(targetTuple.getSocketAddress());
                    controller.addSelectorHandler(tcpSelector);

                    break;
                }

                case SipTransports.TLS: {
                    SSLSelectorHandler tlsSelector = createTLSEndpoint(targetTuple.getSocketAddress());
                    controller.addSelectorHandler(tlsSelector);

                    try {
                        tlsSelector.setProtocolChainInstanceHandler(setupTLSSupport());
                    } catch (IOException e) {
                        //TODO log
                        e.printStackTrace();
                    }

                    break;
                }

                case SipTransports.UNDEFINED://Hmm, should not happen
                    break;
                }
            }
        } catch (LeaseExpiredException e) {
            //TODO logging
            e.printStackTrace();
        }
    }

    void initPipeline() {
        //TODO configurable thread pool size
        ericssonThreadPoolWrapper = new DefaultPipeline(100, 20,
                "SipContainer", 5060);

        SipContainerThreadPool tp = SipContainerThreadPool.getInstance();
        tp.initialize(ericssonThreadPoolWrapper);
    }

    /*public static GrizzlyNetworkManager getInstance() {
        return _singletonInstance;
    }*/
    public void next(SipServletRequestImpl req) {
        LayerHelper.next(req, this, nextLayer);
    }

    public void next(SipServletResponseImpl resp) {
        LayerHelper.next(resp, this, nextLayer);
    }

    public void registerNext(Layer layer) {
        nextLayer = layer;
    }

    public void dispatch(SipServletRequestImpl req) {
        try {
            ByteBuffer buffer = byteBuffersPool.poll();

            if (buffer == null) {
                buffer = ByteBufferFactory.allocateView(false);
            }

            connectionManager.send(req.toBuffer(buffer), req.getRemote());
            buffer.clear();
            byteBuffersPool.offer(buffer);
        } catch (IOException e) {
            logger.log(Level.SEVERE,
                "Failed to send response : " + req.toDebugString(), e);
        }

        if (logger.isLoggable(Level.INFO)) {
            logger.info("Done sending req : " + req.toDebugString());
        }
    }

    public void dispatch(SipServletResponseImpl resp) { //Failback, StreamResponseDispatchern is the normal path

        try {
            connectionManager.send(resp.toBuffer(), resp.getRemote());
        } catch (IOException e) {
            logger.log(Level.SEVERE,
                "Failed to send response : " + resp.toDebugString(), e);
        }

        if (logger.isLoggable(Level.INFO)) {
            logger.info("Done sending resp : " + resp.toDebugString());
        }
    }

    TCPSelectorHandler createTcpEndpoint(InetSocketAddress addr) {
        int keepAliveTimeoutInSeconds = 600;
        int linger = -1;

        TCPSelectorHandler selectorHandler = new TCPSelectorHandler();
        selectorHandler.setPort(addr.getPort());
        selectorHandler.setInet(addr.getAddress());
        selectorHandler.setLinger(linger);
        selectorHandler.setLogger(logger);
        selectorHandler.setReuseAddress(true);
        selectorHandler.setSocketTimeout(keepAliveTimeoutInSeconds * 1000);

        return selectorHandler;
    }

    SSLSelectorHandler createTLSEndpoint(InetSocketAddress addr) {
        int keepAliveTimeoutInSeconds = 600;
        int linger = -1;

        SSLSelectorHandler selectorHandler = new SSLSelectorHandler();
        selectorHandler.setPort(addr.getPort());
        selectorHandler.setInet(addr.getAddress());
        selectorHandler.setLinger(linger);
        selectorHandler.setLogger(logger);
        selectorHandler.setReuseAddress(true);
        selectorHandler.setSocketTimeout(keepAliveTimeoutInSeconds * 1000);

        return selectorHandler;
    }

    UDPSelectorHandler createUdpEndpoint(InetSocketAddress addr) {
        int keepAliveTimeoutInSeconds = 600;
        int linger = -1;

        UDPSelectorHandler selectorHandler = new UDPSelectorHandler();
        selectorHandler.setPort(addr.getPort());
        selectorHandler.setInet(addr.getAddress());
        selectorHandler.setLogger(logger);
        selectorHandler.setReuseAddress(true);

        return selectorHandler;
    }

    public void run() {
        try {
            controller.start();
        } catch (IOException e) {
            logger.log(Level.SEVERE, "Exception, RUN ON GNM called!", e);
        }
    }

    class ConnectionManager {
        private ConcurrentHashMap<TargetTuple, ConnectorHandler> streams = new ConcurrentHashMap<TargetTuple, ConnectorHandler>();

        public ConnectionManager(Controller controller) {
        }

        public void send(ByteBuffer bb, TargetTuple tt)
            throws IOException {
            if ((tt.getProtocol().ordinal() == SipTransports.TCP) ||
                    (tt.getProtocol().ordinal() == SipTransports.TLS)) { //TODO move SipTransports

                ConnectorHandler handler = streams.get(tt);

                if ((handler == null) &&
                        (tt.getProtocol().ordinal() == SipTransports.TCP)) {
                    handler = createHandlerTCP(tt);
                } else if ((handler == null) &&
                        (tt.getProtocol().ordinal() == SipTransports.TLS)) {
                    handler = createHandlerTLS(tt);
                }

                bb.flip();

                try {
                    handler.write(bb, true);
                } catch (IOException ex) {
                    streams.remove(tt); //Clenaup a broken connection
                    logger.log(Level.SEVERE,
                        "Write failed: " + handler.getUnderlyingChannel(), ex);
                    send(bb, tt); //Retry, add counter!?
                }
            } else {
                ConnectorHandler handler = streams.get(tt);

                if (handler == null) {
                    handler = createHandlerUDP(tt);
                }

                bb.flip();

                try {
                    handler.write(bb, false);
                } catch (IOException ex) {
                    streams.remove(tt); //Clenaup a broken connection
                    logger.log(Level.SEVERE,
                        "Write failed: " + handler.getUnderlyingChannel(), ex);
                    throw ex;
                }
            }
        }

        private ConnectorHandler createHandlerTCP(TargetTuple tt)
            throws IOException {
            InetSocketAddress remote = tt.getSocketAddress();
            final TCPConnectorHandler connectorHandler = (TCPConnectorHandler) controller.acquireConnectorHandler(Protocol.TCP);

            CallbackHandler<Context> callbackHandler = new CallbackHandler<Context>() {
                    public void onConnect(IOEvent<Context> ioEvent) {
                        SelectionKey key = ioEvent.attachment().getSelectionKey();

                        try {
                            connectorHandler.finishConnect(key);
                            controller.registerKey(key, SelectionKey.OP_READ,
                                Controller.Protocol.TCP);
                        } catch (IOException ex) {
                            logger.log(Level.SEVERE, "Connect failed: ", ex);
                        }
                    }

                    public void onRead(IOEvent<Context> ioEvent) {
                        try {
                            Context ctx = ioEvent.attachment();
                            SelectionKey key = ctx.getSelectionKey();
                            // disable OP_READ on key before doing anything else
                            key.interestOps(key.interestOps() &
                                (~SelectionKey.OP_READ));
                            ctx.getProtocolChain().execute(ioEvent.attachment());
                        } catch (Throwable e) {
                            logger.log(Level.SEVERE, "Read failed: ", e);
                        }
                    }

                    public void onWrite(IOEvent<Context> ioEvent) {
                        throw new IllegalStateException(
                            "Should not end up here!");
                    }
                };

            long t1 = 0;

            if (logger.isLoggable(Level.FINE)) {
                t1 = System.currentTimeMillis();
            }

            connectorHandler.connect(remote, callbackHandler);

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "================>>> " + (System.currentTimeMillis() - t1));
            }

            ConnectorHandler cached = streams.putIfAbsent(tt, connectorHandler);

            if (cached != null) {
                connectorHandler.close(); //Race cond, remove if multiple are ok.

                return cached;
            }

            return connectorHandler;
        }

        public void waitOnLatch(CountDownLatch latch, int timeout,
            TimeUnit timeUnit) {
            try {
                latch.await(timeout, timeUnit);
            } catch (InterruptedException ex) {
                //TODO Rethrow
                logger.log(Level.SEVERE, "Wait on latch: ", ex);
            }
        }

        private SSLConnectorHandler createHandlerTLS(TargetTuple tt)
            throws IOException {
            InetSocketAddress remote = tt.getSocketAddress();

            final CountDownLatch handshakeDoneLatch = new CountDownLatch(1);
            final SSLConnectorHandler connectorHandler = (SSLConnectorHandler) new SSLConnectorHandler(sslContext);
            connectorHandler.setController(controller);

            final ByteBuffer inBuffer = ByteBufferFactory.allocateView(false);

            SSLCallbackHandler<Context> sslCallbackHander = new SSLCallbackHandler<Context>() {
                    public void onConnect(IOEvent<Context> ioEvent) {
                        try {
                            SelectionKey key = ioEvent.attachment()
                                                      .getSelectionKey();

                            try {
                                connectorHandler.finishConnect(key);
                            } catch (IOException ex) {
                                logger.log(Level.SEVERE, "Connect failed: ", ex);

                                return;
                            }

                            key.attach(connectorHandler.getSSLEngine());

                            if (connectorHandler.isConnected()) {
                                if (connectorHandler.handshake(inBuffer, true)) {
                                    onHandshake(ioEvent);
                                }

                                connectorHandler.getSSLEngine().getSession()
                                                .putValue(SSLReadFilter.HANDSHAKE,
                                    Boolean.TRUE);
                                controller.registerKey(key,
                                    SelectionKey.OP_READ,
                                    Controller.Protocol.TLS);
                            }
                        } catch (IOException e) {
                            logger.log(Level.SEVERE, "Connect failed: ", e);
                        }
                    }

                    // After handshake completes
                    public void onHandshake(IOEvent<Context> ioEvent) {
                        inBuffer.clear();
                        handshakeDoneLatch.countDown();
                    }

                    // Data is ready to be read
                    public void onRead(IOEvent<Context> ioEvent) {
                        try {
                            Context ctx = ioEvent.attachment();
                            SelectionKey key = ctx.getSelectionKey();
                            // disable OP_READ on key before doing anything else
                            key.interestOps(key.interestOps() &
                                (~SelectionKey.OP_READ));
                            ctx.getProtocolChain().execute(ioEvent.attachment());
                        } catch (Throwable e) {
                            logger.log(Level.SEVERE, "Read failed: ", e);
                        }
                    }

                    // Channel is ready to write data
                    public void onWrite(IOEvent<Context> ioEvent) {
                        throw new IllegalStateException(
                            "Should not end up here!");
                    }
                };

            connectorHandler.connect(remote, sslCallbackHander);
            waitOnLatch(handshakeDoneLatch, 10, TimeUnit.SECONDS);
            streams.put(tt, connectorHandler);

            return connectorHandler;
        }

        private UDPConnectorHandler createHandlerUDP(TargetTuple tt)
            throws IOException {
            InetSocketAddress remote = tt.getSocketAddress();
            final UDPConnectorHandler connectorHandler = (UDPConnectorHandler) controller.acquireConnectorHandler(Protocol.UDP);

            CallbackHandler<Context> callbackHandler = new CallbackHandler<Context>() {
                    public void onConnect(IOEvent<Context> ioEvent) {
                        SelectionKey key = ioEvent.attachment().getSelectionKey();

                        try {
                            connectorHandler.finishConnect(key);
                            controller.registerKey(key, SelectionKey.OP_READ,
                                Controller.Protocol.UDP);
                        } catch (IOException ex) {
                            logger.log(Level.SEVERE, "Connect failed: ", ex);
                        }
                    }

                    public void onRead(IOEvent<Context> ioEvent) {
                        try {
                            ioEvent.attachment().getProtocolChain()
                                   .execute(ioEvent.attachment());
                        } catch (Throwable e) {
                            logger.log(Level.SEVERE, "Read failed: ", e);
                        }
                    }

                    public void onWrite(IOEvent<Context> ioEvent) {
                        throw new IllegalStateException(
                            "Should not end up here!");
                    }
                };

            connectorHandler.connect(remote, callbackHandler);
            streams.put(tt, connectorHandler);

            return connectorHandler;
        }
    }
}
