/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container;

import org.apache.catalina.Request;

import org.apache.coyote.Constants;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;

import java.net.URLDecoder;
import java.net.URLEncoder;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;


/**
 * This class is used for recreating the original form data
 * for a request. This data may then be sent using the
 * application/x-www-form-urlencoded media type.
 *
 * @author epiesan
 * @since Apr 27, 2006
 * @reviewed qmigkra 2006-oct-06
 */
public class HttpFormData {
    private static final Logger log = (Logger) Logger.getLogger("SipContainer");
    private final static int INITIAL_FORM_DATA_SIZE = 4096;

    /**
     * The output stream we use when recreating the form data.
     */
    private ByteArrayOutputStream data = new ByteArrayOutputStream(INITIAL_FORM_DATA_SIZE);

    /**
     * Constructor for the HttpFormData class.
     * This constructor recreates the form data if possible.
     *
     * @param request The request we want to recreate the form data for.
     */
    public HttpFormData(Request request) {
        recreateFormData(request);
    }

    /**
     * Returns the recreated form data.
     *
     * @return The recreated form data.
     */
    public byte[] getData() {
        return data.toByteArray();
    }

    /**
     * Returns the length of the recreated form data.
     *
     * @return The length of the recreated form data.
     */
    public int getLength() {
        return data.size();
    }

    /**
     * Recreates the form data out of the parameters specified
     * by the request.
     *
     * @param req The request.
     */
    private void recreateFormData(Request req) {
        // Get the encoding to be used when encoding form parameters.
        javax.servlet.http.HttpServletRequest request = getHttpRequest(req);

        String encoding = Constants.DEFAULT_CHARACTER_ENCODING;

        if (request.getCharacterEncoding() != null) {
            encoding = request.getCharacterEncoding();
        }

        try {
            boolean isFirstParameter = true;
            OutputStreamWriter writer = new OutputStreamWriter(data, encoding);
            Map<String, HashSet<String>> excludedParameters = getQueryParameters(req);
            Enumeration parameterNames = request.getParameterNames();

            while (parameterNames.hasMoreElements()) {
                String parameterName = (String) parameterNames.nextElement();
                String[] parameterValues = request.getParameterValues(parameterName);

                for (String parameterValue : parameterValues) {
                    // Only include parameters that are not part of the query string.
                    if ((excludedParameters != null) &&
                            containsParameter(excludedParameters,
                                parameterName, parameterValue)) {
                        continue;
                    }

                    // Append '&' for subsequent parameters.
                    if (!isFirstParameter) {
                        writer.append('&');
                    } else {
                        isFirstParameter = false;
                    }

                    // Append the parameter name.
                    writer.append(URLEncoder.encode(parameterName, encoding));

                    // Only append parameter value if a value exists.
                    if ((parameterValue != null) &&
                            (parameterValue.length() > 0)) {
                        writer.append('=');
                        writer.append(URLEncoder.encode(parameterValue, encoding));
                    }
                }
            }

            writer.flush();
        } catch (UnsupportedEncodingException e) {
            log.log(Level.WARNING,
                "Unknown character encoding used. Unable to recreate form data.");
        } catch (IOException e) {
            // Won't occur since we are writing to a ByteArrayOutputStream.
        }
    }

    /**
     * Constructs a multivalue map of parameters from the
     * request's query string.
     *
     * @param req The request.
     * @return A multivalue map of parameters.
     * @throws UnsupportedEncodingException If an usupported character
     *   encoding is specified in the request and the flag
     *   useBodyEncodingForURI is set on the connector.
     */
    private static Map<String, HashSet<String>> getQueryParameters(Request req)
        throws UnsupportedEncodingException {
        javax.servlet.http.HttpServletRequest request = getHttpRequest(req);

        String query = request.getQueryString();

        if (query == null) {
            return null;
        }

        Map<String, HashSet<String>> parameters = new HashMap<String, HashSet<String>>();

        // Get the encoding to be used when decoding query parameters.
        String encoding = Constants.DEFAULT_CHARACTER_ENCODING;

        // GLASSFISH TODO Check the getConnector().getUseBodyEncodingForURI()

        //      if (req.getConnector().getUseBodyEncodingForURI())
        //      {
        //         if (request.getCharacterEncoding() != null)
        //         {
        //            encoding = request.getCharacterEncoding();
        //         }
        //      }
        String[] pairs = query.split("&");

        for (String pair : pairs) {
            String[] fields = pair.split("=");
            String name = URLDecoder.decode(fields[0], encoding);
            String value = URLDecoder.decode(fields[1], encoding);

            HashSet<String> values = parameters.get(name);

            if (values == null) {
                values = new HashSet<String>();
                parameters.put(name, values);
            }

            values.add(value);
        }

        return parameters;
    }

    /**
     * Helper method for determining if a parameter-value pair
     * is in a multivalue parameter map or not.
     *
     * @param parameters The multivalue parameter map to search.
     * @param parameterName The name of the parameter.
     * @param parameterValue The value of the parameter.
     * @return True if the parameter-value pair is found.
     */
    private boolean containsParameter(Map<String, HashSet<String>> parameters,
        String parameterName, String parameterValue) {
        if (parameters.containsKey(parameterName)) {
            return parameters.get(parameterName).contains(parameterValue);
        }

        return false;
    }

    private static javax.servlet.http.HttpServletRequest getHttpRequest(
        Request req) {
        return (org.apache.coyote.tomcat5.CoyoteRequest) req;
    }
}
