/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.SipParser;
import com.ericsson.ssa.sip.SipServletMessageImpl;

//import com.ericsson.ssa.sip.SipServletMessageImpl.SipMessageType;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetResolver;
import com.ericsson.ssa.sip.dns.TargetTuple;

import com.sun.grizzly.Context;
import com.sun.grizzly.Controller.Protocol;
import com.sun.grizzly.DefaultPipeline;
import com.sun.grizzly.ProtocolFilter;
import com.sun.grizzly.SelectorHandler;
import com.sun.grizzly.filter.ReadFilter;
import com.sun.grizzly.util.ByteBufferInputStream;
import com.sun.grizzly.util.WorkerThread;

import java.io.IOException;

import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.net.Socket;

import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;

import java.security.cert.X509Certificate;

import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 *
 * @author ekrigro
 */
public class MessageProcessorFilter implements ProtocolFilter, Callable {
    private final static Logger logger = Logger.getLogger("SipContainer");
    private final static DefaultPipeline highPriorityThreadPool = new DefaultPipeline(10,
            10, "SipListenerPriority", 0);
    private static boolean usePriorityQueue = (System.getProperty(
            "sip.network.grizzly.highPriorityThreadPool") != null);
    private final Layer _networkHandler;

    /** Creates a new instance of MessageProcessorFilter */
    public MessageProcessorFilter(Layer layer) {
        _networkHandler = layer;

        if (usePriorityQueue) {
            highPriorityThreadPool.setQueueSizeInBytes(-1);
            highPriorityThreadPool.initPipeline();
            highPriorityThreadPool.startPipeline();
        }
    }

    public boolean execute(final Context ctx) throws IOException {
        SipServletMessageImpl parsedMessage = null;
        Socket s = null;
        InetSocketAddress remoteAddress = null;
        SipServletMessageImpl _message = null;

        final WorkerThread workerThread = ((WorkerThread) Thread.currentThread());

        ByteBuffer buffer = workerThread.getByteBuffer();
        SipParser _parser = SipParser.getInstance();

        Protocol prot = ctx.getProtocol();
        final SelectionKey key = ctx.getSelectionKey();
        final SelectorHandler handler = ctx.getController()
                                           .getSelectorHandler(prot);

        TargetTuple remote = null;
        InetSocketAddress local = null;
        X509Certificate[] x509Certs = null;

        switch (prot) {
        case TCP:
            s = ((SocketChannel) key.channel()).socket();
            remoteAddress = (InetSocketAddress) s.getRemoteSocketAddress();
            remote = new TargetTuple(SipTransports.TCP_PROT, remoteAddress);
            local = (InetSocketAddress) s.getLocalSocketAddress();

            break;

        case UDP:

            DatagramSocket d = ((DatagramChannel) key.channel()).socket();
            remoteAddress = (InetSocketAddress) ctx.getAttribute(ReadFilter.UDP_SOCKETADDRESS);
            remote = new TargetTuple(SipTransports.UDP_PROT, remoteAddress);
            local = (InetSocketAddress) d.getLocalSocketAddress();
       
            break;

        case TLS:
            s = ((SocketChannel) key.channel()).socket();
            remoteAddress = (InetSocketAddress) s.getRemoteSocketAddress();
            remote = new TargetTuple(SipTransports.TLS_PROT, remoteAddress);
            local = (InetSocketAddress) s.getLocalSocketAddress();

            Object[] certs = (Object[]) ctx.removeAttribute(GrizzlyNetworkManager.SIP_CERTS);

            if ((certs != null) && (certs.length > 0)) {
                ArrayList<X509Certificate> al = new ArrayList<X509Certificate>();

                for (int i = 0; i < certs.length; i++) {
                    if (certs[i] instanceof X509Certificate) {
                        al.add((X509Certificate) certs[i]);
                    } else {
                        logger.warning(
                            "Certificate provided by TLS implementation is not of type X509. " +
                            certs[i].getClass());
                    }
                }

                x509Certs = al.toArray(new X509Certificate[al.size()]);
            }

            break;
        }

        try {
            int initialSize = 0;
            buffer.flip();

            int remaining = buffer.remaining();

            while (((remaining > 0) && (initialSize != remaining)) ||
                    (parsedMessage == null)) {
                initialSize = remaining;
                parsedMessage = _parser.parseMessage(_message, buffer,
                        (InetSocketAddress) local, remote, null);
                remaining = buffer.remaining();

                if ((parsedMessage != null) &&
                        parsedMessage.isMessageComplete() && (remaining > 0)) {
                    final SipServletMessageImpl msg = parsedMessage;
                    // Should not only set the first one.
                    msg.setCertificate(x509Certs);
                    parsedMessage = null;
                    _message = null;

                    if (usePriorityQueue) {
                        highPriorityThreadPool.execute(new Callable() {
                                public Object call() {
                                    processMessage(msg, key, handler);

                                    return null;
                                }
                            });
                    } else {
                        SipContainerThreadPool.getInstance().execute(new Runnable() {
                                public void run() {
                                    processMessage(msg, key, handler);
                                }
                            });
                    }

                    continue;
                }

                if ((parsedMessage == null) ||
                        !parsedMessage.isMessageComplete()) {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.fine("The parsed request is not complete " +
                            "- reading more! (" + buffer.position() + ')');
                    }

                    if (buffer.hasRemaining()) {
                        buffer.compact();
                    } else {
                        buffer.clear();
                    }
                                       
                    // Try to receive more bytes with UDP.
                    if (prot == Protocol.UDP){
                        // For UDP, there is no more packet available, so return true.
                        logger.severe("Illegal state exception. UDP packet are missing");
                        return true;
                    } else {
                        // The thread might block when reading more bytes using 
                        // a temporary Selector.
                        ByteBufferInputStream inputStream = new ByteBufferInputStream();
                        inputStream.setSecure((prot == Protocol.TLS));
                        inputStream.setSelectionKey(ctx.getSelectionKey());

                        int nRead = inputStream.read(buffer);

                        if (nRead <= 0) {
                            buffer.clear();
                            logger.severe("Unable to read more bytes after " +
                                "30 seconds pause. Closing the connection.");

                            // Close the connection if we aren't able to parse.
                            ctx.setKeyRegistrationState(Context.KeyRegistrationState.CANCEL);

                            // Do not invoke the next ProtocolFilter, if any.
                            return false;
                        }
                    }

                    remaining = buffer.remaining();
                    _message = parsedMessage;
                }
            }

            buffer.clear();
            _message = null;
        } catch (Throwable t) {
            if (t instanceof Exception) {
                if (logger.isLoggable(Level.WARNING)) {
                    logger.log(Level.WARNING,
                        "Exception processing request " + t.getMessage(), t);
                }
            } else {
                // FIXME alarm for e.g. OutOfMemoryError
                logger.log(Level.SEVERE, "Caught Throwable: ", t);
            }

            buffer.clear();
            /*
            if (PerformanceMBeanListener.isEnabled()) {
                _networkManager.incrEasInvalidSipMessages();
            }
             */
            ctx.setKeyRegistrationState(Context.KeyRegistrationState.CANCEL);

            return false;
        }

        processMessage(parsedMessage, key, handler);

        return true;
    }

    public boolean postExecute(Context context) throws IOException {
        return true;
    }

    protected void processMessage(final SipServletMessageImpl message,
        SelectionKey key, SelectorHandler handler) {
        if (message.getMessageType() == SipServletMessageImpl.SipMessageType.SipRequest) {
            SipServletRequestImpl req = (SipServletRequestImpl) message;

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Network IN request " + req.getMethod() + " --> \r\n" +
                    req.toString());
            }

            if (hasBodyWithoutContentType(message)) {
                return;
            }

            // Respones should use the same link
            if ((req.getRemote().getProtocol().ordinal() == SipTransports.TCP) ||
                    (req.getRemote().getProtocol().ordinal() == SipTransports.TLS)) {
                //TODO maybe store a ref in ctx and first look it up?
                SocketChannel channel = (SocketChannel) key.channel();
                StreamResponseDispatcher rd = new StreamResponseDispatcher(handler,
                        channel, _networkHandler);

                req.pushTransactionDispatcher(rd); //Could be the same channel
            } else {
                // Network Manager handle all responses for DGRAMS for now
                req.pushTransactionDispatcher(_networkHandler); //Could be the same channel
            }

            req.pushApplicationDispatcher(_networkHandler);
            _networkHandler.next(req); //Delegate the
        } else {
            SipServletResponseImpl resp = (SipServletResponseImpl) message;
            Header cseq = resp.getRawHeader(Header.CSEQ);

            /*
            if (cseq == null) {
                if (PerformanceMBeanListener.isEnabled()) {
                    _networkManager.incrEasInvalidSipMessages();
                }
                return; // Drop since missing required header
            }
             */
            String c = cseq.getValue();
            int index = c.indexOf(' ');
            resp.setMethod(c.substring(index + 1));

            if (logger.isLoggable(Level.FINE)) {
                logger.fine("Network IN response " + resp.getStatus() + " " +
                    resp.getMethod() + " --> \r\n" + resp.toString());
            }

            _networkHandler.next(resp);

            /*
            if (PerformanceMBeanListener.isEnabled()) {
                _networkManager.incrEasReceivedSipResponses();
            }
             */
        }
    }

    public boolean hasBodyWithoutContentType(SipServletMessageImpl message) {
        // The SIP-stack should refuse all request with body and without
        // Content-Type header field.
        if (message.hasBody() && (message.getContentType() == null)) {
            try {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "Missing Content-Type header field, the request has a body.");
                }

                SipServletRequestImpl req = (SipServletRequestImpl) message;
                String phraze = "Missing Content-Type header field";
                SipServletResponseImpl resp = req.createTerminatingResponse(400,
                        phraze);

                while (resp.popDispatcher() != null) {
                } // After this layer it shouldn't go anywhere

                TargetResolver tr = TargetResolver.getInstance();
                TargetTuple tt = tr.resolveResponse(resp);

                if (tt != null) {
                    resp.setRemote(tt);
                    _networkHandler.dispatch(resp);
                } else if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "failed to find out where to send error response");
                }
            } catch (Exception ignore) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "unexpected exception: " + ignore);
                }
            }

            return true;
        }

        return false;
    }

    // TODO
    public Object call() throws Exception {
        return null;
    }
}
