/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.fm.FmEventSender;
import com.ericsson.ssa.sip.AddressImpl;
import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipServletMessageImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.timer.ServletTimerImpl;
import com.ericsson.ssa.sip.timer.TimerServiceImpl;

import com.ericsson.util.os.OSUtil;
import com.ericsson.util.os.OSUtilFactory;

import org.apache.catalina.Request;
import org.apache.catalina.Response;
import org.apache.catalina.Valve;

import org.apache.coyote.tomcat5.CoyoteResponse;

import java.io.IOException;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.ServletTimer;
import javax.servlet.sip.TimerListener;


/**
 * Protects the container from overload.
 * </p>
 * When the container detects overload depending on defined thresholds the
 * default response 503 is returned or requests are dropped. Status code 503 is
 * indicating that the server is currently unable to handle the request due to a
 * temporary overloading or maintenance of the server.
 *
 * @author ehsroha
 * @since 2006-mar-29
 *
 */
public class OverloadProtectionManager implements Layer, TimerListener, Valve {
    /**
     * The default threshold percentage level of the cpu for initial sip
     * requests.
     */
    public static int IR_CPU_THRESHOLD = 70;

    /**
     * The default threshold percentage level of the cpu for subsequent sip
     * requests.
     */
    public static int SR_CPU_THRESHOLD = 90;

    /**
     * The default threshold percentage level of the cpu for all messages.
     */
    public static int MM_CPU_THRESHOLD = 99;

    /**
     * The default threshold percentage level of the cpu for http requests.
     */
    public static int HTTP_CPU_THRESHOLD = 70;
    private static final Logger _log = (Logger) Logger.getLogger("SipContainer");
    private static OverloadProtectionManager _singeltonInstance = new OverloadProtectionManager();
    private Layer _nextLayer = null;
    private Valve _nextValve = null;
    private ServletTimerImpl _timer = null;
    private boolean _overloadRegulationEnabled = false;
    private int _sampleRate = 2;
    private int _numberOfSamples = 2;
    private int _initialRequestCPUThreshold = IR_CPU_THRESHOLD;
    private int _subsequentRequestCPUThreshold = SR_CPU_THRESHOLD;
    private int _maxMessageCPUThreshold = MM_CPU_THRESHOLD;
    private int _httpCPUThreshold = HTTP_CPU_THRESHOLD;
    private final OSUtil _osUtil;
    private AtomicInteger _overloadDetectedInitialRequestCounter = new AtomicInteger(0);
    private AtomicInteger _overloadDetectedSubsequentRequestCounter = new AtomicInteger(0);
    private AtomicInteger _overloadDetectedMaxLevelCounter = new AtomicInteger(0);
    private AtomicInteger _overloadDetectedHttpCounter = new AtomicInteger(0);
    private boolean _alarmRaised = false;
    private String _hostName = "";
    private boolean started = false;

    private OverloadProtectionManager() {
        String allProcess = System.getProperty("AllProcess");

        // If AllProcess will be set to true default, if it not set as -D
        // attribute.
        if (allProcess == null) {
            allProcess = "true";

            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE,
                    "AllProcess is not set, will run default 'true'.");
            }
        }

        _osUtil = OSUtilFactory.getOSUtil(allProcess);

        if (_osUtil == null) {
            _log.log(Level.WARNING, "Overload detection failed to start.");
            // lets ensure that this class is not used since no util class was
            // found.
            _overloadRegulationEnabled = false;
        }
    }

    /**
     * Note! The configuration are in preference tree under tm. A list of the
     * parameter:<br> - OverloadRegulation - SampleRate - NumberOfSamples -
     * HttpThreshold - IrThreshold - SrThreshold - MmThreshold
     *
     * -DAllProcess = true: Default monitoring is sun specific monitoring for
     * both linux and windows, which monitors the running jvm total cpu busy
     * activity. -DAllProcess=true, overrides the default sunspecific behaviour
     * and is only valid on Linux and enables accumulated cpu busy monitoring for
     * all process.
     */
    public static OverloadProtectionManager getInstance() {
        return _singeltonInstance;
    }

    /**
     * Initializes this Singleton. It will start to monitor the CPU load in case
     * of raising an alarm.
     */
    public void start() {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "Starting OverloadProtection");
        }

        if (_overloadRegulationEnabled && (_osUtil != null)) {
            // start timer for checking for CPU overload
            if (_timer == null) {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "Starting timer for CPU Load, SampleRate: " +
                        _sampleRate + " * 1000");
                }

                try {
                    _timer = TimerServiceImpl.getInstance()
                                             .createTimer(this,
                            _sampleRate * 1000, false, null);
                } catch (Exception e) {
                    _log.log(Level.SEVERE,
                        "Caught exception when starting timer : " + e);
                }
            }
        }

        started = true;
    }

    private boolean isOverloadDetectedInitialRequest() {
        return _overloadDetectedInitialRequestCounter.get() >= _numberOfSamples;
    }

    private boolean isOverloadDetectedSubsequentRequest() {
        return _overloadDetectedSubsequentRequestCounter.get() >= _numberOfSamples;
    }

    private boolean isOverloadDetectedHttpRequest() {
        return _overloadDetectedHttpCounter.get() >= _numberOfSamples;
    }

    private boolean isOverloadDetectedMaxLevel() {
        return _overloadDetectedMaxLevelCounter.get() >= _numberOfSamples;
    }

    private boolean isInitial(SipServletMessageImpl m) {
        try {
            return m.getAddressHeader(Header.TO)
                    .getParameter(AddressImpl.TAG_PARAM) == null;
        } catch (ServletParseException e) {
            return true;
        }
    }

    public void next(SipServletRequestImpl req) {
        if (_overloadRegulationEnabled) {
            if (!isOverloadDetectedMaxLevel()) {
                boolean initial = isInitial(req);

                if (initial) {
                    if (isOverloadDetectedInitialRequest()) {
                        SipServletResponseImpl resp = req.createTerminatingResponse(503);

                        if (_log.isLoggable(Level.FINE)) {
                            _log.log(Level.FINE,
                                "Method next (req): initial request and isOverloadDetectedInitialRequest");
                            _log.log(Level.FINE,
                                "Method next (req): Terminating response created (503)");
                        }

                        // TR HH52078
                        if (resp == null) {
                            return;
                        }

                        resp.setRemote(req.getRemote());
                        resp.popDispatcher().dispatch(resp);
                        incrEasOverloadRejectedSipResponses();

                        return;
                    }
                } else {
                    // lets handle ACK as an initial request, it's a risk otherwise
                    // that INVITE is stopped but ACK is sent to next layer...
                    if (req.getMethod().equals("ACK")) {
                        if (isOverloadDetectedInitialRequest()) {
                            // drop ACK
                            incrEasOverloadRejectedSipResponses();

                            return;
                        }
                    } else if (isOverloadDetectedSubsequentRequest()) {
                        // TR HH52078
                        SipServletResponseImpl resp = req.createTerminatingResponse(503);

                        if (resp == null) {
                            return;
                        }

                        resp.setRemote(req.getRemote());
                        resp.popDispatcher().dispatch(resp);
                        incrEasOverloadRejectedSipResponses();

                        return;
                    }
                }
            } else {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE, "Max load, drop all sip requests");
                }

                incrEasOverloadRejectedSipResponses();

                return;
            }
        }

        LayerHelper.next(req, this, _nextLayer);
    }

    public void next(SipServletResponseImpl resp) {
        if (_overloadRegulationEnabled && isOverloadDetectedMaxLevel()) {
            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE,
                    "Method next (resp): _overloadRegulationEnabled=true and isOverloadDetectedMaxLevel");
                _log.log(Level.FINE, "Max load, drop all sip responses");
            }

            return;
        }

        LayerHelper.next(resp, this, _nextLayer);
    }

    public void registerNext(Layer layer) {
        _nextLayer = layer;
    }

    public void dispatch(SipServletRequestImpl req) {
        req.popDispatcher().dispatch(req);
    }

    public void dispatch(SipServletResponseImpl resp) {
        resp.peekDispatcher().dispatch(resp);
    }

    private void calculateThreshold(int cpuBusy, AtomicInteger levelCounter,
        Integer threshold) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "calculateThreshold: cpuBusy=" + cpuBusy + " levelCounter=" +
                levelCounter.toString() + " threshold=" + threshold.toString());
        }

        if (cpuBusy >= threshold) {
            if (levelCounter.get() < _numberOfSamples) {
                levelCounter.incrementAndGet();
            }
        } else {
            levelCounter.set(0);
        }
    }

    /**
     * With the set sample rate the overload levels will be re-calculated
     * periodically and a flag will be raised per level if the levels are higher
     * than the protection levels that are set.
     */
    public void timeout(ServletTimer timer) {
        // fetch cpu level from os...
        int busy = cpuBusy();

        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "timeout - cpu: " + busy);
            _log.log(Level.FINE,
                "timeout - initialRequestCPUThreshold: " +
                _initialRequestCPUThreshold);
            _log.log(Level.FINE,
                "timeout - overloadDetectedInitialRequestCounter: " +
                _overloadDetectedInitialRequestCounter);
            _log.log(Level.FINE,
                "timeout - subsequentRequestCPUThreshold: " +
                _subsequentRequestCPUThreshold);
            _log.log(Level.FINE,
                "timeout - overloadDetectedSubsequentRequestCounter: " +
                _overloadDetectedSubsequentRequestCounter);
            _log.log(Level.FINE,
                "timeout - httpCPUThreshold: " + _httpCPUThreshold);
            _log.log(Level.FINE,
                "timeout - overloadDetectedHttpCounter: " +
                _overloadDetectedHttpCounter);
            _log.log(Level.FINE,
                "timeout - maxMessageCPUThreshold: " + _maxMessageCPUThreshold);
            _log.log(Level.FINE,
                "timeout - overloadDetectedMaxLevelCounter: " +
                _overloadDetectedMaxLevelCounter);
        }

        // calculate thresholds...
        calculateThreshold(busy, _overloadDetectedInitialRequestCounter,
            _initialRequestCPUThreshold);
        calculateThreshold(busy, _overloadDetectedSubsequentRequestCounter,
            _subsequentRequestCPUThreshold);
        calculateThreshold(busy, _overloadDetectedHttpCounter, _httpCPUThreshold);
        calculateThreshold(busy, _overloadDetectedMaxLevelCounter,
            _maxMessageCPUThreshold);

        handleAlarm();

        // restart timer again...
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                getClass().getName() +
                " - Restarting timer with _sampleRate: " + _sampleRate +
                " * 1000");
        }

        _timer = TimerServiceImpl.getInstance()
                                 .createTimer(this, _sampleRate * 1000, false,
                null);
    }

    private void handleAlarm() {
        // Check if any threshold has been exceeded, if so raise alarm, if not
        // (and alarm is raised), cease it.
        if (isOverloadDetectedInitialRequest() ||
                isOverloadDetectedSubsequentRequest() ||
                isOverloadDetectedHttpRequest() ||
                isOverloadDetectedMaxLevel()) {
            if (!_alarmRaised) {
                if (_log.isLoggable(Level.INFO)) {
                    _log.log(Level.INFO,
                        "The CPU overload limit is exceeded, will raise an alarm.");
                }

                _alarmRaised = true;

                // Raise the alarm.
                // HH34467 remove reason from raiseAlarmCpuLoadLimitExceeded
                FmEventSender.raiseAlarmCpuLoadLimitExceeded(null);
            }
        } else { // No threshold exceeded

            if (_alarmRaised) { // An alarm has been raised, cease it.

                if (_log.isLoggable(Level.INFO)) {
                    _log.log(Level.INFO,
                        "The CPU overload limit is below the threshold, will clear an alarm.");
                }

                FmEventSender.clearAlarmCpuLoadLimitExceeded();
                _alarmRaised = false;
            }
        }
    }

    private int cpuBusy() {
        return (_osUtil != null) ? _osUtil.cpuBusy() : 0;
    }

    /**
     * Returns the threshold of the cpu for initial requests (range 0-100%). A
     * 503 response will be returned when this level is reached. Default
     * IR_CPU_THRESHOLD.
     *
     * @return the threshold of the cpu for initial requests.
     */
    public Integer getIrThreshold() {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "IrThreshold: " + _initialRequestCPUThreshold);
        }

        return _initialRequestCPUThreshold;
    }

    /**
     * Sets the threshold of the cpu for initial requests (range 0-100%). A 503
     * response will be returned when this level is reached. Default
     * IR_CPU_THRESHOLD.
     *
     * @param threshold
     *           The threshold to set.
     */
    public void setIrThreshold(Integer threshold) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "IrThreshold value will be set to " + threshold.intValue());
        }

        // default IR_CPU_THRESHOLD, answer 503
        if ((threshold >= 0) && (threshold <= 100)) {
            _initialRequestCPUThreshold = threshold;
        }
    }

    /**
     * Returns the threshold of the cpu for subsequent requests (range 0-100%). A
     * 503 response will be returned when this level is reached. Default
     * SR_CPU_THRESHOLD.
     *
     * @return the threshold of the cpu for subsequent requests.
     */
    public Integer getSrThreshold() {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "SrThreshold: " + _subsequentRequestCPUThreshold);
        }

        return _subsequentRequestCPUThreshold;
    }

    /**
     * Sets the threshold of the cpu for subsequent requests (range 0-100%). A
     * 503 response will be returned when this level is reached. Default
     * SR_CPU_THRESHOLD.
     *
     * @param threshold
     *           The threshold to set.
     */
    public void setSrThreshold(Integer threshold) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "SrThreshold value will be set to " + threshold.intValue());
        }

        // default HTTP_CPU_THRESHOLD, answer 503
        if ((threshold >= 0) && (threshold <= 100)) {
            _subsequentRequestCPUThreshold = threshold;
        }
    }

    /**
     * Returns the threshold of the cpu for http requests (range 0-100%). A 503
     * response will be returned when this level is reached. Default
     * HTTP_CPU_THRESHOLD.
     *
     * @return the threshold of the cpu for subsequent requests.
     */
    public Integer getHttpThreshold() {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "HttpThreshold: " + _httpCPUThreshold);
        }

        return _httpCPUThreshold;
    }

    /**
     * Sets the threshold of the cpu for subsequent requests (range 0-100%). A
     * 503 response will be returned when this level is reached. Default
     * HTTP_CPU_THRESHOLD.
     *
     * @param threshold
     *           The threshold to set.
     */
    public void setHttpThreshold(Integer threshold) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "HttpThreshold value will be set to " + threshold.intValue());
        }

        // default SR_CPU_THRESHOLD, answer 503
        if ((threshold >= 0) && (threshold <= 100)) {
            _httpCPUThreshold = threshold;
        }
    }

    /**
     * Returns the threshold of the cpu for maximum load possible for handling
     * messages (range 0-100%). All messages will be dropped when this level is
     * reached. Default MM_CPU_THRESHOLD.
     *
     * @return the threshold of the cpu for max messages
     */
    public Integer getMmThreshold() {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "MmThreshold: " + _maxMessageCPUThreshold);
        }

        return _maxMessageCPUThreshold;
    }

    /**
     * Sets the threshold of the cpu for max load possible for handling messages
     * (range 0-100%). All messages will be dropped when this level is reached.
     * Default MM_CPU_THRESHOLD.
     *
     * @param threshold
     *           The threshold to set.
     */
    public void setMmThreshold(Integer threshold) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "MmThreshold value will be set to " + threshold.intValue());
        }

        // default MM_CPU_THRESHOLD, drop all messages
        if ((threshold >= 0) && (threshold <= 100)) {
            _maxMessageCPUThreshold = threshold;
        }
    }

    /**
     * Returns whether overload protection is used. If true overload protection
     * is turned on.
     *
     * @return whether overload protection is turned on.
     */
    public Boolean getOverloadRegulation() {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "OverloadRegulation: " + _overloadRegulationEnabled);
        }

        return _overloadRegulationEnabled;
    }

    /**
     * Enable/disable the overload Regulation.
     *
     * @param enabled
     *           if true over load protection is used
     */
    public void setOverloadRegulation(Boolean enabled) {
        if ((_overloadRegulationEnabled != enabled.booleanValue()) &&
                (_osUtil != null)) {
            if (enabled.booleanValue() == true) {
                if (_log.isLoggable(Level.INFO)) {
                    _log.log(Level.INFO,
                        "OverloadRegulation was disabled, will be enabled");
                    _log.log(Level.INFO,
                        "Starting timer for CPU Load, SampleRate: " +
                        _sampleRate + " * 1000");
                }

                try {
                    _timer = TimerServiceImpl.getInstance()
                                             .createTimer(this,
                            _sampleRate * 1000, false, null);
                    _overloadRegulationEnabled = enabled.booleanValue();
                } catch (Exception e) {
                    if (started) {
                        _log.log(Level.SEVERE,
                            "Caught exception when trying to start timer. Will re-try later.");
                    }
                }
            } else {
                if (_log.isLoggable(Level.INFO)) {
                    _log.log(Level.INFO,
                        "OverloadRegulation was enabled, will be disabled");
                    _log.log(Level.INFO,
                        "Cease alarm and stop OverloadRegulation timer");
                }

                if (_alarmRaised) { // An alarm has been raised, cease it.
                    FmEventSender.clearAlarmCpuLoadLimitExceeded();
                    _alarmRaised = false;
                }

                _timer.cancel();
                _timer = null;
                _overloadRegulationEnabled = enabled.booleanValue();
            }
        }
    }

    /**
     * Returns the sample rate of updating the overload protection levels.
     *
     * @return the sample rate of updating the overload protection levels.
     */
    public Integer getSampleRate() {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "SampleRate: " + _sampleRate);
        }

        return _sampleRate;
    }

    /**
     * Sets the sample rate of updating the overload protection levels. Must be a
     * positive value.
     *
     * @param rate
     */
    public void setSampleRate(Integer rate) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "SampleRate value will be set to " + rate.intValue());
        }

        if (rate > 0) {
            _sampleRate = rate;
        }
    }

    /**
     * Returns the number of consequence samples that is needed before overload
     * is raised.
     *
     * @return the number of consequence samples that is needed before overload
     *         is raised.
     */
    public Integer getNumberOfSamples() {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "NumberOfSamples: " + _numberOfSamples);
        }

        return _numberOfSamples;
    }

    /**
     * Sets the number of consequence samples that is needed before overload is
     * raised. The sample rate could minimum be set to 2.
     *
     * @param rate
     *           the sample rate
     */
    public void setNumberOfSamples(Integer rate) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE,
                "NumberOfSamples value will be set to " + rate.intValue());
        }

        _numberOfSamples = (rate < 2) ? 2 : rate;
    }

    // Valve methods
    public void backgroundProcess() {
    }

    public String getInfo() {
        return "Overload Regulation for SIP&HTTP";
    }

    public Valve getNext() {
        return _nextValve;
    }

    public int invoke(Request request, Response response)
        throws IOException, ServletException {
        if (_overloadRegulationEnabled) {
            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE,
                    "Method invoke: _overloadRegulationEnabled=true");
            }

            if (isOverloadDetectedMaxLevel()) {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "Method invoke: Max load, drop all http messages");
                }

                return END_PIPELINE;
            } else if (isOverloadDetectedHttpRequest()) {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "Method invoke: isOverloadDetectedHttpRequest, return 503");
                }

                // GLASSFISH TODO We have to Cast until the Request API is added
                CoyoteResponse cResponse = (CoyoteResponse) response;
                cResponse.setStatus(503);

                ServletOutputStream out = cResponse.getOutputStream();
                out.flush();
                incrEasOverloadRejectedHttpResponses();

                return END_PIPELINE;
            }
        }

        // getNext may return null.
        if (_nextValve == null) {
            _log.log(Level.SEVERE,
                "OverloadProtectionManager.invoke(Request,Response) method getNext() returned null, will throw ServletException.");

            throw new ServletException(OverloadProtectionManager.class.getName() +
                " getNext() return null.");
        }

        return INVOKE_NEXT;
    }

    public void postInvoke(Request request, Response response)
        throws IOException, ServletException {
        //
    }

    public void setNext(Valve valve) {
        _nextValve = valve;
    }

    void incrEasOverloadRejectedSipResponses() {
        OverloadManager.getInstance().incrEasOverloadRejectedSipResponses();
    }

    void incrEasOverloadRejectedHttpResponses() {
        OverloadManager.getInstance().incrEasOverloadRejectedHttpResponses();
    }
}
