/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.sip.Dispatcher;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.RemoteLayer;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;


/**
 *
 * @author ekrigro
 * @since 2005-maj-12
 *
 */
public class SerialNetworkManager implements Runnable, Layer, RemoteLayer {
    private static SerialNetworkManager _singeltonInstance = new SerialNetworkManager();
    private Logger _log = (Logger) Logger.getLogger("SipContainer");
    private Layer _nextLayer = null;
    private ScheduledExecutorService _threadPool = null;
    protected boolean _stopped = true;
    private int _threadPoolSize = 10;

    private SerialNetworkManager() {
        super();
    }

    public static SerialNetworkManager getInstance() {
        return _singeltonInstance;
    }

    public synchronized void start() throws Exception {
        _threadPool = Executors.newScheduledThreadPool(_threadPoolSize);
        _stopped = false;
    }

    public synchronized void stop() throws Exception {
        _threadPool.shutdown();
        _stopped = true;
    }

    public void run() {
        while (!_stopped) {
            try {
                Thread.sleep(1000);
            } catch (InterruptedException ignore) {
            }
        }
    }

    public void next(SipServletRequestImpl req) {
        long startTime = 0;
        long stopTime = 0;

        if (_log.isLoggable(Level.FINE)) {
            startTime = System.nanoTime();
        }

        if (req.peekDispatcher() != null) {
            req.pushTransactionDispatcher(this);
            req.pushApplicationDispatcher(this);
            LayerHelper.next(req, this, _nextLayer);
        } else {
            req.pushTransactionDispatcher(this);
            req.pushApplicationDispatcher(this);

            final SipServletRequestImpl r = req;
            final Layer l = this;
            _threadPool.execute(new Runnable() {
                    public void run() {
                        try {
                            LayerHelper.next(r, l, _nextLayer);
                        } catch (Exception e) {
                            _log.log(Level.SEVERE, "", e);
                        }
                    }
                });
        }

        if (_log.isLoggable(Level.FINE)) {
            stopTime = System.nanoTime();
            _log.log(Level.FINE,
                "Run of " + getClass().getName() + " took : " +
                ((stopTime - startTime) / 1000) + " ms");
        }
    }

    public void next(SipServletResponseImpl resp) {
        if (resp.peekDispatcher() != null) {
            LayerHelper.next(resp, this, _nextLayer);
        } else {
            final SipServletResponseImpl r = resp;
            final Layer l = this;
            _threadPool.execute(new Runnable() {
                    public void run() {
                        LayerHelper.next(r, l, _nextLayer);
                    }
                });
        }
    }

    public void registerNext(Layer layer) {
        _nextLayer = layer;
    }

    public void dispatch(SipServletRequestImpl req) {
        // First version always uses local NetworkManager
        Dispatcher d = req.popDispatcher();

        while (d != null)
            d = req.popDispatcher();

        NetworkManager.getInstance().dispatch(req);
    }

    public void dispatch(SipServletResponseImpl resp) {
        // First version always uses local NetworkManager
        Dispatcher d = resp.popDispatcher();

        while (d != null)
            d = resp.popDispatcher();

        NetworkManager.getInstance().dispatch(resp);
    }

    /**
     * @param threadPoolSize
     *            The threadPoolSize to set.
     */
    public void setThreadPoolSize(int threadPoolSize) {
        if (threadPoolSize > 0) {
            _threadPoolSize = threadPoolSize;
        }
    }
}
