/*
 * SipBindingResolver.java
 *
 * Created on August 28, 2007, 3:39 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.Constants;
import com.ericsson.ssa.config.event.ConfigAddEvent;
import com.ericsson.ssa.config.event.ConfigChangeListener;
import com.ericsson.ssa.config.event.ConfigRemoveEvent;
import com.ericsson.ssa.config.event.ConfigUpdateEvent;
import com.ericsson.ssa.config.lease.Lease;
import com.ericsson.ssa.config.lease.LeaseGranter;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 *
 * @author ELNELBO
 */
public class SipBindingResolver extends LeaseGranter<TargetTuple[]> {
    /*
     * A lot to tricky things still to resolve.
     * - Multithreading, Initialization, multiple users and the Config event thread.
     * - Transaction awareness of the ConfigEvent (add node, but prefs not yet set, but event sent)
     * - Expects valid prefs with defaults set.
     */
    private static final SipBindingResolver INSTANCE = new SipBindingResolver();
    private static final String CTX_PARENT = "/SipService/SipListener";
    private static final String CTX_PARENT_REGEXP = "/SipService/SipListener";
    private static final String CTX_REGEXP = "/SipService/SipListener/([^/]+)";
    private static final String SIP_CONTAINER = "/SipContainer";
    private static final String SIP_CONTAINER_PROPERTY = "/SipContainer/ElementProperty";

    /** The context to be used for obtaining a lease for the public binding.
     */
    public static final String PUBLIC_BINDING_CTX = "PUBLIC_BINDING_CTX";
    private String m_LanAddress = null;
    private Config config = ConfigFactory.getConfig();
    private Logger logger = Logger.getLogger("SipContainer");
    private ParentCtxChangeListener parentCtxListener = new ParentCtxChangeListener();
    private Map<String, TargetTuple[]> contexts = new HashMap<String, TargetTuple[]>();
    private Map<String, CtxChangeListener> ctxListeners = new HashMap<String, CtxChangeListener>();
    private Map<String, List<Lease<TargetTuple[]>>> leases = new HashMap<String, List<Lease<TargetTuple[]>>>();
    private List<SipBindingListener> sipBindingListeners = new ArrayList<SipBindingListener>();

    /** Creates a new instance of SipBindingResolver */
    public SipBindingResolver() {
    }

    public static SipBindingResolver instance() {
        return INSTANCE;
    }

    public void init() {
        ConfigFactory.instance()
                     .registerConfigChangeListener(CTX_REGEXP, parentCtxListener);

        //Init the local bindings confgured
        for (String childName : config.getChildNames(CTX_PARENT)) {
            String ctx = getCtxForName(childName);
            contexts.put(ctx, resolveLocalTargetTuples(ctx));
        }

        //TODO setup CtxChangelisteners for the public binding
        contexts.put(PUBLIC_BINDING_CTX, resolvePublicTargetTuples());
    }

    //Lease
    public String[] getLocalContexts() {
        List<String> ctxs = new ArrayList<String>(contexts.keySet());
        ctxs.remove(PUBLIC_BINDING_CTX);

        return ctxs.toArray(new String[ctxs.size()]);
    }

    public Lease<TargetTuple[]> lease(String ctx) {
        Lease<TargetTuple[]> lease = null;

        TargetTuple[] tuples = contexts.get(ctx);

        if (tuples != null) {
            lease = new Lease<TargetTuple[]>(tuples, ctx, this);

            if (leases.containsKey(ctx)) {
                leases.get(ctx).add(lease);
            } else {
                List<Lease<TargetTuple[]>> leasesOfCtx = new ArrayList<Lease<TargetTuple[]>>();
                leasesOfCtx.add(lease);
                leases.put(ctx, leasesOfCtx);
            }
        }

        return lease;
    }

    //Called by Lease
    public boolean renew(Lease<TargetTuple[]> lease) {
        return leases.containsKey(lease.getLeaseCtx()) &&
        leases.get(lease.getLeaseCtx()).contains(lease);
    }

    public void cancel(Lease<TargetTuple[]> lease) {
        if (leases.containsKey(lease.getLeaseCtx())) {
            leases.get(lease.getLeaseCtx()).remove(lease);
        }
    }

    //SipBindingListeners
    public void registerSipBindingListener(SipBindingListener listener) {
        sipBindingListeners.add(listener);
    }

    public void deregisterSipBindingListener(SipBindingListener listener) {
        sipBindingListeners.remove(listener);
    }

    private void processNewSipBindingAvalable(String context) {
        for (SipBindingListener listener : sipBindingListeners) {
            listener.newSipBindingAvaliable(context);
        }
    }

    //ConfigChangeListeners    
    private String getCtxForName(final String name) {
        return CTX_PARENT + "/" + name;
    }

    private String getCtxRegExp(final String ctx) {
        return ctx;
    }

    //Resolvement
    private TargetTuple[] resolveLocalTargetTuples(String ctx) {
        String address = resolveLocalAddress(config.get(ctx, "Address"));
        int port = resolveLocalPort(config.get(ctx, "Port"));
        String transport = config.get(ctx, "Transport");
        List<SipTransports> transports = new ArrayList<SipTransports>();

        for (String prot : transport.split("_")) {
            try {
                transports.add(SipTransports.getTransport(prot));
            } catch (Exception ex) {
                //TODO logging!
                throw new IllegalArgumentException(ex);
            }
        }

        TargetTuple[] targetTuples = new TargetTuple[transports.size()];

        int i = 0;

        for (SipTransports st : transports) {
            targetTuples[i++] = new TargetTuple(st, address, port);
        }

        return targetTuples;
    }

    private String resolveLocalAddress(String anLocalAddress) {
        String address = ((anLocalAddress != null) &&
            (anLocalAddress.length() > 0)) ? anLocalAddress
                                           : Constants.BIND_TO_ANY;

        return address;
    }

    private int resolveLocalPort(String aLocalPort) {
        int port = ((aLocalPort != null) && (aLocalPort.length() > 0))
            ? Integer.parseInt(aLocalPort)
            : Integer.parseInt(Constants.DEFAULT_SIP_PORT);

        return port;
    }

    private TargetTuple[] resolvePublicTargetTuples() {
        String address = resolvePublicAddress(config.get(SIP_CONTAINER,
                    "ExtenalAddress"));

	//TODO Aslong as sip-container.external-sip-port and sip-container.external-sips-port
	//are not provided they are expected as sip-container.property.external-sip-port and sip-container.property.external-sips-port
        //int sipPort = resolvePublicPort(config.get(SIP_CONTAINER, "ExternalSipPort"));
        //int sipsPort = resolvePublicPort(config.get(SIP_CONTAINER, "ExternalSipsPort"));
        int sipPort = resolvePublicPort(config.get(SIP_CONTAINER_PROPERTY, "ExternalSipPort"));
        int sipsPort = resolvePublicPort(config.get(SIP_CONTAINER_PROPERTY, "ExternalSipsPort"));

        TargetTuple[] targetTuples = new TargetTuple[] {
                new TargetTuple(SipTransports.UDP_PROT, address, sipPort),
                new TargetTuple(SipTransports.TCP_PROT, address, sipPort),
                new TargetTuple(SipTransports.TLS_PROT, address, sipsPort)
	};

        return targetTuples;
    }

    private String resolvePublicAddress(String anPublicAddress) {
        String address = ((anPublicAddress != null) &&
            (anPublicAddress.length() > 0)) ? anPublicAddress : getLanAddress();

        return address;
    }

    private int resolvePublicPort(String aPublicPort) {
        int port = ((aPublicPort != null) && (aPublicPort.length() > 0))
            ? Integer.parseInt(aPublicPort)
            : Integer.parseInt(Constants.DEFAULT_SIP_PORT);

        return port;
    }

    /**
     * getLanAddress() tries to find a configured IPv4 address of a physical
     * network interface. Some heuristics will be applied so that on TSP/Linux
     * one of the internal LAN addresses will most likely be returned. <br>
     * getLanAddress() always returns a correctly formatted dotted-decimal IP
     * address string; in the event of a machine with no configured IP addresses
     * it will return "0.0.0.0". <br>
     *
     * @return java.lang.String
     * @author erarafo
     */
    private String getLanAddress() {
        if (m_LanAddress != null) {
            return m_LanAddress;
        }

        String lanAddress = "";
        String preferedPrefix = "172."; // TSP/Linux internal LAN address prefix,
                                        // preferred

        String avoidedPrefix = "127."; // Loopback interface, to be avoided
        String avoidedInterfacenamePrefix = "i4dd"; // VIP addresses, to be
                                                    // avoided

        String me = "getLanAddress(): ";

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                me +
                "try to find a configured IP address of a physical network interface");
        }

        int score = 0;
        int newscore;
        Enumeration<NetworkInterface> nifs = null;

        try {
            nifs = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException se) {
            logger.log(Level.SEVERE,
                me +
                "SocketException when trying to get our network interfaces", se);
        }

        if (nifs != null) {
            while (nifs.hasMoreElements()) {
                NetworkInterface nif = nifs.nextElement();
                String ifName = nif.getName();

                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, me + "found interface: " + ifName);
                }

                int addrCount = 0;

                for (Enumeration<InetAddress> addrs = nif.getInetAddresses();
                        addrs.hasMoreElements();) {
                    InetAddress addr = addrs.nextElement();
                    addrCount++;

                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE,
                            me + "found address " + addr + " for interface " +
                            ifName);
                    }

                    String addrString = addr.getHostAddress();

                    if (addrString.startsWith(preferedPrefix)) {
                        newscore = 40; // 40: preferred address

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    } else if (addrString.startsWith(avoidedPrefix)) {
                        newscore = 10; // 10: avoid this address if possible

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    } else if (ifName.startsWith(avoidedInterfacenamePrefix)) {
                        newscore = 20; // 20: avoid this address if possible

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    } else if (addrString.indexOf(":") >= 0) {
                        newscore = 25; // score 25 for IPv6 address

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    } else {
                        newscore = 30; // 30: neither preferred nor avoided

                        if (score < newscore) {
                            lanAddress = addrString;
                            score = newscore;

                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore +
                                    ", best so far");
                            }
                        } else {
                            if (logger.isLoggable(Level.FINE)) {
                                logger.log(Level.FINE,
                                    me + "score: " + newscore + ", ignored");
                            }
                        }
                    }
                }

                if (addrCount < 1) {
                    if (logger.isLoggable(Level.INFO)) {
                        logger.log(Level.INFO,
                            me +
                            "found no configured IP addresses for interface " +
                            ifName);
                    }
                } else if (addrCount > 1) {
                    if (logger.isLoggable(Level.INFO)) {
                        logger.log(Level.INFO,
                            me + "found " + addrCount +
                            " configured IP addresses for interface " + ifName);
                    }
                }
            }
        }

        if (lanAddress == null) {
            logger.log(Level.SEVERE,
                me +
                "no interface with a configured IP address found, using 0.0.0.0");
            lanAddress = Constants.BIND_TO_ANY;
        } else if (score < 30) {
            logger.log(Level.WARNING,
                me + "only found unsuitable IP addresses; using " + lanAddress +
                " in lack of better");
        }

        if (logger.isLoggable(Level.INFO)) {
            logger.log(Level.INFO,
                "parameter <LanAddress>: using " +
                ((score > 30) ? "preferred " : "") + "value '" + lanAddress +
                "'");
        }

        m_LanAddress = lanAddress;

        return lanAddress;
    }

    public String reportBindings() {
        StringBuffer buf = new StringBuffer("SipBindings: ");

        for (Map.Entry<String, TargetTuple[]> entries : contexts.entrySet()) {
            buf.append("\ncontext: '");
            buf.append(entries.getKey());
            buf.append("'.\n");

            for (TargetTuple tuple : entries.getValue()) {
                buf.append(tuple);
            }
        }

        return buf.toString();
    }

    class ParentCtxChangeListener implements ConfigChangeListener {
        /**
         * Prepares the new TargetTuples for the ctx.
         * registeres CtxChangeListener
         */
        public void handleConfigEvent(ConfigAddEvent event) {
            contexts.put(event.getNode(),
                resolveLocalTargetTuples(event.getNode()));

            CtxChangeListener listener = new CtxChangeListener();
            ctxListeners.put(event.getNode(), listener);
            ConfigFactory.instance()
                         .registerConfigChangeListener(getCtxRegExp(
                    event.getNode()), listener);
            processNewSipBindingAvalable(event.getNode());
        }

        public void handleConfigEvent(ConfigUpdateEvent event) {
            //ignore
        }

        /**
         * Prepares the new TargetTuples for the ctx.
         * Cancels all associated leases.
         * deregisteres CtxChangeListener
         */
        public void handleConfigEvent(ConfigRemoveEvent event) {
            List<Lease<TargetTuple[]>> leasesOfCtx = leases.remove(event.getNode());
            TargetTuple[] tuples = contexts.remove(event.getNode());

            if (leasesOfCtx != null) {
                for (Lease<TargetTuple[]> lease : leasesOfCtx) {
                    lease.cancel();
                }
            }

            CtxChangeListener listener = ctxListeners.remove(event.getNode());

            if (listener != null) {
                ConfigFactory.instance().deregisterConfigChangeListener(listener);
            }
        }
    }

    class CtxChangeListener implements ConfigChangeListener {
        public void handleConfigEvent(ConfigAddEvent event) {
            //ignore
        }

        /**
         * Prepares the new TargetTuples for the ctx.
         * Cancels all associated leases.
         */
        public void handleConfigEvent(ConfigUpdateEvent event) {
            contexts.put(event.getNode(),
                resolveLocalTargetTuples(event.getNode()));

            List<Lease<TargetTuple[]>> leasesOfCtx = leases.remove(event.getNode());

            if (leasesOfCtx != null) {
                for (Lease<TargetTuple[]> lease : leasesOfCtx) {
                    lease.cancel();
                }
            }
        }

        public void handleConfigEvent(ConfigRemoveEvent event) {
            //ignore
        }
    }
}
