/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.Constants;
import com.ericsson.ssa.container.startup.PerformanceMBeanListener;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.SipServletMessageImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;

import java.io.IOException;

import java.net.InetAddress;
import java.net.InetSocketAddress;

import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;

import java.util.logging.Level;


/**
 * TODO Add comments (class description)
 *
 * @author ehsroha
 * @since 2006-jan-04
 */
public class UDPBlockingListener extends UDPListener {
    private Thread m_Thread = null;
    private Config _config = ConfigFactory.getConfig();

    public UDPBlockingListener(int port, OLDNetworkManager n, Layer l) {
        super(port, n, l);
    }

    @Override
    public void open() throws IOException {
        _dgramChannel = DatagramChannel.open();
        // this is a blocking udp channel...
        _dgramChannel.configureBlocking(true);

        InetSocketAddress local;
        InetAddress ia = InetAddress.getByName(_config.get(Constants.HOST));
        local = new InetSocketAddress(ia, _port);
        _dgramChannel.socket().bind(local);
        // channel.socket().setSoTimeout( timeoutValue );
        _readBuffer = ByteBuffer.allocate(UDP_BUFFER_SIZE);
        _writeBuffer = ByteBuffer.allocate(UDP_BUFFER_SIZE);
        // start this channel in its own thread...
        m_Thread = new Thread(this, getClass().getName());
        m_Thread.start();

        // register with read event
        _networkManager.getSelector().wakeup();
    }

    @Override
    protected void closeImpl() throws IOException {
        if (m_Thread != null) {
            m_Thread.interrupt();
        }

        if (_dgramChannel != null) {
            _dgramChannel.close();
            _dgramChannel = null;
        }
    }

    @Override
    public void run() {
        // loop and block for incoming udp data...
        while (!Thread.interrupted()) {
            InetSocketAddress remote = null;

            try {
                remote = (InetSocketAddress) _dgramChannel.receive(_readBuffer);
            } catch (IOException ioe) {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE, "", ioe);
                }
            }

            _readBuffer.flip();

            if (_readBuffer.hasRemaining()) {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "In Dgram :" + Thread.currentThread().getName());
                    _log.log(Level.FINE, "Buffer = " + _readBuffer.limit());
                }

                handleMessage(_readBuffer, getLocal(),
                    new TargetTuple(SipTransports.UDP_PROT, remote));
            } else {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "Zero bytes in UDP ByteBuffer = " +
                        _readBuffer.limit());
                }
            }
        }
    }

    @Override
    protected void handleMessage(ByteBuffer buffer, InetSocketAddress local,
        TargetTuple remote) {
        SipServletMessageImpl parsedMessage = null;

        try {
            // TODO - change so that the parser can do next(), to avoid
            // instanceof
            int initialSize = 0;
            int remaining = buffer.remaining();

            while ((remaining > 0) && (initialSize != remaining)) {
                initialSize = remaining;
                parsedMessage = _parser.parseMessage(_message, buffer, local,
                        remote, null);
                remaining = buffer.remaining();

                if ((parsedMessage != null) &&
                        parsedMessage.isMessageComplete() && (remaining > 0)) {
                    final SipServletMessageImpl msg = parsedMessage;
                    parsedMessage = null;
                    _threadPool.execute(new Runnable() {
                            public void run() {
                                processMessage(msg);
                            }
                        });
                }

                if ((remaining > 0) && (initialSize != remaining)) {
                    _message = parsedMessage;
                    buffer.mark(); // TODO - check if reset is called?
                    buffer.compact();
                    buffer.flip();

                    if (_log.isLoggable(Level.FINE)) {
                        _log.log(Level.FINE,
                            "Position = " + buffer.position() + " ,Remains = " +
                            buffer.remaining());
                    }
                }
            }

            if (parsedMessage == null) {
                if (_log.isLoggable(Level.INFO)) {
                    _log.log(Level.INFO,
                        "The parsed request is null - not continuing!(" +
                        buffer.position() + ')');
                }

                int p = buffer.limit();
                buffer.clear();
                buffer.position(p);
                _message = null;

                if (PerformanceMBeanListener.isEnabled()) {
                    _networkManager.incrEasInvalidSipMessages();
                }

                return;
            }

            if (!parsedMessage.isMessageComplete()) {
                if (_log.isLoggable(Level.INFO)) {
                    _log.log(Level.INFO,
                        "The parsed request is not complete - not continuing! (" +
                        buffer.position() + ')');
                }

                if (buffer.hasRemaining()) {
                    buffer.compact();
                } else {
                    buffer.clear();
                }

                _message = parsedMessage;

                return;
            }

            buffer.clear();
            _message = null;
        } catch (Exception e) {
            if ((parsedMessage != null) && parsedMessage.isMessageComplete() &&
                    parsedMessage instanceof SipServletRequestImpl) {
                try {
                    SipServletRequestImpl req = (SipServletRequestImpl) parsedMessage;

                    //             TR HH52078
                    SipServletResponseImpl resp = req.createTerminatingResponse(500);

                    if (resp == null) {
                        return;
                    }

                    while (resp.popDispatcher() != null) {
                    } // After this layer it shouldn't go anywhere

                    dispatch(resp);
                } catch (Exception ignore) {
                }
            }

            if (_log.isLoggable(Level.INFO)) {
                _log.log(Level.INFO,
                    "Exception processing request " + e.getMessage(), e);
            }

            if (PerformanceMBeanListener.isEnabled()) {
                _networkManager.incrEasInvalidSipMessages();
            }

            return;
        }

        final SipServletMessageImpl sendParsedMessage = parsedMessage;
        _threadPool.execute(new Runnable() {
                public void run() {
                    processMessage(sendParsedMessage);
                }
            });
    }
}
