/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.Constants;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;

import java.io.IOException;

import java.net.InetAddress;
import java.net.InetSocketAddress;

import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;

import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;


/**
 * @author ehsroha
 * @reveiwed ehsroha 2006-nov-14
 */
public class UDPListener extends LinkBase implements Link {
    protected static final String _transport = "UDP";

    // // Max UDP Datagram unit
    protected static final int UDP_BUFFER_SIZE = 65535;
    protected Logger _log = (Logger) Logger.getLogger("SipContainer");
    protected DatagramChannel _dgramChannel = null;
    protected int _port = -1;
    private Config _config = ConfigFactory.getConfig();

    public UDPListener(int port, OLDNetworkManager n, Layer l) {
        super(n, l);
        _networkManager = n;
        _port = port;
    }

    public String getTransport() {
        return _transport;
    }

    public TargetTuple getTargetTuple() {
        return null;
    }

    public boolean isOpen() {
        return (_dgramChannel != null) ? _dgramChannel.isOpen() : false;
    }

    public void open() throws IOException {
        _dgramChannel = DatagramChannel.open();
        _dgramChannel.configureBlocking(false);

        InetSocketAddress local;
        InetAddress ia = InetAddress.getByName(_config.get(Constants.HOST));
        local = new InetSocketAddress(ia, _port);
        _dgramChannel.socket().bind(local);
        // channel.socket().setSoTimeout( timeoutValue );
        _dgramChannel.register(_networkManager.getSelector(),
            SelectionKey.OP_READ, this);
        _readBuffer = ByteBuffer.allocate(UDP_BUFFER_SIZE);
        _writeBuffer = ByteBuffer.allocate(UDP_BUFFER_SIZE);

        // register with read event
        _networkManager.getSelector().wakeup();

        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "Opening " + ia + ":" + _port);
        }
    }

    protected void closeImpl() throws IOException {
        _dgramChannel.close();
    }

    public void run() {
        _readMutex.lock();

        InetSocketAddress remote = null;

        try {
            remote = (InetSocketAddress) _dgramChannel.receive(_readBuffer);
        } catch (IOException ioe) {
            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE, "Caught IOException: ", ioe);
            }

            _readMutex.unlock();
        } catch (Throwable t) {
            // FIXME alarm for e.g. OutOfMemoryError
            _log.log(Level.SEVERE, "Caught Throwable: ", t);
            _readMutex.unlock();
        } finally {
            _networkManager.releaseSelectorSemaphore();
        }

        _readBuffer.flip();

        if (_readBuffer.hasRemaining()) {
            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE,
                    "In Dgram :" + Thread.currentThread().getName());
                _log.log(Level.FINE, "Buffer = " + _readBuffer.limit());
            }

            handleMessage(_readBuffer, getLocal(),
                new TargetTuple(SipTransports.UDP_PROT, remote));
        } else {
            if (_log.isLoggable(Level.FINE)) {
                _log.log(Level.FINE,
                    "Zero bytes in UDP ByteBuffer = " + _readBuffer.limit());
            }

            _readBuffer.clear();
            _readMutex.unlock();
        }
    }

    public void write(InetSocketAddress remote, ByteBuffer message)
        throws IOException {
        while (message.hasRemaining()) {
            _dgramChannel.send(message, remote);
        }
    }

    public InetSocketAddress getLocal() {
        return (InetSocketAddress) _dgramChannel.socket().getLocalSocketAddress();
    }

    public InetSocketAddress getRemote() {
        return (InetSocketAddress) _dgramChannel.socket()
                                                .getRemoteSocketAddress();
    }

    public SelectableChannel getSelectableChannel() {
        return _dgramChannel;
    }

    public String getInfo() {
        return getClass().getName();
    }
}
