/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 */
package com.ericsson.ssa.container.auth;

import com.ericsson.ssa.container.sim.ServletDispatcher;
import com.ericsson.ssa.dd.ResourceCollection;
import com.ericsson.ssa.dd.SecurityConstraint;
import com.ericsson.ssa.dd.SecurityConstraint;
import com.ericsson.ssa.dd.SipApplication;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;

import com.sun.enterprise.security.SecurityContext;

import com.sun.web.security.WebPrincipal;
import javax.servlet.sip.SipApplicationRouterInfo;
import org.jvnet.glassfish.comms.security.auth.impl.DigestAuthenticator;
import org.jvnet.glassfish.comms.security.auth.impl.PAssertedAuthenticator;
import org.jvnet.glassfish.comms.security.authorize.SipResourcePermission;
import org.jvnet.glassfish.comms.security.authorize.SipRoleRefPermission;
import org.jvnet.glassfish.comms.security.authorize.SipUserDataPermission;
import org.jvnet.glassfish.comms.security.util.PolicyBuilder;

import java.net.URL;

import java.security.CodeSource;
import java.security.Policy;
import java.security.Principal;
import java.security.PrivilegedExceptionAction;
import java.security.ProtectionDomain;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Set;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.security.auth.Subject;
import javax.security.jacc.PolicyContext;
import javax.security.jacc.WebUserDataPermission;

import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipServletResponse;


/**
 *
 * @author K.Venugopal@sun.com
 */
public final class AuthModule {
    private SipApplication sipApplicationModel;
    private HashMap<Key, ServletConstraints> servletConstraints = null;
    private PolicyBuilder policyUtil = new PolicyBuilder();
    private String contextId = null;
    private CodeSource codeSource = null;
    private Policy policy = null;
    private String sarName = "";

    public AuthModule(SipApplication am,String sarName) {
        this.sipApplicationModel = am;
        policy = Policy.getPolicy();
        policy.refresh();
        this.sarName = sarName;
        contextId = sipApplicationModel.getAppName();
        if(contextId == null || contextId.length() == 0){
            contextId = sarName;
        }
        contextId = contextId + "/_" + contextId;

        String code = removeSpaces(contextId);
        
        servletConstraints = new HashMap<Key, ServletConstraints>();

        java.net.URI uri = null;

        try {
            uri = new java.net.URI("file:///" + code);

            if (uri != null) {
                codeSource = new CodeSource(new URL(uri.toString()),
                        (java.security.cert.Certificate[]) null);
            }
        } catch (java.net.URISyntaxException use) {
            //log
            throw new RuntimeException(use);
        } catch (java.net.MalformedURLException mue) {
            //log
            throw new RuntimeException(mue);
        }

        Collection<SecurityConstraint> scCollection = sipApplicationModel.getSecurityConstraints();
        Iterator<SecurityConstraint> scItr = scCollection.iterator();

        while (scItr.hasNext()) {
            SecurityConstraint constraint = scItr.next();
            @SuppressWarnings(value = "unchecked")
            Collection<ResourceCollection> resourceCollection = constraint.getResourceCollections();
            Iterator<ResourceCollection> rcItr = resourceCollection.iterator();

            while (rcItr.hasNext()) {
                ResourceCollection rc = rcItr.next();
                @SuppressWarnings(value = "unchecked")
                Collection<String> sn = rc.getServletNames();
                @SuppressWarnings(value = "unchecked")
                Collection<String> sm = rc.getSipMethods();
                Collection<String> roles = constraint.getAuthorizationConstraintRoleNames();
                Iterator<String> snItr = sn.iterator();

                while (snItr.hasNext()) {
                    String servletName = snItr.next();
                    Iterator<String> smItr = sm.iterator();

                    while (smItr.hasNext()) {
                        String sipMethod = smItr.next();
                        Key key = new Key(servletName, sipMethod);
                        Set<String> roleSet = null;

                        if (servletConstraints.containsKey(key)) {
                            ServletConstraints sc = servletConstraints.get(key);
                            roleSet = sc.roles;
                        } else {
                            roleSet = new HashSet<String>();

                            boolean proxy = constraint.isProxyAuthenticated();
                            ServletConstraints sc = new ServletConstraints(proxy,
                                    roleSet);
                            servletConstraints.put(key, sc);
                        }

                        roleSet.addAll(roles);
                    }
                }
            }
        }
    }

    
      
    public final void authenticate(SipServletRequest request, String servletName)
        throws Exception {
        if (sipApplicationModel != null) {
            String authMethod = sipApplicationModel.getLoginConfigAuthMethod();

            if (!((authMethod == null) || (authMethod.length() == 0))) {
                int statusCode = 401;
                boolean proxyAuthenticate = false;
                Key key = new Key(servletName, request.getMethod());

                if (servletConstraints.containsKey(key)) {
                    ServletConstraints sc = servletConstraints.get(key);
                    proxyAuthenticate = sc.proxy;
                }

                if (proxyAuthenticate) {
                    statusCode = 407;
                }

                if ("DIGEST".equals(authMethod)) {
                    DigestAuthenticator auth = new DigestAuthenticator(sipApplicationModel.getLoginConfigRealmName());

                    if (!auth.hasAuthorizationHeader(request)) {
                        SipServletResponseImpl resp =((com.ericsson.ssa.sip.SipServletRequestImpl) request).createTerminatingResponse(statusCode);
                                //(SipServletResponseImpl) sd.createResponse((SipServletRequestImpl)request,401);
                                //((com.ericsson.ssa.sip.SipServletRequestImpl) request).createTerminatingResponse(statusCode);
                        auth.setAuthenticateHeader(resp, request,
                            proxyAuthenticate);
                        resp.popDispatcher().dispatch(resp);
                        return;
                    } else {
                        Principal authPrincipal = auth.authenticate(request);

                        if (authPrincipal != null) {
                            ((SipServletRequestImpl) request).setUserPrincipal(authPrincipal);
                            ((SipServletRequestImpl) request).setUser(authPrincipal.getName());
                        }
                    }
                } else if ("P-Asserted".equals(authMethod)) {
                    PAssertedAuthenticator auth = new PAssertedAuthenticator(sipApplicationModel.getLoginConfigRealmName());
                    Principal authPrincipal = auth.authenticate(request);

                    if (authPrincipal == null) {
                        SipServletResponseImpl resp = ((com.ericsson.ssa.sip.SipServletRequestImpl) request).createTerminatingResponse(statusCode);
                        resp.popDispatcher().dispatch(resp);

                        return;
                    }
                }
            }
        }
    }

    public boolean hasResourcePermissions(SipServletRequest request,
        String servletName) throws Exception {
        return hasResourcePermissions(request.getMethod(), servletName);
    }

    public boolean hasResourcePermissions(String sipMethod, String servletName) {
        try {
            SecurityContext secCtx = SecurityContext.getCurrent();
            policyUtil.setPolicyContext(contextId);

            Subject subject = secCtx.getSubject();

            Set<Principal> principalSet = subject.getPrincipals();

            Principal[] principals = (principalSet == null) ? null
                                                            : principalSet.toArray(new Principal[0]);

            ProtectionDomain pd = new ProtectionDomain(codeSource, null, null,
                    principals);
            SipResourcePermission srp = new SipResourcePermission(servletName,
                    sipMethod);

            return policy.implies(pd, srp);
        } catch (Throwable ex) {
            Logger.getLogger(AuthModule.class.getName())
                  .log(Level.SEVERE, null, ex);
            throw new RuntimeException(ex);
        }
    }

    public boolean checkServletConstraints(SipServletRequest request,
        String servletName) throws Exception {
        if (hasRoleRefPermission(request, servletName) &&
                hasResourcePermissions(request, servletName)) {
            return true;
        }

        SipServletResponseImpl resp = ((com.ericsson.ssa.sip.SipServletRequestImpl) request).createTerminatingResponse(403);
        resp.popDispatcher().dispatch(resp);

        return false;
    }

    /**
     *  All Security constraints must be satisfied.
     */
    public boolean hasRoleRefPermission(SipServletRequest request,
        String servletName) {
        return hasRoleRefPermission(request.getMethod(), servletName);
    }

    public boolean hasRoleRefPermission(String sipMethod, String servletName) {
        SecurityContext secCtx = SecurityContext.getCurrent();

        try {
            policyUtil.setPolicyContext(contextId);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }

        Subject subject = secCtx.getSubject();

        Set<Principal> principalSet = subject.getPrincipals();

        Key key = new Key(servletName, sipMethod);
        ServletConstraints sc = servletConstraints.get(key);

        if (sc == null) {
            return false;
        }

        Set<String> roles = sc.roles;
        Iterator<String> roleItr = roles.iterator();
        boolean isInRole = false;

        Principal[] principals = principals = ((principalSet == null) ? null
                                                                      : (Principal[]) principalSet.toArray(new Principal[0]));

        ProtectionDomain pd = new ProtectionDomain(codeSource, null, null,
                principals);

        while (roleItr.hasNext()) {
            String role = roleItr.next();
            SipRoleRefPermission srp = new SipRoleRefPermission(servletName,
                    role);
            isInRole = policy.implies(pd, srp);

            if (!isInRole) {
                break;
            }
        }

        return isInRole;
    }

    public boolean hasUserDataPermissions(String servletName, String sipMethod,
        boolean isSecured) {
        SecurityContext secCtx = SecurityContext.getCurrent();

        try {
            policyUtil.setPolicyContext(contextId);
        } catch (Throwable ex) {
            throw new RuntimeException(ex);
        }

        Subject subject = secCtx.getSubject();
        Set<Principal> principalSet = subject.getPrincipals();

        boolean isInRole = false;
        Principal[] principals = ((principalSet == null) ? null
                                                         : (Principal[]) principalSet.toArray(new Principal[0]));

        ProtectionDomain pd = new ProtectionDomain(codeSource, null, null,
                principals);
        String transport = "NONE";

        if (isSecured) {
            transport = "CONFIDENTIAL";
        }

        Set<String> sm = new HashSet<String>();
        sm.add(sipMethod);

        SipUserDataPermission wdp = new SipUserDataPermission(servletName, sm,
                transport);

        return policy.implies(pd, wdp);
    }

    private String removeSpaces(String withSpaces) {
        return withSpaces.replace(' ', '_');
    }

    public boolean needsAuthentication(String servletName, String sipMethod) {
        Key key = new Key(servletName, sipMethod);

        if (servletConstraints.containsKey(key)) {
            return true;
        }

        return false;
    }

    class Key {
        String servletName = null;
        String methodName = null;
        int hashCode = -1;

        public Key() {
        }

        public Key(String servletName, String methodName) {
            this.servletName = servletName;
            this.methodName = methodName;
        }

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof Key)) {
                return false;
            }

            Key key = (Key) obj;

            if (((servletName != null) && servletName.equals(key.servletName)) &&
                    ((methodName != null) && methodName.equals(key.methodName))) {
                return true;
            }

            return false;
        }

        @Override
        public int hashCode() {
            if (hashCode == -1) {
                StringBuilder builder = new StringBuilder();
                builder.append(servletName);
                builder.append(methodName);
                hashCode = builder.toString().hashCode();
            }

            return hashCode;
        }

        public void reset() {
            servletName = null;
            methodName = null;
        }
    }

    class ServletConstraints {
        boolean proxy = false;
        Set<String> roles = null;

        public ServletConstraints(boolean proxy, Set<String> roles) {
            this.proxy = proxy;
            this.roles = roles;
        }
    }
}
