/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.container.startup;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.DynamicMBean;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.ReflectionException;
import javax.management.RuntimeMBeanException;
import javax.management.openmbean.OpenDataException;
import javax.management.openmbean.OpenMBeanAttributeInfoSupport;
import javax.management.openmbean.OpenMBeanConstructorInfoSupport;
import javax.management.openmbean.OpenMBeanInfoSupport;
import javax.management.openmbean.OpenMBeanOperationInfoSupport;
import javax.management.openmbean.SimpleType;


/**
 * This class describes the management interface exposed for
 * Traffic Management in the EAS.
 *
 * @author erafope
 * @reviewed ehswolm 2007-feb-23
 * @reviewed ehswolm 2007-mar-14
 *
 */
public class TmSpaMBean implements DynamicMBean {
    /** The Logger instance for this class. */
    private Logger logger = Logger.getLogger("SipContainer");

    /** The name of this class. */
    private String className = this.getClass().getName();

    /** The description of this class. */
    private String description = "Traffic Management";

    /** The management interface exposed by this MBean */
    private OpenMBeanInfoSupport mbeanInfo;

    /** Holds the preference node(s) for Traffic Management */
    private Preferences tmPrefNode = null;
    private Preferences dnsPrefNode = null;
    private Preferences tmTimerPrefNode = null;

    /**
      * This method constructs the management interface to be exposed
      * by this MBean.
      *
      * @throws RuntimeMBeanException the exception wrapped into this object.
      */
    public TmSpaMBean() {
        super();

        /* Log creation started for this MBean. */
        if (logger.isLoggable(Level.INFO)) {
            logger.log(Level.INFO,
                "Creation of the " + description + " has started");
        }

        try {
            /* Call the method constructing the management interface */
            setMBeanInfo(className, description);

            /* Call the method retrieving the preference node */
            tmPrefNode = Preferences.userRoot().node("/com/ericsson/eas/tm");
            dnsPrefNode = Preferences.userRoot()
                                     .node("/com/ericsson/eas/oam/dnsmanagement");
            tmTimerPrefNode = Preferences.userRoot()
                                         .node("/com/ericsson/eas/tm/timer");
        } catch (Exception e) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Failed to create the " + description + " due to: " +
                    e.getCause() + ".");
            }

            /* Pass on the exception to the SPA console. */
            throw new RuntimeMBeanException(new RuntimeException(e),
                "Failed to create the " + this.getClass().getName() + " MBean");
        }
    }

    /**
     * This method constructs the management interface
     * exposed by this MBean.
     *
     * @param className
     * @param description
     */
    private void setMBeanInfo(String className, String description)
        throws OpenDataException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method setMBeanInfo has been called");
        }

        ArrayList<Object> arrayList = new ArrayList<Object>();

        /* Define the 'DefaultTCPTransport' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.DEFAULT_TCP_TRANSPORT,
                "Determines whether or not the transport parameter will be " +
                "inserted into the Uniform Resource Identifiers (URIs) of " +
                "Contact and Record-Route headers, that EAS inserts into " +
                "messages.", SimpleType.BOOLEAN, true, // isReadable
                false, // isWritable
                false, // isIs, i.e. a boolean
                TmSpaConfig.DEFAULT_TCP_TRANSPORT_DEF));

        /* Define the 'JmxDefaultPort' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.JMX_DEFAULT_PORT,
                "Defines the port on which the RMI registry will run.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.JMX_DEFAULT_PORT_DEF,
                TmSpaConfig.JMX_DEFAULT_PORT_MIN,
                TmSpaConfig.JMX_DEFAULT_PORT_MAX));

        /* Define the 'ThreadPoolSize' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.THREAD_POOL_SIZE,
                "Defines the initial number of worker threads.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.THREAD_POOL_SIZE_DEF,
                TmSpaConfig.THREAD_POOL_SIZE_MIN,
                TmSpaConfig.THREAD_POOL_SIZE_MAX));

        /* Define the 'ErrorResponseEnabled' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.ERROR_RESPONSE_ENABLED,
                "Determines whether or not a bad request should result in error response 400 Bad Request. If not, the SIP stack just drops the request, that is, the client does not get a SIP response.",
                SimpleType.BOOLEAN, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.ERROR_RESPONSE_ENABLED_DEF));

        /* Define the 'ApplicationDispatcherLoopInternal' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.APPLICATION_DISPATCHER_LOOP_INTERNAL,
                "Determines if the request will be handled internally by " +
                "the container or sent out to the network.",
                SimpleType.BOOLEAN, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.APPLICATION_DISPATCHER_LOOP_INTERNAL_DEF));

        /* Define the 'EnumTopDomain' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.ENUM_TOP_DOMAIN,
                "Defines the topdomain to be used when translating a " +
                "global telephone number to a ENUM domain name.",
                SimpleType.STRING, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.ENUM_TOP_DOMAIN_DEF));

        /* Define the 'DnsCacheSize' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.DNS_CACHE_SIZE, "Defines the dns cache size.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.DNS_CACHE_SIZE_DEF));

        /* Define the 'DnsServerList' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.DNS_SERVER_LIST,
                "Defines a comma separated list of DNS " +
                "servers that is used when querying external domains.",
                SimpleType.STRING, true, // isReadable
                false, // isWritable
                false, // isIs
                ""));

        /* Define the 'ProxyPoolSize' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.PROXY_POOL_SIZE,
                "Defines the HTTP proxy connection pool size.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.PROXY_POOL_SIZE_DEF,
                TmSpaConfig.PROXY_POOL_SIZE_MIN, null));

        /* Define the 'ProxyRetries' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.PROXY_RETRIES,
                "Defines the number of retries the HTTP proxy " +
                "performs before considering the proxy attempt to have failed.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.PROXY_RETRIES_DEF, TmSpaConfig.PROXY_RETRIES_MIN,
                null));

        /* Define the 'ProxySocketTimeout' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.PROXY_SOCKET_TIMEOUT,
                "Defines the maximum time, in milliseconds, " +
                "the HTTP proxy waits for a response before considering the proxy attempt to have failed.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.PROXY_SOCKET_TIMEOUT_DEF,
                TmSpaConfig.PROXY_SOCKET_TIMEOUT_MIN, null));

        /* Define the 'ProxyConnectionTimeout' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.PROXY_CONNECTION_TIMEOUT,
                "Defines the maximum time, in milliseconds, " +
                "the HTTP proxy tries to obtain a connection from a connection pool.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.PROXY_CONNECTION_TIMEOUT_DEF,
                TmSpaConfig.PROXY_CONNECTION_TIMEOUT_MIN, null));

        /* Define the 'OverloadRegulation' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.OVERLOAD_REGULATION,
                "Determines whether or not overload " +
                "regulation is enabled.", SimpleType.BOOLEAN, true, // isReadable
                true, // isWritable
                true, // isIs
                TmSpaConfig.OVERLOAD_REGULATION_DEF));

        /* Define the 'SampleRate' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.SAMPLE_RATE,
                "Defines the sample rate of updating the threshold levels.",
                SimpleType.INTEGER, true, // isReadable
                true, // isWritable
                false, // isIs
                TmSpaConfig.SAMPLE_RATE_DEF, TmSpaConfig.SAMPLE_RATE_MIN,
                TmSpaConfig.SAMPLE_RATE_MAX));

        /* Define the 'NumberOfSamples' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.NUMBER_OF_SAMPLES,
                "Defines the number of consequence samples that is needed " +
                "before threshold is raised.", SimpleType.INTEGER, true, // isReadable
                true, // isWritable
                false, // isIs
                TmSpaConfig.NUMBER_OF_SAMPLES_DEF,
                TmSpaConfig.NUMBER_OF_SAMPLES_MIN,
                TmSpaConfig.NUMBER_OF_SAMPLES_MAX));

        /* Define the 'HttpThreshold' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.HTTP_THRESHOLD,
                "Defines the threshold level of the CPU for HTTP requests.",
                SimpleType.INTEGER, true, // isReadable
                true, // isWritable
                false, // isIs
                TmSpaConfig.HTTP_THRESHOLD_DEF, TmSpaConfig.HTTP_THRESHOLD_MIN,
                TmSpaConfig.HTTP_THRESHOLD_MAX));

        /* Define the 'IrThreshold' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.IR_THRESHOLD,
                "Defines the threshold level of the CPU for SIP initial requests.",
                SimpleType.INTEGER, true, // isReadable
                true, // isWritable
                false, // isIs
                TmSpaConfig.IR_THRESHOLD_DEF, TmSpaConfig.IR_THRESHOLD_MIN,
                TmSpaConfig.IR_THRESHOLD_MAX));

        /* Define the 'SrThreshold' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.SR_THRESHOLD,
                "Defines the threshold level of the CPU for SIP subsequent requests.",
                SimpleType.INTEGER, true, // isReadable
                true, // isWritable
                false, // isIs
                TmSpaConfig.SR_THRESHOLD_DEF, TmSpaConfig.SR_THRESHOLD_MIN,
                TmSpaConfig.SR_THRESHOLD_MAX));

        /* Define the 'MmThreshold' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.MM_THRESHOLD,
                "Defines the threshold level of the CPU when all messages are dropped.",
                SimpleType.INTEGER, true, // isReadable
                true, // isWritable
                false, // isIs
                TmSpaConfig.MM_THRESHOLD_DEF, TmSpaConfig.MM_THRESHOLD_MIN,
                TmSpaConfig.MM_THRESHOLD_MAX));

        /* Define the 'timerT1' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(TmSpaConfig.TIMER_T1,
                "Defines the SIP Timer T1 in milliseconds.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.TIMER_T1_DEF));

        /* Define the 'timerT2' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(TmSpaConfig.TIMER_T2,
                "Defines the SIP Timer T2 in milliseconds.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.TIMER_T2_DEF));

        /* Define the 'timerT4' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(TmSpaConfig.TIMER_T4,
                "Defines the SIP Timer T4 in milliseconds.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.TIMER_T4_DEF));

        /* Define the 'SipLinkTimeout' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.SIP_LINK_TIMEOUT,
                "Defines timeout value in milliseconds for a single write operation of a sip link.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.SIP_LINK_TIMEOUT_DEF));

        /* Define the 'SipLinkTimeoutRetries' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.SIP_LINK_TIMEOUT_RETRIES,
                "Defines the number of retries to perform of a single write operation of a sip link.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.SIP_LINK_TIMEOUT_RETRIES_DEF));

        /* Define the 'SipLinkMaxThreadQueueLength' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.SIP_LINK_MAX_QUEUE_LENGTH,
                "Defines the maximum number simultaneous write/connect requests that can be waiting to write on a link.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.SIP_LINK_MAX_QUEUE_LENGTH_DEF));

        /* Define the 'SipLinkAliveTimeout' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.SIP_LINK_ALIVE_TIMEOUT,
                "Defines the duration (in seconds) an inactive SIP link is kept alive before the connection is torn down.",
                SimpleType.INTEGER, true, // isReadable
                false, // isWritable
                false, // isIs
                TmSpaConfig.SIP_LINK_ALIVE_TIMEOUT_DEF));

        /* Define the 'Eas503Disabled' attribute */
        arrayList.add(new OpenMBeanAttributeInfoSupport(
                TmSpaConfig.EAS_503_DISABLED,
                "Determines if the functionality described in RFC3263 4.3 Details of RFC 2782 Process is disabled.",
                SimpleType.BOOLEAN, true, // isReadable
                false, // isWritable
                true, // isIs
                TmSpaConfig.EAS_503_DISABLED_DEF));

        /* Declare the array of MBean attributes exposed for management. */
        OpenMBeanAttributeInfoSupport[] attributes = new OpenMBeanAttributeInfoSupport[arrayList.size()];

        /* Returns the array of the defined MBean attributes */
        arrayList.toArray(attributes);

        /* Construct the MBeanInfo object */
        mbeanInfo = new OpenMBeanInfoSupport(className, description,
                attributes, new OpenMBeanConstructorInfoSupport[0],
                new OpenMBeanOperationInfoSupport[0],
                new MBeanNotificationInfo[0]);

        /* Log the success of constructing the MBeanInfo object */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "MBeanInfo constructed for the " +
                this.mbeanInfo.getDescription() + " MBean");
        }
    }

    /**
     * This method returns the requested MBean attribute value.
     *
     * @param attributeName the name of the attribute to be returned.
     * @return the value of the attribute.
     * @throws AttributeNotFoundException Attribute not recognized.
     * @throws MBeanException Failed to read the attribute value.
     * @throws ReflectionException Reflection failed.
     * */
    public Object getAttribute(String attributeName)
        throws AttributeNotFoundException, MBeanException, ReflectionException {
        /* Log reading started for this attribute. */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "Reading started for attribute " + attributeName);
        }

        /* Check that the attribute exists in this MBean. */
        if (attributeName.equals(
                    TmSpaConfig.APPLICATION_DISPATCHER_LOOP_INTERNAL)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getApplicationDispatcherLoopInternal();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.DNS_CACHE_SIZE)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getDnsCacheSize();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.DNS_SERVER_LIST)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getDnsServerList();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.EAS_503_DISABLED)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getEas503Disabled();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.ENUM_TOP_DOMAIN)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getEnumTopDomain();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.HTTP_THRESHOLD)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getHttpThreshold();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.IR_THRESHOLD)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getIrThreshold();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.JMX_DEFAULT_PORT)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getJmxDefaultPort();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.MM_THRESHOLD)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getMmThreshold();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.NUMBER_OF_SAMPLES)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getNumberOfSamples();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.OVERLOAD_REGULATION)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getOverloadRegulation();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.PROXY_CONNECTION_TIMEOUT)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getProxyConnectionTimeout();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.PROXY_POOL_SIZE)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getProxyPoolSize();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.PROXY_RETRIES)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getProxyRetries();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.PROXY_SOCKET_TIMEOUT)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getProxySocketTimeout();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.SAMPLE_RATE)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getSampleRate();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.SIP_LINK_MAX_QUEUE_LENGTH)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getSipLinkMaxQueueLenght();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.SIP_LINK_TIMEOUT)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getSipLinkTimeout();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.SIP_LINK_ALIVE_TIMEOUT)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getSipLinkAliveTimeout();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.SIP_LINK_TIMEOUT_RETRIES)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getSipLinkTimeoutRetries();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.SR_THRESHOLD)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getSrThreshold();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.DEFAULT_TCP_TRANSPORT)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getDefaultTCPTransport();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.THREAD_POOL_SIZE)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getThreadPoolSize();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.ERROR_RESPONSE_ENABLED)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getErrorResponseEnabled();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.TIMER_T1)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getTimerT1();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.TIMER_T2)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getTimerT2();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else if (attributeName.equals(TmSpaConfig.TIMER_T4)) {
            /* Log calling the getter for this attribute. */
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Calling the getter for attribute " + attributeName);
            }

            try {
                /* Call the getter for this attribute. */
                return getTimerT4();
            } catch (Exception e) {
                /* Log failed to read the attribute value. */
                logger.log(Level.SEVERE,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");

                /* Pass on the exception to the SPA console. */
                throw new MBeanException(e,
                    "Failed to read the value of the " + attributeName +
                    " attribute in the " + this.mbeanInfo.getDescription() +
                    " MBean");
            }
        } else {
            /* Log attribute name not found. */
            logger.log(Level.SEVERE,
                "Cannot find the " + attributeName + " attribute in the " +
                description + " MBean");

            /* Pass on the exception to the SPA console. */
            throw new AttributeNotFoundException("Cannot find the " +
                attributeName + " attribute in the " +
                this.mbeanInfo.getDescription() + " MBean");
        }
    }

    /**
     * This method returns the list of all attributes exposed by this MBean.
     *
     * Note: This method is not called by the SPA console.
     *
     * @return attributeList the list of attributes exposed by this MBean
     */
    public AttributeList getAttributes(String[] attributeNames) {
        /* Log reading started for this attribute list. */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Reading started for this of attribute list");
        }

        /* Holds the attribute list returned for this MBean. */
        AttributeList attributeList = new AttributeList();

        /*
         * Check if the attribute list is null.
         * Note: This check is done to avoid a NullPointerException later on.
         */
        if (attributeNames == null) {
            /* Log attribute list set to null. */
            logger.log(Level.SEVERE, "Cannot read an attribute list set to null");

            /* Return an empty attribute list. */
            return attributeList;
        }

        /* Check if the attribute list is empty. */
        if (attributeNames.length == 0) {
            /* Log attribute list empty. */
            logger.log(Level.SEVERE, "Cannot read an empty attribute list");

            /* Return an empty attribute list. */
            return attributeList;
        }

        /* Construct the resulting attribute list. */
        for (int i = 0; i < attributeNames.length; i++) {
            try {
                Object value = getAttribute((String) attributeNames[i]);
                attributeList.add(new Attribute(attributeNames[i], value));
            } catch (Exception e) {
                /*
                 * We end up here if one of the three checked exceptions
                 * are thrown. Since we want the resultList to contain
                 * all successfully obtained attributes we continue to
                 * fill it although an attribute threw an exception.
                 */
            }
        }

        /* Log reading ended for this attribute list. */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Reading ended for this of attribute list");
        }

        /* Return the resulting attribute list */
        return attributeList;
    }

    /**
     * Provides the exposed attributes and actions
     * for this MBean.
     *
     * @return the MBeanInfo for this MBean
     */
    public MBeanInfo getMBeanInfo() {
        /* Log reading of MBeanInfo. */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "MBeanInfo read for the " + this.mbeanInfo.getDescription() +
                " MBean");
        }

        return this.mbeanInfo;
    }

    /**
     * This method invokes an operation on this MBean.
     *
     * @param actionName The name of the operation
     * @param params The parameters for the operation
     * @param signature The signature of the operation
     *
     * @throws MBeanException
     * @throws ReflectionException
     */
    public Object invoke(String actionName, Object[] params, String[] signature)
        throws MBeanException, ReflectionException {
        /* Log Unknown operation for this MBean. */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method invoke has been called.");
        }

        logger.log(Level.SEVERE, "Unknown operation for this MBean");

        /* Pass on the exception to the SPA console. */
        throw new MBeanException(new NoSuchMethodException(actionName),
            "Unknown operation for this MBean");
    }

    /**
     * This method validates and writes the value for an attribute.
     *
     * Note: This method is not called by the SPA console directly.
     *
     * @param  attribute The attribute to set
     * @throws AttributeNotFoundException       If attribute is null or not found in mbean.
     * @throws InvalidAttributeValueException   If attribute is invalid.
     * @throws MBeanException                   Represents "user defined" exceptions
     *                                          thrown by MBean methods in the agent.
     * @throws ReflectionException              If reflection is unsuccessful.
     *
     */
    public void setAttribute(Attribute attribute)
        throws AttributeNotFoundException, InvalidAttributeValueException,
            MBeanException, ReflectionException {
        /* Log writing started for this attribute. */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "Writing started for attribute " + attribute.getName());
        }

        try {
            if (attribute.getName().equals(TmSpaConfig.HTTP_THRESHOLD)) {
                setHttpThreshold(attribute.getValue());
            } else if (attribute.getName().equals(TmSpaConfig.IR_THRESHOLD)) {
                /* Call the setter for the attribute. */
                setIrThreshold(attribute.getValue());
            } else if (attribute.getName().equals(TmSpaConfig.MM_THRESHOLD)) {
                /* Call the setter for the attribute. */
                setMmThreshold(attribute.getValue());
            } else if (attribute.getName().equals(TmSpaConfig.NUMBER_OF_SAMPLES)) {
                /* Call the setter for the attribute. */
                setNumberOfSamples(attribute.getValue());
            } else if (attribute.getName().equals(TmSpaConfig.SAMPLE_RATE)) {
                /* Call the setter for the attribute. */
                setSampleRate(attribute.getValue());
            } else if (attribute.getName().equals(TmSpaConfig.SR_THRESHOLD)) {
                /* Call the setter for the attribute. */
                setSrThreshold(attribute.getValue());
            } else if (attribute.getName()
                                    .equals(TmSpaConfig.OVERLOAD_REGULATION)) {
                /* Call the setter for the attribute. */
                setOverloadRegulation(attribute.getValue());
            } else {
                logger.log(Level.SEVERE,
                    "Cannot find the " + attribute + " attribute in the " +
                    this.mbeanInfo.getDescription() + " Mbean");
                throw new AttributeNotFoundException("Cannot find the " +
                    attribute + " attribute in the " +
                    this.mbeanInfo.getDescription() + " Mbean");
            }
        } catch (Exception e) {
            /* Log failed to read the attribute value. */
            logger.log(Level.SEVERE,
                "Failed to write the value of the " + attribute.getName() +
                " attribute in the " + this.mbeanInfo.getDescription() +
                " MBean");

            /* Pass on the exception to the SPA console. */
            throw new MBeanException(e,
                "Failed to write the value of the " + attribute.getName() +
                " attribute in the " + this.mbeanInfo.getDescription() +
                " MBean");
        }

        /* Log writing ended for this attribute. */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "Writing ended for the " + attribute.getName() +
                " attribute. Value set to: " + (String) attribute.getValue());
        }
    }

    /**
     * This method validates and sets the values for a list of
     * attributes.
     *
     * @param attributeList The attribute list to set
     * @return resultList Teh returned attribute list
     */
    public AttributeList setAttributes(AttributeList attributeList) {
        /* Log writing started for this attribute list. */
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Writing started for the attribute list");
        }

        /* Holds the attribute list returned for this MBean. */
        AttributeList resultList = new AttributeList();

        /*
         * Check if the attribute list is null.
         * Note: This check is done to avoid a NullPointerException later on.
         */
        if (attributeList == null) {
            /* Log attribute list set to null. */
            logger.log(Level.SEVERE,
                "Cannot write an attribute list set to null");

            /* Pass on an exception to the SPA console. */
            throw new RuntimeException(
                "Cannot write an attribute list set to null");
        }

        /* Check if the attribute list is empty. */
        if (attributeList.isEmpty()) {
            /* Log attribute list is empty. */
            logger.log(Level.SEVERE, "Cannot write an empty attribute list");

            /* Return an empty attribute list. */
            return resultList;
        }

        /* Write the values of the attributes in the attribute list. */
        Iterator iterator = attributeList.iterator();

        while (iterator.hasNext()) {
            /* Get an attribute from the list of attributes. */
            Attribute attribute = (Attribute) iterator.next();

            try {
                /* Call the setter for an attribute. */
                setAttribute(attribute);

                String name = attribute.getName();
                Object value = attribute.getValue();

                resultList.add(new Attribute(name, value));
            } catch (Exception e) {
                /* Log writing failed for this attribute list. */
                logger.log(Level.SEVERE,
                    "Failed to write attribute list in the " +
                    this.mbeanInfo.getDescription() + " MBean");

                /* Pass on the exception to the SPA console. */
                throw new RuntimeException(e.getCause().getMessage());
            }
        }

        /* Return the resulting attribute list */
        return resultList;
    }

    /**
     * This method reads the 'ApplicationDispatcherLoopInternal' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'OverloadRegulation' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Boolean getApplicationDispatcherLoopInternal()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "Method getApplicationDispatcherLoopInternal called");
        }

        String value = tmPrefNode.get(TmSpaConfig.APPLICATION_DISPATCHER_LOOP_INTERNAL,
                null);

        if (value == null) {
            return null;
        }

        return new Boolean(value);
    }

    /**
     * This method reads the 'DnsCacheSize' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'DnsCacheSize' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getDnsCacheSize()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getDnsCacheSize() called");
        }

        String value = dnsPrefNode.get(TmSpaConfig.DNS_CACHE_SIZE, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'DnsServerList' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'DnsServerList' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private String getDnsServerList()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getDnsServerList() called");
        }

        String value = dnsPrefNode.get(TmSpaConfig.DNS_SERVER_LIST, null);

        if (value == null) {
            return null;
        }

        return value;
    }

    /**
     * This method reads the 'Eas503Disabled' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'Eas503Disabled' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Boolean getEas503Disabled()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getEas503Disabled called");
        }

        String value = tmPrefNode.get(TmSpaConfig.EAS_503_DISABLED, null);

        if (value == null) {
            return null;
        }

        return new Boolean(value);
    }

    /**
     * This method reads the 'EnumTopDomain' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'EnumTopDomain' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private String getEnumTopDomain()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getEnumTopDomain() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.ENUM_TOP_DOMAIN, null);

        if (value == null) {
            return null;
        }

        return value;
    }

    /**
     * This method reads the 'HttpThreshold' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'HttpThreshold' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getHttpThreshold()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getHttpThreshold() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.HTTP_THRESHOLD, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'IrThreshold' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'IrThreshold' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getIrThreshold()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getIrThreshold() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.IR_THRESHOLD, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'JmxDefaultPort' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'JmxDefaultPort' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getJmxDefaultPort()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getJmxDefaultPort() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.JMX_DEFAULT_PORT, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'MmThreshold' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'MmThreshold' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getMmThreshold()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getMmThreshold() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.MM_THRESHOLD, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'NumberOfSamples' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'MmThreshold' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getNumberOfSamples()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getNumberOfSamples() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.NUMBER_OF_SAMPLES, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'OverloadRegulation' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'OverloadRegulation' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Boolean getOverloadRegulation()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getOverloadRegulation called");
        }

        String value = tmPrefNode.get(TmSpaConfig.OVERLOAD_REGULATION, null);

        if (value == null) {
            return null;
        }

        return new Boolean(value);
    }

    /**
     * This method reads the 'ProxyConnectionTimeout' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'ProxyConnectionTimeout' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getProxyConnectionTimeout()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getProxyConnectionTimeout() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.PROXY_CONNECTION_TIMEOUT, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'ProxyPoolSize' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'ProxyPoolSize' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getProxyPoolSize()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getProxyPoolSize() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.PROXY_POOL_SIZE, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'ProxyRetries' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'ProxyRetries' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getProxyRetries()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getProxyRetries() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.PROXY_RETRIES, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'ProxySocketTimeout' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'ProxySocketTimeout' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getProxySocketTimeout()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getProxySocketTimeout() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.PROXY_SOCKET_TIMEOUT, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'SampleRate' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'SampleRate' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getSampleRate()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getSampleRate() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.SAMPLE_RATE, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'SipLinkMaxQueueLenght' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'SipLinkMaxQueueLenght' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getSipLinkMaxQueueLenght()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getSipLinkMaxQueueLenght() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.SIP_LINK_MAX_QUEUE_LENGTH,
                null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'SipLinkTimeout' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'SipLinkTimeout' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getSipLinkTimeout()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getSipLinkTimeout() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.SIP_LINK_TIMEOUT, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'SipLinkTimeoutRetries' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'SipLinkTimeoutRetries' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getSipLinkTimeoutRetries()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getSipLinkTimeoutRetries() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.SIP_LINK_TIMEOUT_RETRIES, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'SipLinkAliveTimeout' attribute from the
     * preference node for Traffic Management function.
     *
     * @return the 'SipLinkAliveTimeout' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     * @throws NumberFormatException
     */
    private Integer getSipLinkAliveTimeout()
        throws IllegalStateException, NullPointerException {
        String value = tmPrefNode.get(TmSpaConfig.SIP_LINK_ALIVE_TIMEOUT, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'SrThreshold' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'SrThreshold' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getSrThreshold()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getSrThreshold() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.SR_THRESHOLD, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'DefaultTCPTransport' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'OverloadRegulation' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Object getDefaultTCPTransport() {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getDefaultTCPTransport called");
        }

        String value = tmPrefNode.get(TmSpaConfig.DEFAULT_TCP_TRANSPORT, null);

        if (value == null) {
            return null;
        }

        return new Boolean(value);
    }

    /**
     * This method reads the 'ThreadPoolSize' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'ThreadPoolSize' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getThreadPoolSize()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getThreadPoolSize() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.THREAD_POOL_SIZE, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'ThreadPoolSize' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'ThreadPoolSize' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Boolean getErrorResponseEnabled()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getErrorResponseEnabled() called");
        }

        String value = tmPrefNode.get(TmSpaConfig.ERROR_RESPONSE_ENABLED, null);

        if (value == null) {
            return null;
        }

        return new Boolean(value);
    }

    /**
     * This method reads the 'TimerT1' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'TimerT1' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getTimerT1()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getTimerT1() called");
        }

        String value = tmTimerPrefNode.get(TmSpaConfig.TIMER_T1, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'TimerT2' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'TimerT2' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getTimerT2()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getTimerT2() called");
        }

        String value = tmTimerPrefNode.get(TmSpaConfig.TIMER_T2, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method reads the 'TimerT4' attribute from
     * the preference node for Traffic Management.
     *
     * @return the 'TimerT4' attribute
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private Integer getTimerT4()
        throws IllegalStateException, NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method getTimerT4() called");
        }

        String value = tmTimerPrefNode.get(TmSpaConfig.TIMER_T4, null);

        if (value == null) {
            return null;
        }

        return new Integer(value);
    }

    /**
     * This method writes the value of the 'HttpThreshold'
     * attribute to the preference node for Traffic Management.
     *
     * @param attributeValue
     * @throws IllegalArgumentException
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private void setHttpThreshold(Object attributeValue)
        throws IllegalArgumentException, IllegalStateException,
            NullPointerException {
        tmPrefNode.put(TmSpaConfig.HTTP_THRESHOLD, (String) attributeValue);
    }

    /**
     * This method writes the value of the 'IrThreshold'
     * attribute to the preference node for Traffic Management.
     *
     * @param attributeValue
     * @throws IllegalArgumentException
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private void setIrThreshold(Object attributeValue)
        throws IllegalArgumentException, IllegalStateException,
            NullPointerException {
        tmPrefNode.put(TmSpaConfig.IR_THRESHOLD, (String) attributeValue);
    }

    /**
     * This method writes the value of the 'MmThreshold'
     * attribute to the preference node for Traffic Management.
     *
     * @param attributeValue
     * @throws IllegalArgumentException
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private void setMmThreshold(Object attributeValue)
        throws IllegalArgumentException, IllegalStateException,
            NullPointerException {
        tmPrefNode.put(TmSpaConfig.MM_THRESHOLD, (String) attributeValue);
    }

    /**
     * This method writes the value of the 'setNumberOfSamples'
     * attribute to the preference node for Traffic Management.
     *
     * @param attributeValue
     * @throws IllegalArgumentException
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private void setNumberOfSamples(Object attributeValue)
        throws IllegalArgumentException, IllegalStateException,
            NullPointerException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Method setNumberOfSamples() called");
            logger.log(Level.FINE, "attributeValue=" + attributeValue);
        }

        tmPrefNode.put(TmSpaConfig.NUMBER_OF_SAMPLES, (String) attributeValue);
    }

    /**
     * This method writes the value of the 'OverloadRegulation'
     * attribute to the preference node for Traffic Management.
     *
     * @param attributeValue
     * @throws IllegalArgumentException
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private void setOverloadRegulation(Object attributeValue)
        throws IllegalArgumentException, IllegalStateException,
            NullPointerException {
        tmPrefNode.put(TmSpaConfig.OVERLOAD_REGULATION, (String) attributeValue);
    }

    /**
     * This method writes the value of the 'setSampleRate'
     * attribute to the preference node for Traffic Management.
     *
     * @param attributeValue
     * @throws IllegalArgumentException
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private void setSampleRate(Object attributeValue)
        throws IllegalArgumentException, IllegalStateException,
            NullPointerException {
        tmPrefNode.put(TmSpaConfig.SAMPLE_RATE, (String) attributeValue);
    }

    /**
     * This method writes the value of the 'setSrThreshold'
     * attribute to the preference node for Traffic Management.
     *
     * @param attributeValue
     * @throws IllegalArgumentException
     * @throws IllegalStateException
     * @throws NullPointerException
     */
    private void setSrThreshold(Object attributeValue)
        throws IllegalArgumentException, IllegalStateException,
            NullPointerException {
        tmPrefNode.put(TmSpaConfig.SR_THRESHOLD, (String) attributeValue);
    }
}
