/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.dd;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EventListener;
import java.util.Iterator;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;

import javax.servlet.ServletContextListener;
import javax.servlet.sip.SipApplicationSessionListener;
import javax.servlet.sip.SipErrorListener;
import javax.servlet.sip.SipServlet;
import javax.servlet.sip.SipSessionAttributeListener;
import javax.servlet.sip.SipSessionListener;
import javax.servlet.sip.TimerListener;


public class SipApplicationListeners {
    private static final Logger logger = Logger.getLogger("SipContainer");

    /**
     * A SipApplicationSession has been created, destroyed, or has timed out.
     */
    private ArrayList<SipApplicationSessionListener> sipApplicationSessionListeners =
        new ArrayList<SipApplicationSessionListener>();

    /**
     * A SipSession has been created or destroyed.
     */
    private ArrayList<SipSessionListener> sipSessionListeners = new ArrayList<SipSessionListener>();

    /**
     * Attributes on the servlet context have been added, removed or replaced.
     */
    private ArrayList<SipSessionAttributeListener> sipSessionAttributeListeners = new ArrayList<SipSessionAttributeListener>();

    /**
     * notification that an expected ACK or PRACK was not received.
     */
    private ArrayList<SipErrorListener> sipErrorListeners = new ArrayList<SipErrorListener>();

    /**
     * Timer listener for this application
     */
    private TimerListener timerListener = null;

    /**
     * ServletContext listener for this application
     */
    private ArrayList<ServletContextListener> servletContextListener = new ArrayList<ServletContextListener>();
    private ArrayList<String> sipServletsWithListeners = new ArrayList<String>();

    /**
     * This will build and manage a list of session listeners for a given Sip
     * Application
     *
     * @param listenerList
     *        A collection of class name
     * @param classLoader
     *        The class loader to use to instantiate the listeners
     */
    public SipApplicationListeners(Collection<String> listenerList,
        ClassLoader classLoader) {
        // Instantiate all the listeners
        Iterator<String> itr = listenerList.iterator();

        while (itr.hasNext()) {
            String className = (String) itr.next();

            try {
                EventListener listener = (EventListener) classLoader.loadClass(className)
                                                                    .newInstance();

                if (listener instanceof SipServlet) {
                    sipServletsWithListeners.add(className);

                    continue;
                }

                if (listener instanceof SipApplicationSessionListener) {
                    addSipApplicationSessionListener((SipApplicationSessionListener) listener);
                }

                if (listener instanceof SipSessionListener) {
                    addSipSessionListener((SipSessionListener) listener);
                }

                if (listener instanceof SipSessionAttributeListener) {
                    addSipSessionAttributeListener((SipSessionAttributeListener) listener);
                }

                if (listener instanceof TimerListener) {
                    setTimerListener((TimerListener) listener);
                }

                if (listener instanceof SipErrorListener) {
                    addSipErrorListener((SipErrorListener) listener);
                }

                if (listener instanceof ServletContextListener) {
                    addServletContextListener((ServletContextListener) listener);
                }
            } catch (Exception e) {
                logger.log(Level.SEVERE,
                    "Cannot instantiate listener class " + className, e);
            }
        }
    }

    public ArrayList<String> getSipServletsWithListeners() {
        return sipServletsWithListeners;
    }

    /**
     * @return Returns the sipApplicationSessionListeners.
     */
    public ArrayList<SipApplicationSessionListener> getSipApplicationSessionListeners() {
        return sipApplicationSessionListeners;
    }

    /**
     * Add a Sip Application Session Listener
     * @param listener
     */
    public void addSipApplicationSessionListener(
        SipApplicationSessionListener listener) {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "registering SipApplicationSessionListener: " +
                listener.getClass().getName());
        }

        sipApplicationSessionListeners.add(listener);
    }

    /**
     * @return Returns the sipSessionAttributeListeners.
     */
    public ArrayList<SipSessionAttributeListener> getSipSessionAttributeListeners() {
        return sipSessionAttributeListeners;
    }

    /**
     * Add a sip session attribute listener
     * @param listener
     */
    public void addSipSessionAttributeListener(
        SipSessionAttributeListener listener) {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "registering SipSessionAttributeListener: " +
                listener.getClass().getName());
        }

        sipSessionAttributeListeners.add(listener);
    }

    /**
     * @return Returns the sipSessionListeners.
     */
    public ArrayList<SipSessionListener> getSipSessionListeners() {
        return sipSessionListeners;
    }

    /**
     * Add a sip session listener
     * @param listener
     */
    public void addSipSessionListener(SipSessionListener listener) {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "registering SipSessionListener: " +
                listener.getClass().getName());
        }

        sipSessionListeners.add(listener);
    }

    /**
     * @return Returns the timer listener.
     */
    public TimerListener getTimerListener() {
        return timerListener;
    }

    /**
     * Add a timer listener
     * @param listener
     */
    public void setTimerListener(TimerListener listener) {
        if (timerListener != null) {
            logger.log(Level.WARNING,
                "ignoring TimerListener: " + listener.getClass().getName());

            return;
        }

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "registering TimerListener: " + listener.getClass().getName());
        }

        timerListener = listener;
    }

    /**
     * @return Returns the sip error listeners.
     */
    public ArrayList<SipErrorListener> getSipErrorListeners() {
        return sipErrorListeners;
    }

    /**
     * Add a sip error listener
     * @param listener
     */
    public void addSipErrorListener(SipErrorListener listener) {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "registering SipErrorListener: " +
                listener.getClass().getName());
        }

        sipErrorListeners.add(listener);
    }

    /**
     * Add a ServletContextListener to list.
     * @param listener The listener to add.
     */
    public void addServletContextListener(ServletContextListener listener) {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE,
                "registering ServletContextListener: " +
                listener.getClass().getName());
        }

        servletContextListener.add(listener);
    }

    /**
     * The method will return an ArrayList of the registered ServletContextListener.
     * @return A list with the all context Listener.
     */
    public ArrayList<ServletContextListener> getServletContextListener() {
        return servletContextListener;
    }

    /**
     * Empty vectors to allow garbage collection
     */
    public void clean() {
        sipApplicationSessionListeners.clear();
        sipSessionListeners.clear();
        sipSessionAttributeListeners.clear();
        sipErrorListeners.clear();
        servletContextListener.clear();
        timerListener = null;
    }
}
