/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;

import java.util.Collections;
import java.util.Iterator;

import javax.servlet.sip.Address;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipFactory;
import javax.servlet.sip.URI;


/**
 * @author ekrigro
 */
public class AddressImpl extends ParameterableImpl implements Address,
    Serializable {
    private static final long serialVersionUID = 3761123855715873593L;
    static String DUPLICATE_PARAM = "The parsed Name Address has two parameter with the same name : ";
    public static final String Q_PARAM = "q";
    public static final String EXPIRES_PARAM = "expires";
    public static final String TAG_PARAM = "tag";
    transient private static SipFactory sf = SipFactoryImpl.getInstance();
    private String _displayName;
    private URI _uri;

    // System headers Address cannot be modified.
    private boolean _readOnly = false;
    private boolean _wildcard = false;

    private AddressImpl() {
    }

    public AddressImpl(String na) throws ServletParseException {
        if (na.trim().equals("*")) {
            _wildcard = true;
        } else {
            byte[] bytes = null;

            try {
                bytes = na.getBytes(SipFactoryImpl.SIP_CHARSET);
            } catch (UnsupportedEncodingException e) {
            }

            parse(bytes, 0, bytes.length);
        }
    }

    public AddressImpl(URI uri) {
        _uri = uri;
    }

    private void writeObject(ObjectOutputStream stream)
        throws IOException {
        try {
            stream.defaultWriteObject();
        } catch (Exception ignore) {
        }
    }

    private void readObject(ObjectInputStream stream) throws IOException {
        try {
            // stream.registerValidation(this, 0);
            stream.defaultReadObject();
            sf = SipFactoryImpl.getInstance();
        } catch (Exception ignore) {
            ignore.printStackTrace();
        }
    }

    // From: "A. G. Bell" <sip:agb@bell-telephone.com> ;tag=a48s
    // From: sip:+12125551212@server.phone2net.com;tag=887s
    // f: Anonymous <sip:c8oqz84zk7z@privacy.org>;tag=hyh8
    private void parse(byte[] bytes, int offset, int length)
        throws ServletParseException {
        int leftDquoteIndex = -1;
        int rightDquoteIndex = -1;
        int laquotIndex = -1;
        int raquotIndex = -1;
        int semiIndex = -1;
        int quoteCnt = 0;
        int endParamIndex = length;

        for (int i = offset; i < length; i++) {
            if (bytes[i] == '"') {
                quoteCnt++;

                if (laquotIndex == -1) {
                    if (leftDquoteIndex == -1) {
                        leftDquoteIndex = i;
                    } else if (rightDquoteIndex < 0) {
                        rightDquoteIndex = i;
                    }
                }
            } else if ((laquotIndex == -1) && (bytes[i] == '<') &&
                    ((quoteCnt % 2) == 0)) {
                laquotIndex = i;
            } else if ((raquotIndex == -1) && (bytes[i] == '>') &&
                    ((quoteCnt % 2) == 0)) {
                raquotIndex = i;
            } else if ((bytes[i] == ';') && (semiIndex < 0) &&
                    ((quoteCnt % 2) == 0)) {
                if (laquotIndex >= 0) {
                    if ((raquotIndex > 0) && (i > raquotIndex)) // Set only if there is
                                                                // a closing >
                     {
                        semiIndex = i;
                    }
                } else {
                    semiIndex = i; // we don't have any <>
                }
            }
        }

        if (((laquotIndex == -1) && (raquotIndex > 0)) ||
                ((laquotIndex > 0) && (raquotIndex == -1))) {
            throw new ServletParseException("invalid nr of > or <");
        }

        // TODO - Maybe a way to remove the trim()
        try {
            // Taking care of Display name
            if (rightDquoteIndex > -1) {
                // There is a displayname in quotes
                _displayName = new String(bytes, leftDquoteIndex + 1,
                        rightDquoteIndex - (leftDquoteIndex + 1),
                        SipFactoryImpl.SIP_CHARSET);
                offset = rightDquoteIndex + 1;
            } else if (laquotIndex > -1) {
                // There could be a display name without quotes
                _displayName = new String(bytes, offset, laquotIndex - offset,
                        SipFactoryImpl.SIP_CHARSET).trim();
                offset = laquotIndex;
            }

            if (laquotIndex > -1) {
                // The URI is behind <>
                // TODO send the buffer without doing string
                String uri = new String(bytes, laquotIndex + 1,
                        raquotIndex - (laquotIndex + 1),
                        SipFactoryImpl.SIP_CHARSET);
                // _uri = new ()
                _uri = sf.createURI(uri);

                // After there are possible NA parameters
            } else {
                // There is only a URI
                // TODO send the buffer without doing string
                int endIndex = semiIndex;

                if (endIndex < 0) {
                    endIndex = length;
                }

                String uri = new String(bytes, offset, endIndex - offset,
                        SipFactoryImpl.SIP_CHARSET).trim();
                // _uri = new ()
                _uri = sf.createURI(uri);

                // After there are possible NA parameters
            }

            offset = semiIndex + 1;

            if (semiIndex > 0) {
                int numBytes = endParamIndex - semiIndex;
                setParameterByteMap(bytes, semiIndex, numBytes, ';');
            }
        } catch (UnsupportedEncodingException uee) {
        } // C.C.L.
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#getDisplayName()
     */
    public String getDisplayName() {
        return _displayName;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#setDisplayName(java.lang.String)
     */

    // If should be pretty print no "" should be there if it's only one word
    public void setDisplayName(String dn) {
        if (_wildcard) {
            throw new IllegalStateException(
                "Cannot set display name on wildcard address");
        }

        if (_readOnly) {
            throw new IllegalArgumentException(
                "You cannot modify a system header.");
        }

        _displayName = dn;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#getURI()
     */
    public URI getURI() {
        if (_wildcard) {
            (new Throwable()).printStackTrace();

            return null;
        } else {
            return _uri;
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#setURI(javax.servlet.sip.URI)
     */
    public void setURI(URI uri) {
        if (_wildcard) {
            throw new IllegalStateException(
                "Cannot set URI on wildcard address");
        }

        if (_readOnly) {
            throw new IllegalArgumentException(
                "You cannot modify a system header.");
        }

        _uri = uri;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#setParameter(java.lang.String,
     *      java.lang.String)
     */
    @Override
    public void setParameter(String param, String value) {
        if (_wildcard) {
            throw new IllegalStateException(
                "Cannot set parameter on wildcard address");
        }

        if (_readOnly) {
            throw new IllegalArgumentException(
                "You cannot modify a system header.");
        }

        super.setParameter(param, value);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#removeParameter(java.lang.String)
     */
    @Override
    public void removeParameter(String param) {
        if (_wildcard) {
            throw new IllegalStateException(
                "Cannot set parameter on wildcard address");
        }

        if (_readOnly) {
            throw new IllegalArgumentException(
                "You cannot modify a system header.");
        }

        super.removeParameter(param);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#isWildcard()
     */
    public boolean isWildcard() {
        return _wildcard;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#getQ()
     */
    public float getQ() {
        float q = (float) -1.0;
        String value = getParameter(Q_PARAM);

        if (value != null) {
            try {
                q = Float.parseFloat(value);
            } catch (Exception e) {
            } // Bad luck :-)
        }

        return q;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#setQ(float)
     */
    public void setQ(float value) {
        if (value < 0) {
            super.removeParameter(Q_PARAM);
        } else {
            setParameter(Q_PARAM, String.valueOf(value));
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#getExpires()
     */
    public int getExpires() {
        int expires = -1;
        String value = getParameter(EXPIRES_PARAM);

        if (value != null) {
            try {
                expires = Integer.parseInt(value);
            } catch (Exception e) {
            } // Bad luck :-)
        }

        return expires;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.Address#setExpires(int)
     */
    public void setExpires(int value) {
        if (value < 0) {
            super.removeParameter(EXPIRES_PARAM);
        } else {
            setParameter(EXPIRES_PARAM, String.valueOf(value));
        }
    }

    /**
     * Returns a clone of this AddressImpl instance. The clone will be equal to
     * its original, except that it will have no tag parameter. This is the
     * implementation of the javax.servlet.sip.Address.clone() method. Also it
     * will not have the readOnly property, since an application may wish to
     * modify the clone even if it was created from a System Header address
     * object.
     */
    public Object clone() {
        return clone(false, false);
    }

    /**
     * Returns a clone of this AddressImpl instance. The clone will be equal to
     * its original, except that if copyTag is false it will have no tag
     * parameter.
     */
    public Object clone(boolean copyTag, boolean copyReadOnly) {
        AddressImpl newAddress = (AddressImpl) super.clone();
        // AddressImpl newAddress = (ParameterableImpl)super.clone();
        newAddress._displayName = _displayName;
        newAddress._uri = _uri;
        newAddress._wildcard = _wildcard;

        if (copyReadOnly) {
            newAddress._readOnly = _readOnly;
        }

        return newAddress;
    }

    /* TODO : Have a look at the implementation  */
    public boolean equals(Object o) {
        if (o == null) {
            return false;
        }

        if (this == o) {
            return true;
        }

        if (!(o instanceof AddressImpl)) {
            return false;
        }

        AddressImpl a = (AddressImpl) o;

        if (!_displayName.equals(a.getDisplayName())) {
            return false;
        }

        if (!_uri.equals(a.getURI())) {
            return false;
        }

        if (_wildcard != a.isWildcard()) {
            return false;
        }

        return true;
    }

    public boolean isReadOnly() {
        return _readOnly;
    }

    public void setReadOnly(boolean writeable) {
        _readOnly = writeable;
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#toString()
     */
    public String toString() {
        if (_wildcard) {
            return "*";
        }

        StringBuilder sb = new StringBuilder();

        if ((_displayName != null) && (_displayName.length() > 0)) {
            sb.append('"');
            sb.append(_displayName);
            sb.append('"');
        }

        sb.append('<');
        sb.append(_uri);
        sb.append('>');
        // Duplicated in SipURIImpl
        // if (_parameters != null && !_parameters.isEmpty())
        sb.append(super.toString());

        return sb.toString();
    }
}
