/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.UnsupportedEncodingException;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CoderResult;


/**
 * The <code>Ascii7String</code> class represents strings that only contains
 * us7ascii data. <br>
 * 1) To be used to minimize footprint <br>
 * 2) Beware. If someone is using its data as a key in a hashmap then its a good
 * chance that the footprint INCREASES! <br>
 * 3) Beware. Do not use if the data to be minimized is a shared string I.e
 * applicationname is represented by a string when sip.xml is parsed and stored
 * in a "global" structure Then a reference to this string is kept within a
 * sipapplicationSession for convenience. <br>
 * If this local String is copied to a
 * Ascii7String to save memory the footprint INCREASES!
 */
public class Ascii7String {
    private static Charset charset7bit = Charset.forName("US-ASCII");

    /** The value is used for character storage. */
    private byte[] myData;

    /**
     * Initializes a newly created <code>String</code> object so that it
     * represents the same us7ascii data as the argument;
     *
     * @param original
     *        a <code>String</code>.
     */
    public Ascii7String(String original) {
        byte[] orgBytes = getBytes(original);
        int size = orgBytes.length;
        myData = new byte[size];
        System.arraycopy(orgBytes, 0, myData, 0, orgBytes.length);
    }

    /**
     * Package Access constructor to be used carefully if it is
     * certain that the data only contains us7ascii data.
     * No validation of the supplied byte[] will be done.
     * @param bytes
     */
    Ascii7String(byte[] bytes) {
        myData = bytes;
    }

    public Ascii7String(byte[] bytes, int offset, int length, String charset)
        throws UnsupportedEncodingException {
        this(new String(bytes, offset, length, charset));
    }

    public Ascii7String(StringBuilder builder) {
        this(builder.toString());
    }

    /**
     * Returns the length of this string. The length is equal to the number of
     * characters in the string.
     *
     * @return the length of the sequence of characters represented by this
     *         object.
     */
    public int length() {
        return myData.length;
    }

    public String toString() {
        try {
            return new String(myData, "US-ASCII");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);

            // TODO should we add encoding exception to throws clause?
        }
    }

    /**
     * Return the internal byte array
     *
     * @return The internal byte array
     * @since JDK1.1
     */
    public byte[] getBytes() {
        return myData;
    }

    /**
     * Compares this string to the specified object. The result is
     * <code>true</code> if and only if the argument is not <code>null</code>
     * and is a <code>String</code> object that represents the same sequence of
     * characters as this object.
     *
     * @param anObject
     *        the object to compare this <code>String</code> against.
     * @return <code>true</code> if the <code>String </code>are equal;
     *         <code>false</code> otherwise.
     * @see java.lang.String#compareTo(java.lang.String)
     * @see java.lang.String#equalsIgnoreCase(java.lang.String)
     */
    public boolean equals(Object anObject) {
        if (this == anObject) {
            return true;
        }

        if (anObject instanceof Ascii7String) {
            Ascii7String anotherString = (Ascii7String) anObject;
            int n = length();

            if (n == anotherString.length()) {
                byte[] v1 = myData;
                byte[] v2 = anotherString.getBytes();

                while (n-- != 0) {
                    if (v1[n] != v2[n]) {
                        return false;
                    }
                }

                return true;
            }
        } else if (anObject instanceof String) {
            // TODO Inefficient??
            return (new Ascii7String((String) anObject)).equals(this);
        }

        return false;
    }

    /**
     * Returns a hash code for this string. The hash code for a
     * <code>String</code> object is computed as <blockquote>
     *
     * <pre>
     *      s[0]*31&circ;(n-1) + s[1]*31&circ;(n-2) + ... + s[n-1]
     * </pre>
     *
     * </blockquote> using <code>int</code> arithmetic, where <code>s[i]</code>
     * is the <i>i</i>th character of the string, <code>n</code> is the length
     * of the string, and <code>^</code> indicates exponentiation. (The hash
     * value of the empty string is zero.)
     *
     * @return a hash code value for this object.
     */
    public int hashCode() {
        int h = 0;

        for (int i = 0; i < myData.length; i++) {
            h = (31 * h) + myData[i];
        }

        return h;
    }

    public boolean equalsIgnoreCase(Ascii7String anotherString) {
        return (this == anotherString) ? true
                                       : ((anotherString != null) &&
        (anotherString.myData.length == myData.length) &&
        regionMatches(true, 0, anotherString, 0, myData.length));
    }

    public boolean regionMatches(boolean ignoreCase, int toffset,
        Ascii7String other, int ooffset, int len) {
        byte[] ta = myData;
        byte[] pa = other.myData;

        // Note: toffset, ooffset, or len might be near -1>>>1.
        if ((ooffset < 0) || (toffset < 0) ||
                (toffset > ((long) myData.length - len)) ||
                (ooffset > ((long) other.myData.length - len))) {
            return false;
        }

        while (len-- > 0) {
            int char1 = ta[toffset++];
            int char2 = pa[ooffset++];

            if (char1 == char2) {
                continue;
            }

            if (ignoreCase) {
                // If characters don't match but case may be ignored,
                // try converting both characters to uppercase.
                // If the results match, then the comparison scan should
                // continue.
                int u1 = Character.toUpperCase(char1);
                int u2 = Character.toUpperCase(char2);

                if (u1 == u2) {
                    continue;
                }

                // Unfortunately, conversion to uppercase does not work properly
                // for the Georgian alphabet, which has strange rules about case
                // conversion. So we need to make one last check before
                // exiting.
                if (Character.toLowerCase(u1) == Character.toLowerCase(u2)) {
                    continue;
                }
            }

            return false;
        }

        return true;
    }

    public static byte[] getBytes(String str) {
        char[] ca = str.toCharArray();
        byte[] ba = new byte[ca.length];

        if (ca.length == 0) {
            return ba;
        }

        CharsetEncoder ce = charset7bit.newEncoder();
        ce.reset();

        ByteBuffer bb = ByteBuffer.wrap(ba);
        CharBuffer cb = CharBuffer.wrap(ca, 0, ca.length);

        try {
            CoderResult cr = ce.encode(cb, bb, true);

            if (!cr.isUnderflow()) {
                cr.throwException();
            }

            cr = ce.flush(bb);

            if (!cr.isUnderflow()) {
                cr.throwException();
            }
        } catch (CharacterCodingException x) {
            // Substitution is always enabled,
            // so this shouldn't happen
            throw new IllegalStateException(x);
        }

        return ba;
    }
}
