/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

import javax.servlet.sip.Address;


/**
 * TODO Add comments (class description)
 *
 * @author ehsroha
 * @since 2005-maj-26
 */
public class DialogSet implements Externalizable {
    /**
     *
     */
    private static final long serialVersionUID = -6352106202257349396L;
    private /* final */ Ascii7String m_CallId;
    private /* final */ Address m_From;
    private Queue<DialogFragment> m_Dialogs = new ConcurrentLinkedQueue<DialogFragment>();
    private AtomicInteger m_DialogSetFragmentId = new AtomicInteger(DialogFragment.DEFAULT_FRAGMENT_ID);
    private boolean m_IsEarlyDialogValid = true;

    public DialogSet(String callId, Address from) {
        m_CallId = new Ascii7String(callId);
        m_From = from;
    }

    // For serialization
    public DialogSet() {
    }

    public void writeExternal(ObjectOutput out) throws IOException {
        out.writeUTF(m_CallId.toString());
        out.writeObject(m_From);
        out.writeInt(m_Dialogs.size());

        Iterator<DialogFragment> i = m_Dialogs.iterator();

        while (i.hasNext()) {
            out.writeObject(i.next());
        }

        out.writeInt(m_DialogSetFragmentId.get());
    }

    public void readExternal(ObjectInput in)
        throws IOException, ClassNotFoundException {
        m_CallId = new Ascii7String(in.readUTF());
        m_From = (Address) in.readObject();
        m_Dialogs.clear();

        int len = in.readInt();

        for (int i = 0; i < len; i++) {
            m_Dialogs.add((DialogFragment) in.readObject());
        }

        m_DialogSetFragmentId = new AtomicInteger(in.readInt());
    }

    public String getCallId() {
        return m_CallId.toString();
    }

    public Address getFrom() {
        return m_From;
    }

    /**
     * @return the from tag of this Dialog.
     */
    public String getFromTag() {
        return getFrom().getParameter(AddressImpl.TAG_PARAM);
    }

    public void addDialog(DialogFragment d) {
        m_Dialogs.add(d);
    }

    public void removeDialog(DialogFragment d) {
        m_Dialogs.remove(d);
        removeEarlyDialog();
    }

    public void removeEarlyDialog() {
        if (m_IsEarlyDialogValid) {
            synchronized (this) {
                if (m_IsEarlyDialogValid) {
                    SessionManager.getInstance().removeEarlyDialog(this);
                    m_IsEarlyDialogValid = false;
                }
            }
        }
    }

    public DialogFragment getDialog(int fragmentId) {
        for (DialogFragment d : m_Dialogs) {
            if (fragmentId == d.getFragmentId()) {
                // match, lets return it...
                return d;
            }
        }

        // no match...
        return null;
    }

    public Iterator<DialogFragment> getDialogs() {
        return m_Dialogs.iterator();
    }

    public DialogFragment searchForDialog(String toTag, int fragmentId) {
        String key = DialogFragment.createKey(m_CallId.toString(),
                getFromTag(), toTag, fragmentId);

        return DialogFragmentManager.getInstance().findDialogFragment(key);
    }

    /**
     * Needed for creational NOTIFY
     *
     * @param fragmentId
     * @return a cloned DialogFragment matching the fragmentId or null otherwise
     */
    public DialogFragment cloneDialog(int fragmentId) {
        for (DialogFragment d : m_Dialogs) {
            if (fragmentId == d.getFragmentId()) {
                // match, lets return a clone...
                return (DialogFragment) d.clone();
            }
        }

        // no match...
        return null;
    }

    public int nextFragmentId() {
        return m_DialogSetFragmentId.incrementAndGet();
    }
}
