/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.container.SipContainerThreadPool;

import java.util.logging.Level;

import javax.servlet.sip.SipServletMessage;


/**
 * This represents the finite state machine of a UAC, UAS or combined
 *
 * @author ehsroha
 */
public abstract class FSM {
    protected FSM() {
    }

    /**
     * Will return and if needed create a FSM responsible of the processing of
     * the message or null if no FSM was capable.
     *
     * @param req
     *        the incoming request is the key to decide which FSM to create
     * @return a suitable FSM or null if no one is found
     */
    public static FSM createFSM(SipServletMessage m) {
        FSM fsm = INVITESession.createFSM(m);

        if (fsm != null) {
            return fsm;
        }

        fsm = SUBSCRIBE_REFERSession.createFSM(m);

        if (fsm != null) {
            return fsm;
        }

        fsm = GeneralSession.createFSM(m);

        if (fsm != null) {
            return fsm;
        }

        // list of rest of fsm if any
        return null;
    }

    /**
     * Returns a clone of current FSM
     *
     * @return a clone of current FSM
     */
    public abstract Object clone();

    /**
     * Whether this FSM is responsible of processing the message or not.
     *
     * @param m
     *        the incoming message
     * @return if this instance is responsible for the message or not
     */
    public abstract boolean isResponsible(SipServletMessage m);

    /**
     * Whether it's safe to remove this instance or not
     *
     * @return if it's safe to delete it or not
     */
    public abstract boolean isDeletable();

    /**
     * Handles an incoming request from the local UAC. Updates the session and
     * forwards the request to the remote UAS upon success.
     *
     */
    public void send(SipServletRequestImpl req, UA uac)
        throws IllegalStateException {
        final SipServletRequestImpl forward = req;
        // push the request to the dispatcher in a new thread
        SipContainerThreadPool.getInstance().execute(new Runnable() {
                public void run() {
                    // push the request to the dispatcher
                    forward.popDispatcher().dispatch(forward);
                }
            });
    }

    /**
     * Handles an incoming response from the local UAS. Updates the session and
     * forwards the response to the remote UAC upon success.
     */
    public void send(SipServletResponseImpl resp, UA uas)
        throws IllegalStateException {
        final SipServletResponseImpl forward = resp;
        // push the response to the dispatcher in a new thread
        SipContainerThreadPool.getInstance().execute(new Runnable() {
                public void run() {
                    // push the response to the dispatcher
                    forward.popDispatcher().dispatch(forward);
                }
            });
    }

    /**
     * Handles an incoming request from the remote UAC and updates the session.
     */
    public abstract void dispatch(SipServletRequestImpl req, UA uas);

    /**
     * Handles an incoming response from the remote UAS and updates the session.
     */
    public abstract void dispatch(SipServletResponseImpl resp, UA uac);
}
