/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.UnsupportedEncodingException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;

import javax.servlet.sip.Address;
import javax.servlet.sip.Parameterable;
import javax.servlet.sip.ServletParseException;


/**
 * @author ekrigro TODO To change the template for this generated type comment
 *         go to Window - Preferences - Java - Code Style - Code Templates
 */
public abstract class Header implements Externalizable {
    /**
     * This class encapsulates all the strange aspects of SIP header parsing
     * Another thing that it encapsulates is the way of restricting access to
     * write to system headers according to the SSA 1.0 specification.
     */
    static Logger _log = (Logger) Logger.getLogger("SipContainer");

    // RFC3261 Headers
    // if no Accept header field is present, the server SHOULD assume a default
    // value of application/sdp.
    // An empty Accept header field means that no formats are acceptable.
    public static final String ACCEPT = "Accept";
    public static final String ACCEPT_ENCODING = "Accept-Encoding";
    public static final String ACCEPT_LANGUAGE = "Accept-Language";
    public static final String ACCEPT_INFO = "Alert-Info";
    public static final String ALLOW = "Allow";
    public static final String ALLOW_EVENTS = "Allow-Events"; // rfc3265
    public static final String ALLOW_EVENTS_SHORT = "U"; // rfc3265
    public static final String AUTHENTICATION_INFO = "Authentication-Info";
    public static final String AUTHORIZATION = "Authorization";
    public static final String CALL_INFO = "Call-Info";
    public static final String CONTENT_TYPE = "Content-Type";
    public static final String CONTENT_TYPE_SHORT = "C";
    public static final String CONTENT_LENGTH = "Content-Length";
    public static final String CONTENT_LENGTH_SHORT = "L";
    public static final String CONTENT_DISPOSITION = "Content-Disposition";
    public static final String CONTENT_ENCODING = "Content-Encoding";
    public static final String CONTENT_ENCODING_SHORT = "E";
    public static final String CONTENT_LANGUAGE = "Content-Language";
    public static final String DATE = "Date";
    public static final String ERROR_INFO = "Error-Info";
    public static final String EVENT = "Event"; // rfc3265
    public static final String EVENT_SHORT = "O"; // rfc3265
    public static final String EXPIRES = "Expires";
    public static final String IN_REPLY_TO = "In-Reply-To";
    public static final String MAX_FORWARDS = "Max-Forwards";
    public static final String MIME_VERSION = "Mime-Version";
    public static final String MIN_EXPIRES = "Min-Expires";
    public static final String ORGANIZATION = "Organization";
    public static final String PRIORITY = "Priority";
    public static final String PROXY_AUTHENTICATE = "Proxy-Authenticate";
    public static final String PROXY_AUTHORIZATION = "Proxy-Authorization";
    public static final String PROXY_REQUIRE = "Proxy-Require";
    public static final String REPLY_TO = "Reply-To";
    public static final String REQUIRE = "Require";
    public static final String RETRY_AFTER = "Retry-After";
    public static final String SERVER = "Server";
    public static final String SUBJECT = "Subject";
    public static final String SUBJECT_SHORT = "S";
    public static final String SUBSCRIPTION_STATE = "Subscription-State"; // rfc3265
    public static final String SUPPORTED = "Supported";
    public static final String SUPPORTED_SHORT = "K";
    public static final String TIMESTAMP = "Timestamp";
    public static final String UNSUPPORTED = "Unsupported";
    public static final String USER_AGENT = "User-Agent";
    public static final String WARNING = "Warning";
    public static final String WWW_AUTHENTICATE = "Www-Authenticate";

    // System Headers
    public static final String CALL_ID = "Call-Id";
    public static final String CALL_ID_SHORT = "I";
    public static final String CONTACT = "Contact"; // NameAddress
    public static final String CONTACT_SHORT = "M";
    public static final String CSEQ = "Cseq";
    public static final String FROM = "From"; // NameAddress
    public static final String FROM_SHORT = "F";

    // public static final Header FROM_HEADER = new Header("From");
    public static final String RECORD_ROUTE = "Record-Route";
    public static final String ROUTE = "Route";
    public static final String TO = "To"; // NameAddress
    public static final String TO_SHORT = "T";
    public static final String VIA = "Via";
    public static final String VIA_SHORT = "V";

    // 3262 system headers
    public static final String RACK = "Rack";
    public static final String RSEQ = "Rseq";

    // 3515 Refer RFC
    public static final String REFER_TO = "Refer-To";
    public static final String REFER_TO_SHORT = "R";

    // 3892 Referred-By
    public static final String REFERRED_BY = "Referred-By";
    public static final String REFERRED_BY_SHORT = "B";

    // P_ASSERTED_ID
    public static final String P_ASSERTED_ID = "P-Asserted-Identity";
    public static final String ILLEGAL_ACCESS = "You can not modify a system header!";

    // 4028 Session Timers
    public static final String MIN_SE = "Min-Se";
    public static final String SESSION_EXPIRES = "Session-Expires";
    public static final String SESSION_EXPIRES_SHORT = "X";

    // 3323 Privacy
    public static final String PRIVACY = "Privacy";

    // 3891 Replaces
    public static final String REPLACES = "Replaces";

    // 3903 Event State Publication
    public static final String SIP_ETAG = "Sip-Etag";
    public static final String SIP_IF_MATCH = "Sip-If-Match";

    // 4488 REFER Method Implicit Subscription
    public static final String REFER_SUB = "Refer-Sub";

    // 3327 Registering Non-Adjacent Contacts
    public static final String PATH = "Path";

    // TODO If you add a header here make sure that it is added to the ALL MAP
    // and to other if applies.
    public static final Map<String, String> ALL_HEADERS_MAP = new HashMap<String, String>();

    static {
        ALL_HEADERS_MAP.put(ACCEPT, ACCEPT);
        ALL_HEADERS_MAP.put(ACCEPT_ENCODING, ACCEPT_ENCODING);
        ALL_HEADERS_MAP.put(ACCEPT_LANGUAGE, ACCEPT_LANGUAGE);
        ALL_HEADERS_MAP.put(ACCEPT_INFO, ACCEPT_INFO);
        ALL_HEADERS_MAP.put(ALLOW, ALLOW);
        ALL_HEADERS_MAP.put(ALLOW_EVENTS, ALLOW_EVENTS);
        ALL_HEADERS_MAP.put(AUTHENTICATION_INFO, AUTHENTICATION_INFO);
        ALL_HEADERS_MAP.put(AUTHORIZATION, AUTHORIZATION);
        ALL_HEADERS_MAP.put(CALL_INFO, CALL_INFO);
        ALL_HEADERS_MAP.put(CONTENT_TYPE, CONTENT_TYPE);
        ALL_HEADERS_MAP.put(CONTENT_LENGTH, CONTENT_LENGTH);
        ALL_HEADERS_MAP.put(CONTENT_DISPOSITION, CONTENT_DISPOSITION);
        ALL_HEADERS_MAP.put(CONTENT_ENCODING, CONTENT_ENCODING);
        ALL_HEADERS_MAP.put(CONTENT_LANGUAGE, CONTENT_LANGUAGE);
        ALL_HEADERS_MAP.put(DATE, DATE);
        ALL_HEADERS_MAP.put(ERROR_INFO, ERROR_INFO);
        ALL_HEADERS_MAP.put(EVENT, EVENT);
        ALL_HEADERS_MAP.put(EXPIRES, EXPIRES);
        ALL_HEADERS_MAP.put(IN_REPLY_TO, IN_REPLY_TO);
        ALL_HEADERS_MAP.put(MAX_FORWARDS, MAX_FORWARDS);
        ALL_HEADERS_MAP.put(MIME_VERSION, MIME_VERSION);
        ALL_HEADERS_MAP.put(MIN_EXPIRES, MIN_EXPIRES);
        ALL_HEADERS_MAP.put(ORGANIZATION, ORGANIZATION);
        ALL_HEADERS_MAP.put(PRIORITY, PRIORITY);
        ALL_HEADERS_MAP.put(PROXY_AUTHENTICATE, PROXY_AUTHENTICATE);
        ALL_HEADERS_MAP.put(PROXY_AUTHORIZATION, PROXY_AUTHORIZATION);
        ALL_HEADERS_MAP.put(PROXY_REQUIRE, PROXY_REQUIRE);
        ALL_HEADERS_MAP.put(REPLY_TO, REPLY_TO);
        ALL_HEADERS_MAP.put(REQUIRE, REQUIRE);
        ALL_HEADERS_MAP.put(RETRY_AFTER, RETRY_AFTER);
        ALL_HEADERS_MAP.put(SERVER, SERVER);
        ALL_HEADERS_MAP.put(SUBJECT, SUBJECT);
        ALL_HEADERS_MAP.put(SUBSCRIPTION_STATE, SUBSCRIPTION_STATE);
        ALL_HEADERS_MAP.put(SUPPORTED, SUPPORTED);
        ALL_HEADERS_MAP.put(TIMESTAMP, TIMESTAMP);
        ALL_HEADERS_MAP.put(UNSUPPORTED, UNSUPPORTED);
        ALL_HEADERS_MAP.put(USER_AGENT, USER_AGENT);
        ALL_HEADERS_MAP.put(WARNING, WARNING);
        ALL_HEADERS_MAP.put(WWW_AUTHENTICATE, WWW_AUTHENTICATE);
        // System Headers
        ALL_HEADERS_MAP.put(CALL_ID, CALL_ID);
        ALL_HEADERS_MAP.put(CONTACT, CONTACT);
        ALL_HEADERS_MAP.put(CSEQ, CSEQ);
        ALL_HEADERS_MAP.put(FROM, FROM);
        ALL_HEADERS_MAP.put(RECORD_ROUTE, RECORD_ROUTE);
        ALL_HEADERS_MAP.put(ROUTE, ROUTE);
        ALL_HEADERS_MAP.put(TO, TO);
        ALL_HEADERS_MAP.put(VIA, VIA);
        ALL_HEADERS_MAP.put(RACK, RACK);
        ALL_HEADERS_MAP.put(RSEQ, RSEQ);
        // 3515 Refer RFC
        ALL_HEADERS_MAP.put(REFER_TO, REFER_TO);
        // 3892 Referred-By
        ALL_HEADERS_MAP.put(REFERRED_BY, REFERRED_BY);
        // P_ASSERTED_ID
        ALL_HEADERS_MAP.put(P_ASSERTED_ID, P_ASSERTED_ID);
        // 4028 Session Timers
        ALL_HEADERS_MAP.put(MIN_SE, MIN_SE);
        ALL_HEADERS_MAP.put(SESSION_EXPIRES, SESSION_EXPIRES);
        // 3323 Privacy
        ALL_HEADERS_MAP.put(PRIVACY, PRIVACY);
        // 3891 Replaces
        ALL_HEADERS_MAP.put(REPLACES, REPLACES);
        // 3903 Event State Publication
        ALL_HEADERS_MAP.put(SIP_ETAG, SIP_ETAG);
        ALL_HEADERS_MAP.put(SIP_IF_MATCH, SIP_IF_MATCH);
        // 4488 REFER Method Implicit Subscription
        ALL_HEADERS_MAP.put(REFER_SUB, REFER_SUB);
        // 3327 Registering Non-Adjacent Contacts
        ALL_HEADERS_MAP.put(PATH, PATH);
    }

    public static final Map<String, String> SHORT_TO_LONG_MAP = new HashMap<String, String>();

    static {
        SHORT_TO_LONG_MAP.put(CONTENT_TYPE_SHORT, CONTENT_TYPE);
        SHORT_TO_LONG_MAP.put(CONTENT_LENGTH_SHORT, CONTENT_LENGTH);
        SHORT_TO_LONG_MAP.put(CONTENT_ENCODING_SHORT, CONTENT_ENCODING);
        SHORT_TO_LONG_MAP.put(SUBJECT_SHORT, SUBJECT);
        SHORT_TO_LONG_MAP.put(SUPPORTED_SHORT, SUPPORTED);
        SHORT_TO_LONG_MAP.put(CALL_ID_SHORT, CALL_ID);
        SHORT_TO_LONG_MAP.put(CONTACT_SHORT, CONTACT);
        SHORT_TO_LONG_MAP.put(FROM_SHORT, FROM);
        SHORT_TO_LONG_MAP.put(TO_SHORT, TO);
        SHORT_TO_LONG_MAP.put(VIA_SHORT, VIA);
        // RFC 3265
        SHORT_TO_LONG_MAP.put(ALLOW_EVENTS_SHORT, ALLOW_EVENTS);
        SHORT_TO_LONG_MAP.put(EVENT_SHORT, EVENT);
        // RFC 3515
        SHORT_TO_LONG_MAP.put(REFER_TO_SHORT, REFER_TO);
        // RFC 3892
        SHORT_TO_LONG_MAP.put(REFERRED_BY_SHORT, REFERRED_BY);
        // 4028 Session Timers
        SHORT_TO_LONG_MAP.put(SESSION_EXPIRES_SHORT, SESSION_EXPIRES);
    }

    public static final Map<String, String> SYSTEM_HEADER_MAP = new HashMap<String, String>();

    static {
        SYSTEM_HEADER_MAP.put(FROM, FROM);
        SYSTEM_HEADER_MAP.put(TO, TO);
        SYSTEM_HEADER_MAP.put(CALL_ID, CALL_ID);
        SYSTEM_HEADER_MAP.put(CSEQ, CSEQ);
        SYSTEM_HEADER_MAP.put(VIA, VIA);
        SYSTEM_HEADER_MAP.put(ROUTE, ROUTE);
        SYSTEM_HEADER_MAP.put(RECORD_ROUTE, RECORD_ROUTE);
        SYSTEM_HEADER_MAP.put(CONTACT, CONTACT); // REGISTER requests and
                                                 // responses, as well as 3xx and
                                                 // 485 responses OK

        SYSTEM_HEADER_MAP.put(RACK, RACK);
        SYSTEM_HEADER_MAP.put(RSEQ, RSEQ);
    }

    public static final Map<String, String> SINGLE_LINE_HEADER_MAP = new HashMap<String, String>();

    static {
        SINGLE_LINE_HEADER_MAP.put(FROM, FROM);
        SINGLE_LINE_HEADER_MAP.put(TO, TO);
        SINGLE_LINE_HEADER_MAP.put(CALL_ID, CALL_ID);
        SINGLE_LINE_HEADER_MAP.put(CSEQ, CSEQ);
        SINGLE_LINE_HEADER_MAP.put(CONTENT_LENGTH, CONTENT_LENGTH);
        SINGLE_LINE_HEADER_MAP.put(CONTENT_TYPE, CONTENT_TYPE);
        SINGLE_LINE_HEADER_MAP.put(MAX_FORWARDS, MAX_FORWARDS);
        SINGLE_LINE_HEADER_MAP.put(RACK, RACK);
        SINGLE_LINE_HEADER_MAP.put(RSEQ, RSEQ);
        SINGLE_LINE_HEADER_MAP.put(CONTENT_DISPOSITION, CONTENT_DISPOSITION);
        SINGLE_LINE_HEADER_MAP.put(DATE, DATE);
        SINGLE_LINE_HEADER_MAP.put(EXPIRES, EXPIRES);
        SINGLE_LINE_HEADER_MAP.put(MIME_VERSION, MIME_VERSION);
        SINGLE_LINE_HEADER_MAP.put(MIN_EXPIRES, MIN_EXPIRES);
        SINGLE_LINE_HEADER_MAP.put(ORGANIZATION, ORGANIZATION);
        SINGLE_LINE_HEADER_MAP.put(PRIORITY, PRIORITY);
        SINGLE_LINE_HEADER_MAP.put(RETRY_AFTER, RETRY_AFTER);
        SINGLE_LINE_HEADER_MAP.put(SERVER, SERVER);
        SINGLE_LINE_HEADER_MAP.put(SUBJECT, SUBJECT);
        SINGLE_LINE_HEADER_MAP.put(TIMESTAMP, TIMESTAMP);
        SINGLE_LINE_HEADER_MAP.put(USER_AGENT, USER_AGENT);
        // RFC 3515
        SINGLE_LINE_HEADER_MAP.put(REFER_TO, REFER_TO);
        // RFC 3892
        SINGLE_LINE_HEADER_MAP.put(REFERRED_BY, REFERRED_BY);
        // 4028 Session Timers
        SINGLE_LINE_HEADER_MAP.put(MIN_SE, MIN_SE);
        SINGLE_LINE_HEADER_MAP.put(SESSION_EXPIRES, SESSION_EXPIRES);
        // 3323 Privacy
        SINGLE_LINE_HEADER_MAP.put(PRIVACY, PRIVACY);
        // 3891 Replaces
        SINGLE_LINE_HEADER_MAP.put(REPLACES, REPLACES);
        // 3903 Event State Publication
        SINGLE_LINE_HEADER_MAP.put(SIP_ETAG, SIP_ETAG);
        SINGLE_LINE_HEADER_MAP.put(SIP_IF_MATCH, SIP_IF_MATCH);
        // 4488 REFER Method Implicit Subscription
        SINGLE_LINE_HEADER_MAP.put(REFER_SUB, REFER_SUB);
        // 3327 Registering Non-Adjacent Contacts
        SINGLE_LINE_HEADER_MAP.put(PATH, PATH);
    }

    protected boolean _readOnly = false;
    protected boolean _singleLine = true;
    protected List<Boolean> _multiValuesOnSameLineIndicator = new ArrayList<Boolean>();
    protected String _name;
    protected SipFactoryImpl _sipFactory;

    /*
     * Package local constructor used to create headers internaly in the
     * container. Should be used in cases when the header name is already pretty
     * printed.
     */
    protected Header(String pretty, boolean systemHeader, boolean singleLine) {
        _name = pretty;
        _readOnly = systemHeader;
        _singleLine = singleLine;
        _sipFactory = SipFactoryImpl.getInstance();
    }

    public Header() {
    } // Only for the Externalizable interface!

    // TODO This method shuld not be used, use create formated
    @Deprecated
    public static Header create(String header, SipServletMessageImpl message) {
        String name = format(header.getBytes(), 0, header.length());

        return createFormated(name, message);
    }

    public static Header create(byte[] bytes, int offset, int length,
        SipServletMessageImpl message) {
        String name = format(bytes, offset, length);

        return createFormated(name, message);
    }

    // TODO The String name can be exchange into a enum for better code
    public static Header createFormated(String name,
        SipServletMessageImpl message) {
        boolean readOnly = isSystemHeader(name, message);
        boolean singleLine = isSingleLineHeader(name);

        if (singleLine) {
            return new SingleLineHeader(name, readOnly);
        } else {
            return new MultiLineHeader(name, readOnly);
        }
    }

    // For RMI serialization
    public void writeExternal(ObjectOutput output) throws IOException {
        try {
            if (_multiValuesOnSameLineIndicator != null) {
                output.writeInt(_multiValuesOnSameLineIndicator.size());

                Iterator<Boolean> i = _multiValuesOnSameLineIndicator.iterator();

                while (i.hasNext()) {
                    output.writeBoolean(i.next());
                }
            } else {
                output.writeInt(0);
            }

            output.writeBoolean(_readOnly);
            output.writeBoolean(_singleLine);
            output.writeUTF(_name);
        } catch (Exception ignore) {
        }
    }

    public void readExternal(ObjectInput input) throws IOException {
        try {
            int len = input.readInt();

            if (_multiValuesOnSameLineIndicator == null) {
                _multiValuesOnSameLineIndicator = new ArrayList<Boolean>(len);
            }

            for (int i = 0; i < len; i++) {
                _multiValuesOnSameLineIndicator.add(input.readBoolean());
            }

            _readOnly = input.readBoolean();
            _singleLine = input.readBoolean();

            String name = input.readUTF();
            String constant = ALL_HEADERS_MAP.get(name);

            if (constant != null) {
                _name = constant; // Memory optimazation
            } else {
                _name = name;
            }

            _sipFactory = SipFactoryImpl.getInstance();
        } catch (Exception ignore) {
        }
    }

    protected abstract Object clone();

    public static String format(String name) {
        return format(name.getBytes(), 0, name.length());
    }

    public static String format(byte[] bytes, int offset, int length) {
        boolean firstChar = true;
        boolean started = false;
        int endIndex = offset + length;

        for (int i = offset; i < endIndex; i++) {
            byte b = bytes[i];

            switch (b) {
            case ' ':
            case '\t':

                if (started) {
                    endIndex = i;
                } else {
                    offset = i + 1;
                }

                break;

            case '-':
                firstChar = true;

                break;

            default:

                if (!firstChar && (0x40 < b) && (b < 0x5B)) { // Upper case
                    bytes[i] = (byte) (b | 0x20); // to lower
                } else if (firstChar && (0x60 < b) && (b < 0x7B)) { // Lower case
                    bytes[i] = (byte) (b & 0xDF); // to upper
                }

                // Not a USASCII char!
                // else if( b < 0x7F ) {}
                started = true;
                firstChar = false;
            }
        }

        int diff = endIndex - offset;

        if (diff <= 0) {
            throw new NullPointerException("Not a valid header value!");
        } else if (diff == 1) {
            try {
                return SHORT_TO_LONG_MAP.get(new String(bytes, offset, diff,
                        "US-ASCII"));
            } catch (UnsupportedEncodingException e) {
            } // Should never be here
        }

        try {
            String name = new String(bytes, offset, diff, "US-ASCII");
            String constant = ALL_HEADERS_MAP.get(name);

            if (constant != null) {
                return constant;
            } else {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "The header " + name +
                        " could be added to ALL_HEADERS_MAP for better performance!");
                }
            }

            return name;
        } catch (UnsupportedEncodingException e) {
        } // Should never be here

        return null;
    }

    public String getName() {
        return _name;
    }

    public boolean isReadOnly() {
        return _readOnly;
    }

    public void setReadOnly(boolean ro) {
        _readOnly = ro;
    }

    public abstract String getValue();

    public abstract void setValue(String value, boolean first);

    public abstract Address getAddressValue() throws ServletParseException;

    public abstract void setAddressValue(Address address, boolean first);

    public abstract ListIterator<String> getValues();

    public abstract ListIterator<Address> getAddressValues()
        throws ServletParseException;

   
    public abstract Parameterable getParameterableValue() throws ServletParseException;

    public abstract void setParameterableValue(Parameterable address, boolean first);

    public abstract ListIterator<Parameterable> getParameterableValues()
        throws ServletParseException;


    public abstract void removeValues();

    public abstract void merge(Header h);

    /*
     * public void setValues( List values ) { } public byte[] getBytes() { return
     * null; } public byte[] getBytes( ByteBuffer bb ) { return null; }
     */
    public boolean equals(Object o) {
        if (o instanceof Header) {
            return ((Header) o).getName().equals(_name);
        } else if (o instanceof String) {
            return ((String) o).equals(_name);
        }

        return false;
    }

    public int hashCode() {
        return _name.hashCode();
    }

    // /**
    // * Old and simplistic version, commented out when fixing TR B0007.
    // */
    // public static boolean isSystemHeader( String name ) { return
    // SYSTEM_HEADER_MAP.containsKey( name ); }
    // /**
    // * This method is currently not called anywhere. Commented out while fixing
    // TR B0007. If a method
    // * like this one is later needed, please note that a SipServletMessageImpl
    // needs to be passed along
    // * too (see definition of isSystemHeader(_,_) below)
    // * @param name
    // * @return
    // */
    // public static boolean isSystemHeader( Header name ) { return
    // isSystemHeader( name.getName() ) ; }
    /**
     * Check if the given string is a System Header name according to SSA 6.4.2.
     */
    public static boolean isSystemHeader(String name,
        SipServletMessageImpl message) {
        if (!SYSTEM_HEADER_MAP.containsKey(name)) {
            return false;
        } else if (!name.equals(CONTACT)) {
            return true;
        } else if (message.getMethod().equals("REGISTER")) // Contact header in
                                                           // REGISTER
         {
            return false;
        } else if (message instanceof SipServletRequestImpl) {
            return true;
        } else {
            int code = ((SipServletResponseImpl) message).getStatus();

            if ((code == 485) || ((code >= 300) && (code < 400))) {
                return false;
            } else {
                return true;
            }
        }
    }

    public static boolean isSingleLineHeader(String name) {
        return SINGLE_LINE_HEADER_MAP.containsKey(name);
    }

    public static boolean isSingleLineHeader(Header name) {
        return isSingleLineHeader(name.getName());
    }

    /**
     * Indicating if the values will be set in one line.
     *
     * @param nextValueOnSameLineIndicator,
     *        indicates if current value in Header will have a follow up value on
     *        the same line eg thisvalue, nextvalue.
     * @param first,
     *        when set to true nextValueOnSameLineIndicator is applicable for the
     *        first value.
     */
    public void setIsNextValueOnSameLine(boolean nextValueOnSameLineIndicator,
        boolean first) {
        if (first) {
            _multiValuesOnSameLineIndicator.add(0, nextValueOnSameLineIndicator);
        } else {
            _multiValuesOnSameLineIndicator.add(nextValueOnSameLineIndicator);
        }
    }

    /**
     * This method will return an indicator whether the values after the first()
     * or last added are on the same line eg first, nextvalue or last,nextvalue.
     *
     * @return A boolean indicating if the next header values is on the same line
     */
    public boolean getIsNextValueOnSameLine(boolean first) {
        if (first) {
            return _multiValuesOnSameLineIndicator.get(0).booleanValue();
        } else {
            int last = _multiValuesOnSameLineIndicator.size() - 1;

            return _multiValuesOnSameLineIndicator.get(last).booleanValue();
        }
    }

    /*
     * TODO maybe it would be better if all the constant headers are enum and not
     * independet strings ekrigro public static enum Type { ACCEPT("Accept");
     * private final String value; Type(String value) { this.value = value; }
     * public String getValue() { return value; } }
     */
}
