/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import javax.servlet.sip.Address;
import javax.servlet.sip.Parameterable;
import javax.servlet.sip.ServletParseException;


/**
 * @author ekrigro TODO To change the template for this generated type comment
 *         go to Window - Preferences - Java - Code Style - Code Templates
 * @reviewed qmigkra 2006-dec-1
 */
public final class MultiLineHeader extends Header implements Externalizable {
    /**
     *
     */
    private static final long serialVersionUID = 3834588816114529840L;
    public static final String LIST_ALTERED_ERROR_REASON = "List backing iterator have been altered with non-Address value after creation of iterator";
    private List<AddressOrValue> addressesOrValues = new ArrayList<AddressOrValue>();

    /**
     * @param pretty
     * @param systemHeader *
     * @param singleLine
     */
    public MultiLineHeader(String pretty, boolean systemHeader) {
        super(pretty, systemHeader, false);
    }

    public MultiLineHeader() {
    } //Only for the Externalizable interface!

    public void writeExternal(ObjectOutput output) throws IOException {
        if (addressesOrValues != null) {
            output.writeInt(addressesOrValues.size());

            Iterator<AddressOrValue> i = addressesOrValues.iterator();

            while (i.hasNext()) {
                output.writeUTF(i.next().toString());
            }
        } else {
            output.writeInt(0);
        }

        super.writeExternal(output);
    }

    public void readExternal(ObjectInput input) throws IOException {
        int len = input.readInt();

        if (addressesOrValues == null) {
            addressesOrValues = new ArrayList<AddressOrValue>(len);
        }

        for (int i = 0; i < len; i++) {
            addressesOrValues.add(new AddressOrValue(input.readUTF()));
        }

        super.readExternal(input);
    }

    protected Object clone() {
        MultiLineHeader newHeader = new MultiLineHeader(new String(_name),
                _readOnly);
        Iterator<Boolean> mv = _multiValuesOnSameLineIndicator.iterator();

        while (mv.hasNext()) {
            Boolean k = mv.next();
            newHeader._multiValuesOnSameLineIndicator.add(new Boolean(k));
        }

        Iterator<AddressOrValue> j = addressesOrValues.iterator();

        while (j.hasNext()) {
            AddressOrValue k = j.next();
            newHeader.addressesOrValues.add((AddressOrValue) ((AddressOrValue) k).clone());
        }

        return newHeader;
    }

    public String getValue() {
        if (!addressesOrValues.isEmpty()) {
            return addressesOrValues.get(0).toString();
        }

        return null;
    }

    public void setValue(String value, boolean first) {
        // TODO throw away the Address map if there is one since if we would
        // parse and get an exception we would not be able to recover
        if (value != null) {
            if (first) {
                addressesOrValues.add(0, new AddressOrValue(value));
            } else {
                addAddrOrValue(new AddressOrValue(value));
            }
        }
    }

    private boolean addAddrOrValue(AddressOrValue addrOrValue) {
        boolean retVal = addressesOrValues.add(addrOrValue);

        if (_name.equals(Header.P_ASSERTED_ID)) {
            // Re-sort list so that the order will be SIP URI:s, SIPS URI:s, others (e.g. TEL URL:s)
            Collections.sort(addressesOrValues,
                new Comparator<AddressOrValue>() {
                    public int compare(AddressOrValue o1, AddressOrValue o2) {
                        try {
                            int w1 = getWeight(o1);
                            int w2 = getWeight(o2);

                            if ((w1 - w2) > 0) {
                                return 1;
                            }

                            if ((w1 - w2) < 0) {
                                return -1;
                            }
                        } catch (ServletParseException e) {
                            // Ignore (return 0 below)
                        }

                        return 0;
                    }

                    private int getWeight(AddressOrValue o)
                        throws ServletParseException {
                        int value = 0;

                        if (o.getAddressValue().getURI().getScheme()
                                 .equals(SipFactoryImpl.SIP_URI_PROTOCOL)) {
                            return 0;
                        } else if (o.getAddressValue().getURI().getScheme()
                                        .equals(SipFactoryImpl.SIPS_URI_PROTOCOL)) {
                            return 1;
                        } else {
                            return 2;
                        }
                    }
                });
        }

        return retVal;
    }

    public Address getAddressValue() throws ServletParseException {
        if (!addressesOrValues.isEmpty()) {
            Address address = addressesOrValues.get(0).getAddressValue();

            // Even the address of a system header should be protected.
            ((AddressImpl) address).setReadOnly(_readOnly);

            return address;
        }

        return null;
    }

    public void setAddressValue(Address address, boolean first) {
        if (first) {
            addressesOrValues.add(0, new AddressOrValue(address));
        } else {
            addAddrOrValue(new AddressOrValue(address));
        }
    }

    public Parameterable getParameterableValue() throws ServletParseException {
        if (!addressesOrValues.isEmpty()) {
            return addressesOrValues.get(0).getParameterableValue();
        }

        return null;
    }

    public void setParameterableValue(Parameterable parameterable, boolean first) {
        if (first) {
            addressesOrValues.add(0, new AddressOrValue(parameterable));
        } else {
            addAddrOrValue(new AddressOrValue(parameterable));
        }
    }

    public ListIterator<String> getValues() {
        final ListIterator<AddressOrValue> addressIterator = addressesOrValues.listIterator();

        return new ListIterator<String>() {
                public boolean hasNext() {
                    return addressIterator.hasNext();
                }

                public String next() {
                    return addressIterator.next().toString();
                }

                public boolean hasPrevious() {
                    return addressIterator.hasPrevious();
                }

                public String previous() {
                    return addressIterator.previous().toString();
                }

                public int nextIndex() {
                    return addressIterator.nextIndex();
                }

                public int previousIndex() {
                    return addressIterator.previousIndex();
                }

                public void remove() {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    addressIterator.remove();
                }

                public void set(String o) {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    addressIterator.set(new AddressOrValue(o));
                }

                public void add(String o) {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    addressIterator.add(new AddressOrValue(o));
                }
            };
    }

    public ListIterator<Address> getAddressValues()
        throws ServletParseException {
        // Verify beforehand that we wont get a "ServletParseException" on next and previous
        for (AddressOrValue address : addressesOrValues) {
            Address a = address.getAddressValue();

            // Even the address of a system header should be protected.
            ((AddressImpl) a).setReadOnly(_readOnly);
        }

        final ListIterator<AddressOrValue> addressIterator = addressesOrValues.listIterator();

        return new ListIterator<Address>() {
                public boolean hasNext() {
                    return addressIterator.hasNext();
                }

                public Address next() {
                    try {
                        return addressIterator.next().getAddressValue();
                    } catch (ServletParseException e) {
                        // Won't happen unless altered
                        throw new IllegalStateException(LIST_ALTERED_ERROR_REASON);
                    }
                }

                public boolean hasPrevious() {
                    return addressIterator.hasPrevious();
                }

                public Address previous() {
                    try {
                        return addressIterator.previous().getAddressValue();
                    } catch (ServletParseException e) {
                        //	Won't happen unless altered
                        throw new IllegalStateException(LIST_ALTERED_ERROR_REASON);
                    }
                }

                public int nextIndex() {
                    return addressIterator.nextIndex();
                }

                public int previousIndex() {
                    return addressIterator.previousIndex();
                }

                public void remove() {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    addressIterator.remove();
                }

                public void set(Address o) {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    addressIterator.set(new AddressOrValue(o));
                }

                public void add(Address o) {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    addressIterator.add(new AddressOrValue(o));
                }
            };
    }

     public ListIterator<Parameterable> getParameterableValues()
        throws ServletParseException {
        // Verify beforehand that we wont get a "ServletParseException" on next and previous
        for (AddressOrValue parameterable : addressesOrValues) {
            parameterable.getParameterableValue();
        }

        final ListIterator<AddressOrValue> parameterableIterator = addressesOrValues.listIterator();

        return new ListIterator<Parameterable>() {
                public boolean hasNext() {
                    return parameterableIterator.hasNext();
                }

                public Parameterable next() {
                    try {
                        return parameterableIterator.next().getAddressValue();
                    } catch (ServletParseException e) {
                        // Won't happen unless altered
                        throw new IllegalStateException(LIST_ALTERED_ERROR_REASON);
                    }
                }

                public boolean hasPrevious() {
                    return parameterableIterator.hasPrevious();
                }

                public Parameterable previous() {
                    try {
                        return parameterableIterator.previous().getAddressValue();
                    } catch (ServletParseException e) {
                        //	Won't happen unless altered
                        throw new IllegalStateException(LIST_ALTERED_ERROR_REASON);
                    }
                }

                public int nextIndex() {
                    return parameterableIterator.nextIndex();
                }

                public int previousIndex() {
                    return parameterableIterator.previousIndex();
                }

                public void remove() {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    parameterableIterator.remove();
                }

                public void set(Parameterable o) {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    parameterableIterator.set(new AddressOrValue(o));
                }

                public void add(Parameterable o) {
                    if (_readOnly) {
                        throw new IllegalStateException(ILLEGAL_ACCESS);
                    }

                    parameterableIterator.add(new AddressOrValue(o));
                }
            };
    }

    public void removeValues() {
        addressesOrValues.clear();
    }

    public void merge(Header other) {
        if (other instanceof MultiLineHeader && _name.equals(other._name)) {
            MultiLineHeader mlh = (MultiLineHeader) other;
            addressesOrValues.addAll(mlh.addressesOrValues);
            _multiValuesOnSameLineIndicator.addAll(mlh._multiValuesOnSameLineIndicator);
        } else {
            throw new UnsupportedOperationException();
        }
    }

    public String toString() {
        if (!addressesOrValues.isEmpty()) {
            StringBuilder sb = new StringBuilder();
            Iterator<AddressOrValue> i = addressesOrValues.iterator();
            Iterator<Boolean> newline = _multiValuesOnSameLineIndicator.iterator();
            sb.append(_name);
            sb.append(": ");

            while (i.hasNext()) {
                sb.append(i.next().toString());

                boolean nextIsOnSameLine = false;

                if (newline.hasNext()) {
                    Boolean aB = newline.next();
                    nextIsOnSameLine = aB.booleanValue();
                }

                if (i.hasNext() && nextIsOnSameLine) {
                    sb.append(", ");
                } // Else can indacte no next values OR not on same line.
                else if (!i.hasNext()) { // No next were done
                    sb.append(SipFactoryImpl.NEW_LINE);
                } else { // There are next headers, but nextIsOnSameLine is
                         // false, so add next header on newline
                    sb.append(SipFactoryImpl.NEW_LINE);
                    sb.append(_name);
                    sb.append(": ");
                }
            }

            return sb.toString();
        }

        return "";
    }
}
