/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */

package com.ericsson.ssa.sip;

import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.IOException;

/**
 * @author jluehe
 */
public abstract class PersistentDialogFragmentManager extends DialogFragmentManager {
    private static final LogUtil SIP_LOGGER =
        new LogUtil(LogUtil.SIP_LOG_DOMAIN);

    // The persistent store for dialog fragments
    private DialogFragmentStore store;

    /**
     * Gets the DialogFragment with the given id.
     *
     * This method first checks the active cache for a DialogFragment with the
     * given id. If not found, it queries the persistent store.
     *
     * @return The DialogFragment with the given id, or null if not found
     */
    public DialogFragment findDialogFragment(String id) {
        
        DialogFragment df = super.findDialogFragment(id);
        if (df != null) {
            return df;
        }

        try {
            df = swapInDialogFragment(id);
        } catch (IOException ioe) {
            SIP_LOGGER.severe(ioe, "sipstack.swap_in_dialog_fragment_error",
                              id);
        }

        return df;
    }

    /**
     * Removes the given DialogFragment.
     *
     * @param d The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */
    public void removeDialogFragment(DialogFragment df, boolean hasTimedOut) {

        super.removeDialogFragment(df, hasTimedOut);

        if (store != null && df != null && df.isReplicable()) {
            try {
                store.remove(Integer.toString(df.getFragmentId()));
            } catch (IOException ioe) {
                SIP_LOGGER.severe(
                    ioe,
                    "sipstack.remove_dialog_fragment_from_store_error",
                    df.getFragmentId());
            }
        }
    }    

    /**
     * Persists the given DialogFragment.
     *
     * @param df The DialogFragment to be persisted
     */
    public void saveDialogFragment(DialogFragment df)
            throws IOException {
        if (store != null && df != null && df.isReplicable()) {
            store.save(df);
        }
    }

    /**
     * Swaps in the DialogFragment with the given id from the persistent
     * store of this dialog manager.
     *
     * @return The DialogFragment with the given id, or null if not found
     */
    protected DialogFragment swapInDialogFragment(String id)
            throws IOException {

        if (store == null) {
            return null;
        }
        
        DialogFragment df = store.load(id);
        if (df == null) {
            return null;
        }

        registerDialogFragment(df);

        return df;
    }

    /**
     * Removes the given DialogFragment from only the active cache
     *
     * @param df The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */    
    protected void removeDialogFragmentFromCache(DialogFragment df,
                                                 boolean hasTimedOut) {
        super.removeDialogFragment(df, hasTimedOut);
    }     
}
