/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.sip.timer.ServletTimerImpl;
import com.ericsson.ssa.sip.timer.ServletTimerStore;

import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.IOException;


/**
 * A session manager capable of persisting SipSessions,
 * SipApplicationSessions, and ServletTimers.
 *
 * @author jluehe
 */
public abstract class PersistentSipSessionManagerBase
    extends SipSessionManagerBase {
    private static final LogUtil SIP_LOGGER =
        new LogUtil(LogUtil.SIP_LOG_DOMAIN);

    // The persistent store for SipSessions
    private SipSessionStore sipSessionStore;

    // The persistent store for SipApplicationSessions
    private SipApplicationSessionStore sipApplicationSessionStore;

    // The persistent store for ServletTimers
    private ServletTimerStore servletTimerStore;

    /**
     * Gets the SipSession with the given id.
     *
     * This method first checks the active cache for a SipSession with the
     * given id. If not found, it queries the persistent session store.
     *
     * @return The SipSession with the given id, or null if not found
     */
    public SipSessionDialogImpl findSipSession(String id) {
        SipSessionDialogImpl sipSession = super.findSipSession(id);

        if (sipSession != null) {
            return sipSession;
        }

        ClassLoader webappCL = null;
        ClassLoader curCL = null;

        if ((getContext() != null) && (getContext().getLoader() != null)) {
            webappCL = getContext().getLoader().getClassLoader();
            curCL = Thread.currentThread().getContextClassLoader();
        }

        if ((webappCL != null) && (curCL != webappCL)) {
            try {
                Thread.currentThread().setContextClassLoader(webappCL);
                sipSession = swapInSipSession(id);
            } catch (IOException ioe) {
                SIP_LOGGER.severe(ioe, "sipstack.swap_in_sip_session_error",
                                  id);
            } finally {
                Thread.currentThread().setContextClassLoader(curCL);
            }
        } else {
            try {
                sipSession = swapInSipSession(id);
            } catch (IOException ioe) {
                SIP_LOGGER.severe(ioe, "sipstack.swap_in_sip_session_error",
                                  id);
            }
        }

        return sipSession;
    }

    /**
     * Removes the given SipSession from both the active cache and the
     * persistent session store of this session manager,
     *
     * @param sipSession The SipSession to be removed
     */
    public void removeSipSession(SipSessionDialogImpl sipSession) {
        super.removeSipSession(sipSession);

        if (sipSessionStore != null) {
            try {
                sipSessionStore.remove(sipSession.getId());
            } catch (IOException ioe) {
                SIP_LOGGER.severe(
                    ioe,
                    "sipstack.remove_sip_session_from_store_error",
                    sipSession.getId());
            }
        }
    }
    
    /**
     * Removes the given SipSession from only the active cache
     *
     * @param sipSession The SipSession to be removed
     */    
    protected void removeSipSessionFromCache(SipSessionDialogImpl sipSession) {
        super.removeSipSession(sipSession);
    }     

    /**
     * Persists the given SipSession.
     *
     * @param sipSession The SipSession to be persisted
     */
    public void saveSipSession(SipSessionDialogImpl sipSession)
        throws IOException {
        if (sipSessionStore != null) {
            sipSessionStore.save(sipSession);
        }
    }

    /**
     * Swaps in the SipSession with the given id from the persistent session
     * store of this session manager.
     *
     * @return The SipSession with the given id, or null if not found
     */
    protected SipSessionDialogImpl swapInSipSession(String id)
        throws IOException {
        if (sipSessionStore == null) {
            return null;
        }

        SipSessionDialogImpl sipSession = sipSessionStore.load(id);

        if (sipSession == null) {
            return null;
        }

        addSipSession(sipSession);

        return sipSession;
    }

    /**
     * Gets the SipApplicationSession with the given id.
     *
     * This method first checks the active cache for a SipApplicationSession
     * with the given id. If not found, it queries the persistent session
     * store.
     *
     * @return The SipApplicationSession with the given id, or null if not
     * found
     */
    public SipApplicationSessionImpl findSipApplicationSession(String id) {
        SipApplicationSessionImpl sas = super.findSipApplicationSession(id);

        if (sas != null) {
            return sas;
        }

        ClassLoader webappCL = null;
        ClassLoader curCL = null;

        if ((getContext() != null) && (getContext().getLoader() != null)) {
            webappCL = getContext().getLoader().getClassLoader();
            curCL = Thread.currentThread().getContextClassLoader();
        }

        if ((webappCL != null) && (curCL != webappCL)) {
            try {
                Thread.currentThread().setContextClassLoader(webappCL);
                sas = swapInSipApplicationSession(id);
            } catch (IOException ioe) {
                SIP_LOGGER.severe(
                    ioe,
                    "sipstack.swap_in_sip_application_session_error",
                    id);
            } finally {
                Thread.currentThread().setContextClassLoader(curCL);
            }
        } else {
            try {
                sas = swapInSipApplicationSession(id);
            } catch (IOException ioe) {
                SIP_LOGGER.severe(
                    ioe,
                    "sipstack.swap_in_sip_application_session_error",
                    id);
            }
        }

        return sas;
    }

    /**
     * Removes the given SipApplicationSession from both the active cache
     * and the persistent session store of this session manager,
     *
     * @param sas The SipApplicationSession to be removed
     */
    public void removeSipApplicationSession(SipApplicationSessionImpl sas) {
        super.removeSipApplicationSession(sas);

        if (sipApplicationSessionStore != null) {
            try {
                sipApplicationSessionStore.remove(sas.getId());
            } catch (IOException ioe) {
                SIP_LOGGER.severe(
                    ioe,
                    "sipstack.remove_sip_application_session_from_store_error",
                    sas.getId());
            }
        }
    }
    
    /**
     * Removes the given SipApplicationSession from only the active cache
     *
     * @param sas The SipApplicationSession to be removed
     */    
    protected void removeSipApplicationSessionFromCache(SipApplicationSessionImpl sas) {
        super.removeSipApplicationSession(sas);
    }
    
    /**
     * Gets the SipApplicationSession with the given id 
     * from the active cache only
     *
     * @return The SipApplicationSession with the given id, or null if not
     * found
     */
    public SipApplicationSessionImpl findSipApplicationSessionFromCacheOnly(String id) {
        return super.findSipApplicationSession(id);
    }    
    
    /**
     * Gets the SipApplicationSession with the given id 
     * from the active cache only
     *
     * @return The SipSession with the given id, or null if not
     * found
     */
    public SipSessionDialogImpl findSipSessionFromCacheOnly(String id) {
        return super.findSipSession(id);
    }     

    /**
     * Persists the given SipApplicationSession.
     *
     * @param sas The SipApplicationSession to be persisted
     */
    public void saveSipApplicationSession(SipApplicationSessionImpl sas)
        throws IOException {
        if (sipApplicationSessionStore != null) {
            sipApplicationSessionStore.save(sas);
        }
    }

    /**
     * Swaps in the SipApplicationSession with the given id from the
     * persistent session store of this session manager.
     *
     * @return The SipApplicationSession with the given id, or null if not
     * found
     */
    protected SipApplicationSessionImpl swapInSipApplicationSession(String id)
        throws IOException {
        if (sipApplicationSessionStore == null) {
            return null;
        }

        SipApplicationSessionImpl sas = sipApplicationSessionStore.load(id);

        if (sas == null) {
            return null;
        }

        addSipApplicationSession(sas);

        return sas;
    }

    /**
     * Gets the ServletTimer with the given id.
     *
     * This method first checks the active cache for a ServletTimer with the
     * given id. If not found, it queries the persistent session store.
     *
     * @return The ServletTimer with the given id, or null if not found
     */
    public ServletTimerImpl findServletTimer(String id) {
        ServletTimerImpl servletTimer = super.findServletTimer(id);

        if (servletTimer != null) {
            return servletTimer;
        }

        ClassLoader webappCL = null;
        ClassLoader curCL = null;

        if ((getContext() != null) && (getContext().getLoader() != null)) {
            webappCL = getContext().getLoader().getClassLoader();
            curCL = Thread.currentThread().getContextClassLoader();
        }

        if ((webappCL != null) && (curCL != webappCL)) {
            try {
                Thread.currentThread().setContextClassLoader(webappCL);
                servletTimer = swapInServletTimer(id);
            } catch (IOException ioe) {
                SIP_LOGGER.severe(ioe, "sipstack.swap_in_servlet_timer_error",
                                  id);
            } finally {
                Thread.currentThread().setContextClassLoader(curCL);
            }
        } else {
            try {
                servletTimer = swapInServletTimer(id);
            } catch (IOException ioe) {
                SIP_LOGGER.severe(ioe, "sipstack.swap_in_servlet_timer_error",
                                  id);
            }
        }

        return servletTimer;
    }

    /**
     * Removes the given ServletTimer from both the active cache and the
     * persistent session store of this session manager,
     *
     * @param servletTimer The ServletTimer to be removed
     */
    public void removeServletTimer(ServletTimerImpl servletTimer) {
        super.removeServletTimer(servletTimer);

        if (servletTimerStore != null) {
            try {
                servletTimerStore.remove(servletTimer.getId());
            } catch (IOException ioe) {
                SIP_LOGGER.severe(
                    ioe,
                    "sipstack.remove_servlet_timer_from_store_error",
                    servletTimer.getId());
            }
        }
    }
    
    /**
     * Removes the given ServletTimer from only the active cache
     *
     * @param servletTimer The ServletTimer to be removed
     */    
    protected void removeServletTimerFromCache(ServletTimerImpl servletTimer) {
        super.removeServletTimer(servletTimer);
    }
    
    /**
     * Gets the ServletTimer with the given id 
     * from the active cache only
     *
     * @return The ServletTimer with the given id, or null if not
     * found
     */
    public ServletTimerImpl findServletTimerFromCacheOnly(String id) {
        return super.findServletTimer(id);
    }     

    /**
     * Persists the given ServletTimer.
     *
     * @param servletTimer The ServletTimer to be persisted
     */
    public void saveServletTimer(ServletTimerImpl servletTimer)
        throws IOException {
        if (servletTimerStore != null) {
            servletTimerStore.save(servletTimer);
        }
    }

    /**
     * Swaps in the ServletTimer with the given id from the persistent session
     * store of this session manager.
     *
     * @return The ServletTimer with the given id, or null if not found
     */
    protected ServletTimerImpl swapInServletTimer(String id)
        throws IOException {
        if (servletTimerStore == null) {
            return null;
        }

        ServletTimerImpl servletTimer = servletTimerStore.load(id);

        if (servletTimer == null) {
            return null;
        }

        addServletTimer(servletTimer);

        return servletTimer;
    }

    /**
     * Sets the persistent store for SipSessions on this session manager.
     *
     * @param store The persistent store for SipSessions
     */
    public void setSipSessionStore(SipSessionStore store) {
        sipSessionStore = store;
        sipSessionStore.setSipSessionManager(this);
    }

    /**
     * Gets the persistent store for SipSessions from this session manager.
     *
     * @return The persistent store for SipSessions
     */
    public SipSessionStore getSipSessionStore() {
        return sipSessionStore;
    }

    /**
     * Sets the persistent store for SipApplicationSessions on this session
     * manager.
     *
     * @param store The persistent store for SipApplicationSessions
     */
    public void setSipApplicationSessionStore(SipApplicationSessionStore store) {
        sipApplicationSessionStore = store;
        sipApplicationSessionStore.setSipSessionManager(this);
    }

    /**
     * Gets the persistent store for SipApplicationSessions from this session
     * manager.
     *
     * @return The persistent store for SipApplicationSessions
     */
    public SipApplicationSessionStore getSipApplicationSessionStore() {
        return sipApplicationSessionStore;
    }

    /**
     * Sets the persistent store for ServletTimers on this session
     * manager.
     *
     * @param store The persistent store for ServletTimers
     */
    public void setServletTimerStore(ServletTimerStore store) {
        servletTimerStore = store;
        servletTimerStore.setSipSessionManager(this);
    }

    /**
     * Gets the persistent store for ServletTimers from this session
     * manager.
     *
     * @return The persistent store for ServletTimers
     */
    public ServletTimerStore getServletTimerStore() {
        return servletTimerStore;
    }
}
