/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.Constants;
import com.ericsson.ssa.config.lease.Lease;
import com.ericsson.ssa.config.lease.LeaseExpiredException;
import com.ericsson.ssa.container.SipBindingResolver;
import com.ericsson.ssa.container.startup.PerformanceMBeanListener;
import com.ericsson.ssa.sip.PathNode.Type;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;

import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;

import javax.servlet.sip.Address;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipURI;
import javax.servlet.sip.URI;


/**
 * A Singleton that stores SipSessions and dialogs and fetches them upon
 * request.
 *
 * @author ehsroha
 * @reviewed ejoelbi 2007-jan-17
 */
public class SessionManager implements Layer {
    private static final Logger m_Log = Logger.getLogger("SipContainer");
    private static SessionManager m_Instance = null;
    private static final int INITIAL_CAPACITY = 1000;
    private static Class m_FactoryClass = null;

    // needed for dialog creational NOTIFY request
    private ConcurrentHashMap<String, DialogSet> m_EarlyDialogs = null;
    private Layer m_NextLayer = null;
    private SipURIImpl m_URI = null;
    private SipURIImpl m_SURI = null;
    private boolean defaultTCPTransport = false;
    private AtomicLong m_EasFailedSipDialogs = new AtomicLong();
    private AtomicLong m_EasSuccessfulSipDialogs = new AtomicLong();
    private Config _config = ConfigFactory.getConfig();

    /**
     * private since its is a Singleton
     */
    protected SessionManager() {
        // TODO read initial capacity from configuration could also set
        // loadfactor and concurrency level
        m_EarlyDialogs = new ConcurrentHashMap<String, DialogSet>(INITIAL_CAPACITY);
    }

    /**
     * Initializes this Singleton via the double checked locking pattern to
     * minimize performance penelty since its accessed often.
     */
    public void start() {
        //TODO Deal with failures and lease expired.

        // Build a Contact URI and a header
        m_URI = new SipURIImpl();
        m_SURI = new SipURIImpl();

        if (Boolean.getBoolean("sip.module.use_new_config")) {
            SipURIImpl[] ifs = getInterfaces();

            //Find first UDP || TCP ||TLS
            for (int i = 0; i < ifs.length; i++) {
                if ((m_URI == null) && defaultTCPTransport &&
                        ifs[i].getTransportParam().equals("TCP")) {
                    m_URI = ifs[i];
                } else if ((m_URI == null) &&
                        ifs[i].getTransportParam().equals("UDP")) {
                    //Should the transport param be removed?
                    m_URI = ifs[i];
                } else if ((m_SURI == null) && ifs[i].isSecure()) {
                    m_SURI = ifs[i];
                }
            }
        } else {
            m_URI.setHost(_config.get(Constants.SIP_PUBLIC_HOST));
            m_URI.setPort(Integer.parseInt(_config.get(
                        Constants.SIP_PUBLIC_PORT)));
            m_SURI.setHost(_config.get(Constants.SIP_PUBLIC_HOST));
            m_SURI.setPort(m_URI.getPort() + 1);
            m_SURI.setSecure(true);
            //Do we have to have it or not!?
            //It should not hurt
            m_SURI.setTransportParam("TLS");
        }

        if (defaultTCPTransport) {
            if (m_Log.isLoggable(Level.FINE)) {
                m_Log.log(Level.FINE,
                    "SessionManager starting, defaultTCPTransport=" +
                    defaultTCPTransport);
            }

            m_URI.setTransportParam("tcp");
        }
    }

    private SipURIImpl[] getInterfaces() {
        SipURIImpl[] interfaces = null;

        try {
            Lease<TargetTuple[]> lease = SipBindingResolver.instance()
                                                           .lease(SipBindingResolver.PUBLIC_BINDING_CTX);
            //TODO traverse all
            TargetTuple[] tt = lease.getResource();
            interfaces = new SipURIImpl[tt.length];

            for (int i = 0; i < tt.length; i++) {
                //SipURIImpl s = new SipURIImpl((tt[i].getProtocol().ordinal() == SipTransports.TLS), null, tt[i].getIP());
                SipURIImpl s = new SipURIImpl();
		s.setSecure(tt[i].getProtocol().ordinal() == SipTransports.TLS);
		s.setHost(tt[i].getIP());
                s.setPort(tt[i].getPort());
                s.setTransportParam(tt[i].getProtocol().name());
                interfaces[i] = s;
            }
        } catch (LeaseExpiredException e) {
            //TODO logging
            e.printStackTrace();
        }

        return interfaces;
    }

    public void addContact(SipServletRequestImpl req) {
        //12.1.2 If Req URI contains sips or top route then contact must be sips
        //UAC behaviour
        DialogFragment df = req.getDialog();
        int fragId;

        URI nextTarget = null;
        Header r = req.getRawHeader(Header.ROUTE);

        if (r == null) { //No route headers, next hop ReqURI
            nextTarget = req.getRequestURI();
        } else {
            //TODO maybe not a SipURI!
            try {
                nextTarget = r.getAddressValue().getURI();
            } catch (ServletParseException ignore) {
            }
        }

        SipURIImpl uri = null;

        if ((nextTarget != null) && nextTarget.isSipURI() &&
                ((SipURI) nextTarget).isSecure()) {
            uri = (SipURIImpl) m_SURI.clone();
        } else {
            uri = (SipURIImpl) m_URI.clone();
        }

        if ((df != null) &&
                ((fragId = df.getFragmentId()) != DialogFragment.DEFAULT_FRAGMENT_ID)) {
            uri.setParameter(SipURIImpl.FRAGID_PARAM, Integer.toString(fragId));
        }

        Header contactHeader = new MultiLineHeader(Header.CONTACT, true);
        contactHeader.setValue("<" + uri.toString() + ">", false);
        req.addHeader(contactHeader);
        req.indicateContact();
    }

    public void addContact(SipServletResponseImpl resp) {
        //12.1.1 Contact should be sips if:
        //Req uri of the request was sips or
        //Top most RR was sips and if none present the contact was sips
        //UAS behaviour
        DialogFragment df = resp.getDialog();
        int fragId;

        URI nextTarget = null;

        if ((resp.getRequestImpl() != null) &&
                (resp.getRequestImpl().getRequestURI() != null) &&
                resp.getRequestImpl().getRequestURI().isSipURI()) {
            nextTarget = resp.getRequestImpl().getRequestURI();
        }

        if ((nextTarget != null) && (!((SipURI) nextTarget).isSecure())) {
            Header rr = resp.getRawHeader(Header.RECORD_ROUTE);

            try {
                if ((rr != null) && rr.getAddressValue().getURI().isSipURI()) {
                    nextTarget = rr.getAddressValue().getURI();
                } else { //Get the other 

                    Header c = resp.getRequestImpl().getRawHeader(Header.CONTACT);

                    if ((c != null) && (c.getAddressValue() != null)) {
                        nextTarget = c.getAddressValue().getURI();
                    }
                }
            } catch (ServletParseException spe) {
                if (m_Log.isLoggable(Level.FINE)) {
                    m_Log.log(Level.FINE, "Failed eval 12.1.1 rules", spe);
                }
            }
        }

        SipURIImpl uri = null;

        if ((nextTarget != null) && nextTarget.isSipURI() &&
                ((SipURI) nextTarget).isSecure()) {
            uri = (SipURIImpl) m_SURI.clone();
        } else {
            uri = (SipURIImpl) m_URI.clone();
        }

        if ((df != null) &&
                ((fragId = df.getFragmentId()) != DialogFragment.DEFAULT_FRAGMENT_ID)) {
            uri.setParameter(SipURIImpl.FRAGID_PARAM, Integer.toString(fragId));
        }

        Header contactHeader = new MultiLineHeader(Header.CONTACT, true);
        contactHeader.setValue("<" + uri.toString() + ">", false);
        resp.addHeader(contactHeader);
        resp.indicateContact();
    }

    /**
     * Adds the default record route to the message
     *
     * @param req
     *           the message that will get the record route header
     * @throws ServletParseException thrown in case thye Record Route URI is not valid
     */
    public void addRecordRoute(SipServletRequestImpl req) {
        DialogFragment df = req.getDialog();
        int fragId;
        Header rrHeader;

        rrHeader = req.getRawHeader(Header.RECORD_ROUTE);

        if (rrHeader == null) {
            rrHeader = new MultiLineHeader(Header.RECORD_ROUTE, true);
            req.addHeader(rrHeader);
        }

        //Do the rfc 3261 sips check 16.6 (4,6,7)
        URI nextTarget = null;
        Header r = req.getRawHeader(Header.ROUTE);

        if (r == null) { //No route headers, next hop ReqURI
            nextTarget = req.getRequestURI();
        } else {
            //TODO maybe not a SipURI!
            try {
                if (r.getAddressValue() != null) {
                    nextTarget = r.getAddressValue().getURI();
                }
            } catch (ServletParseException ignore) {
            }
        }

        SipURIImpl uri = null;

        if ((nextTarget != null) && nextTarget.isSipURI() &&
                ((SipURI) nextTarget).isSecure()) {
            uri = (SipURIImpl) m_SURI.clone();
            uri.setLrParam(true);
        } else {
            uri = (SipURIImpl) m_URI.clone();
            uri.setLrParam(true);
        }

        if (!((df == null) ||
                ((fragId = df.getFragmentId()) == DialogFragment.DEFAULT_FRAGMENT_ID))) {
            uri.setParameter(SipURIImpl.FRAGID_PARAM, Integer.toString(fragId));
            fragId = 0;
        }

        setRecordRouteParams(req, uri);
        rrHeader.setValue("<" + uri.toString() + ">", true);
    }

    private void setRecordRouteParams(SipServletRequestImpl req, SipURIImpl uri) {
        for (Iterator iter = req.getRecordRouteURIParamNames(); iter.hasNext();) {
            String name = (String) iter.next();
            uri.setParameter(name, req.getRecordRouteURIParam(name));
        }
    }

    /**
     * Attaches the session and dialog to the incoming request if found otherwise
     * an error response is returned.
     *
     * @param req
     *           the incoming request
     * @return null if session is recovered otherwise the error response Call
     *         Leg/Transaction does not exist (481).
     */
    private SipServletResponseImpl setDialogContext(SipServletRequestImpl req) {
        SipServletResponseImpl resp = null;
        SipSessionBase s = getSession(req);

        if (s != null) {
            req.setSession(s);
        } else {
            if (!req.getMethod().equals("ACK")) {
                // Status code (481) indicating Call Leg/Transaction does not exist.
                resp = req.createTerminatingResponse(481);
            } else {
                if (m_Log.isLoggable(Level.FINE)) {
                    m_Log.log(Level.FINE,
                        "Call Leg/Transaction does not exist for request " +
                        req.getMethod() + " with callId = " + req.getCallId() +
                        ", from = " + req.getFrom() + ", to = " + req.getTo());
                }
            }
        }

        return resp;
    }

    /**
     * Sets the request to initial or subsequent
     *
     * @param req
     *           the request to set
     * @throws ServletParseException
     */
    public void next(SipServletRequestImpl req) {
        SipServletResponseImpl resp = null;
        req.pushTransactionDispatcher(this);
        req.pushApplicationDispatcher(this);

        if (!req.isInitial()) {
            resp = setDialogContext(req);

            if (resp != null) {
                resp.popDispatcher().dispatch(resp);

                return;
            }
        } else {
            if (SipFactoryImpl.isDialogCreational(req.getMethod())) {
                if (req.getHeader(Header.CONTACT) == null) {
                    // Respond with error code 400 because of the contact header
                    // is missing for a dialog creational request.
                    resp = req.createTerminatingResponse(400,
                            "Missing Contact header field");
                    resp.popDispatcher().dispatch(resp);

                    return;
                } else if (!isValidContact(req.getRawHeader(Header.CONTACT))) {
                    // Respond with error code 400 because of the contact header
                    // is invalid.

                    // TR HH52078
                    resp = req.createTerminatingResponse(400,
                            "Invalid Contact header field");

                    if (resp == null) {
                        return;
                    }

                    resp.popDispatcher().dispatch(resp);

                    return;
                }
            }
        }

        LayerHelper.next(req, this, m_NextLayer);
    }

    private boolean isValidContact(Header header) {
        // Verify the address spec is correct.
        try {
            Address adr = header.getAddressValue();
            adr.toString();
        } catch (ServletParseException e) {
            return false;
        }

        return true;
    }

    /**
     * Replaces PathNodes of the response transactionStack.
     *
     * @param resp
     * @param d
     */
    private void replaceTranscationPath(SipServletResponseImpl resp,
        DialogFragment d) {
        // need to update transaction path since there are new nodes.
        for (int i = 0; i < d.size(); i++) {
            // lets remove old nodes..
            resp.popDispatcher();
        }

        // need to take it in reverse order since its a stack.
        PathNode p = null;
        Iterator<PathNode> i = d.getCallee2CallerPath();

        while (i.hasNext()) {
            // ...and replace them with new nodes
            p = i.next();

            if (resp.getRequestImpl().getSupervised()) {
                resp.pushTransactionDispatcher(p);
            }
        }
    }

    public void next(SipServletResponseImpl resp) {
        // accociate response with session and dialog, clone dialog if
        // necessary...
        SipServletRequestImpl req = resp.getRequestImpl();
        DialogFragment dialog = req.getDialog();

        if (dialog != null) {
            String respToTag = resp.getTo().getParameter(AddressImpl.TAG_PARAM);

            if ((respToTag == null) && !(resp.getStatus() == 100)) {
                // Except for 100 Trying, all responses MUST have a to-tag
                // to be complying to RFC3261 8.2.6.2
                throw new IllegalStateException(
                    "Missing mandatory To-tag in response");
            }

            //
            // A dialog-establishing NOTIFY request might
            // already have created and stored a dialog
            //
            if (req.getMethod().equals("SUBSCRIBE") && req.isInitial()) {
                DialogSet ds = dialog.getDialogSet();
                String id = DialogFragment.createKey(resp.getCallId(),
                        ds.getFromTag(), respToTag, req.getFragmentId());
                DialogFragment clonedOrFetched = DialogFragmentManager.getInstance()
                                                                      .findDialogFragment(id);

                if (clonedOrFetched == null) {
                    boolean success = false;
                    // fetch stored dialog in set,
                    // lets use fragmentId since it might be a spiral
                    clonedOrFetched = ds.getDialog(req.getFragmentId());

                    // is it already occupied?
                    if (clonedOrFetched != null) {
                        success = clonedOrFetched.tryToSetToTagAndRegisterDialog(respToTag,
                                true);
                    }

                    if (!success) {
                        // need to clone dialog
                        clonedOrFetched = ds.cloneDialog(req.getFragmentId());
                    }
                }

                resp.setDialog(clonedOrFetched);
                resp.copyTransactionStack();

                // it might be same dialog as from original request...
                if (clonedOrFetched != dialog) {
                    if (clonedOrFetched != null) {
                        replaceTranscationPath(resp, clonedOrFetched);
                    } else {
                        if (m_Log.isLoggable(Level.FINE)) {
                            Iterator<DialogFragment> allDialogFragments = ds.getDialogs();
                            int counter = 0;

                            while (allDialogFragments.hasNext()) {
                                counter++;
                            }

                            m_Log.log(Level.FINE,
                                "[clonedOrFetched=" + clonedOrFetched +
                                ", FromTag=" + ds.getFromTag() +
                                ", FragmentId=" + req.getFragmentId() +
                                ", allDialogFragments size=" + counter +
                                " FragmentId=" + dialog.getFragmentId());
                        }

                        // No dialogFragment exist.
                        return;
                    }
                }
            } else if (req.isInitial() && (dialog.getToTag() != null) &&
                    !dialog.getToTag().equals(respToTag)) {
                String id = DialogFragment.createKey(resp.getCallId(),
                        dialog.getDialogSet().getFromTag(), respToTag,
                        req.getFragmentId());
                DialogFragment clonedOrFetched = DialogFragmentManager.getInstance()
                                                                      .findDialogFragment(id);

                if (clonedOrFetched == null) {
                    // need to clone dialog
                    clonedOrFetched = (DialogFragment) dialog.clone();
                }

                resp.setDialog(clonedOrFetched);
                resp.copyTransactionStack();
                replaceTranscationPath(resp, clonedOrFetched);
            } else {
                resp.setDialog(dialog);
                resp.copyTransactionStack();
            }

            SipServletRequestImpl transReq = req.getTransactionRequest();

            if (transReq != null) {
                resp.setRequest(transReq);
                resp.setSession(transReq.getSessionImpl());
            } else {
                resp.setSession(req.getSessionImpl());
            }

            LayerHelper.next(resp, this, m_NextLayer);
        } else {
            // no dialog exist, lets return and go back...
            resp.copyTransactionStack();
            dispatch(resp);
        }
    }

    public void registerNext(Layer layer) {
        m_NextLayer = layer;
    }

    public void dispatch(SipServletRequestImpl req) {
        if (m_Log.isLoggable(Level.FINE)) {
            m_Log.log(Level.FINE, req.toDebugString());
        }

        // add Record Route if its enabled and no contact exist
        if (!req.isContactIndicated() && req.isRecordRouteIndicated()) {
            addRecordRoute(req);
        }

        // lets register the early dialog...
        registerEarlyDialog(req);
        // done...
        req.popDispatcher().dispatch(req);
    }

    public void dispatch(SipServletResponseImpl resp) {
        if (m_Log.isLoggable(Level.FINE)) {
            m_Log.log(Level.FINE, resp.toDebugString());
        }

        if (PerformanceMBeanListener.isEnabled() &&
                resp.getRequest().isInitial() &&
                SipFactoryImpl.isDialogCreational(resp.getRequest().getMethod())) {
            if ((resp.getStatus() >= 200) && (resp.getStatus() < 300)) {
                // OK
                m_EasSuccessfulSipDialogs.incrementAndGet();
            } else if ((resp.getStatus() >= 400) && (resp.getStatus() < 700)) {
                // FAILED
                m_EasFailedSipDialogs.incrementAndGet();
            }
        }

        resp.popDispatcher().dispatch(resp);
    }

    /**
     * Returns the instance of the SessionManager
     *
     * @return the instance of the SessionManager
     */
    public static SessionManager getInstance() {
        if (m_Instance == null) {
            synchronized (SessionManager.class) {
                if (m_Instance == null) {
                    if (m_FactoryClass == null) {
                        m_Instance = new SessionManager();
                    } else {
                        try {
                            m_Instance = (SessionManager) m_FactoryClass.newInstance();
                        } catch (Exception e) {
                            m_Log.log(Level.SEVERE,
                                "Failed to instantiate factory class : ", e);
                        }
                    }
                }
            }
        }

        return m_Instance;
    }

    /**
     * Returns the instance of the SessionManager
     *
     * @return the instance of the SessionManager
     */
    public static void initInstance(Class instance) {
        if (m_Log.isLoggable(Level.FINE)) {
            m_Log.log(Level.FINE,
                "instantiates this layer with class = " +
                instance.getCanonicalName());
        }

        if (m_Instance == null) {
            m_FactoryClass = instance;
        } else {
            throw new IllegalStateException(
                "Can not init class after the factory has given a reference away!");
        }
    }

    /**
     * Get an early dialog for the given id.
     *
     * @param id
     *           the key to find an early dialog
     * @return an early dialog, or null
     */
    public DialogSet getEarlyDialog(String id) {
        return m_EarlyDialogs.get(id);
    }

    /**
     * Clean up to gurantee that early dialogs are properly removed
     *
     * @param d
     *           the early dialog
     */
    public void removeEarlyDialog(DialogSet d) {
        if (!m_EarlyDialogs.isEmpty()) {
            String id = DialogFragment.createKey(d.getCallId(), d.getFromTag());

            if (m_Log.isLoggable(Level.FINE)) {
                m_Log.log(Level.FINE,
                    "removeEarlyDialog with key = " + id + ", DialogSet = " +
                    d);
            }

            m_EarlyDialogs.remove(id);
        }
    }

    /**
     * Register the early dialog.
     *
     * @param req
     *           the SUBSCRIBE request
     */
    private void registerEarlyDialog(SipServletRequestImpl req) {
        if (SipFactoryImpl.isDialogCreational(req.getMethod()) &&
                req.isInitial()) {
            DialogFragment d = req.getDialog();

            if (d != null) {
                DialogSet ds = d.getDialogSet();
                String id = DialogFragment.createKey(ds.getCallId(),
                        ds.getFromTag());
                m_EarlyDialogs.putIfAbsent(id, ds);
            }
        }
    }

    /**
     * Fetches the early dialog and clones it.
     *
     * @param m
     *           the NOTIFY request
     * @return the clone of the early dialog if found otherwise null
     */
    private DialogFragment createDialogByCloningEarlyDialog(
        SipServletMessageImpl m) {
        DialogFragment d = null;

        if (m.getMethod().equals("NOTIFY") && !m_EarlyDialogs.isEmpty()) {
            // According to RFC3265, 3.3.4 "to header of NOTIFY
            // should match from header of SUBSCRIBE"
            String toTag = m.getTo().getParameter(AddressImpl.TAG_PARAM);
            String id = DialogFragment.createKey(m.getCallId(), toTag);
            DialogSet ds = m_EarlyDialogs.get(id);

            if (ds != null) {
                boolean success = false;
                // fetch stored dialog in set,
                // lets use fragmentId since it might be a spiral
                d = ds.getDialog(m.getFragmentId());

                // is it already occupied?
                if (d != null) {
                    success = d.tryToSetToTagAndRegisterDialog(m.getFrom()
                                                                .getParameter(AddressImpl.TAG_PARAM),
                            true);
                }

                if (!success) {
                    d = ds.cloneDialog(m.getFragmentId());
                }

                m.setDialog(d);
            }
        }

        return d;
    }

    /**
     * According to RFC3265, 3.3.4 Dialog creation and termination: a matching
     * response or NOTIFY will establish a dialog. This method supports the
     * creation of dialog using a NOTIFY request and returns a established
     * session if found otherwise null.
     *
     * @param m
     *           the NOTIFY request
     * @return the session which now is established or null if not found
     */
    private SipSessionBase getEarlySession(SipServletRequestImpl req) {
        SipSessionBase s = null;
        DialogFragment d = createDialogByCloningEarlyDialog(req);

        if (d != null) {
            Iterator<PathNode> iter = null;
            PathNode p = null;
            req.setDirection(Type.Callee);
            req.setDialog(d);
            iter = d.getCaller2CalleePath();

            // add the dispatchers
            while (iter.hasNext()) {
                p = iter.next();
                req.pushApplicationDispatcher(p);
            }

            s = p.getSipSession();
        }

        return s;
    }

    /**
     * Fetches a established session for incoming subsequent requests.
     *
     * @param m
     *           the subsequent request
     * @return the established session or null if not found
     */
    public SipSessionBase getSession(SipServletRequestImpl m) {
        // 1. fetch matching dialog [callId, to-tag, from-tag]
        // 2. fetch the first path node from the application path of the dialog
        // depending on the direction (caller to callee or vice versa)
        // 3. Return the SipSession of the path node.
        SipSessionBase s = null;
        String toTag = null;
        String fromTag = null;
        DialogFragment d = null;
        Iterator<PathNode> iter = null;
        PathNode p = null;

        // we need the fragmentId here
        int fragmentId = m.getFragmentId();
        // fetch to-tag & from-tag
        toTag = m.getTo().getParameter(AddressImpl.TAG_PARAM);
        fromTag = m.getFrom().getParameter(AddressImpl.TAG_PARAM);

        String id = DialogFragment.createKey(m.getCallId(), toTag, fromTag,
                fragmentId);
        d = DialogFragmentManager.getInstance().findDialogFragment(id);

        if (d != null) {
            m.setDirection(Type.Callee);
            m.setDialog(d);
            iter = d.getCaller2CalleePath();

            // add the dispatchers
            while (iter.hasNext()) {
                p = iter.next();
                m.pushApplicationDispatcher(p);
            }

            s = p.getSipSession();
        } else {
            // couldn't find session lets try to swap to-tag and from-tag
            id = DialogFragment.createKey(m.getCallId(), fromTag, toTag,
                    fragmentId);
            d = DialogFragmentManager.getInstance().findDialogFragment(id);

            if (d != null) {
                m.setDirection(Type.Caller);
                m.setDialog(d);
                iter = d.getCallee2CallerPath();

                // add the dispatchers
                while (iter.hasNext()) {
                    p = iter.next();
                    m.pushApplicationDispatcher(p);
                }

                s = p.getSipSession();
            } else {
                s = getEarlySession(m);
            }
        }

        return s;
    }

    /**
     * Sets the value for this parameter, this is done via reflection when
     * starting EAS. see <code> SSAContainerListener.initializeServer()</code>.
     *
     * @param defaultTCPTransport
     *           true if TCP should be used false if UDP should be used.
     */
    public void setDefaultTCPTransport(Boolean defaultTCPTransport) {
        this.defaultTCPTransport = defaultTCPTransport;
    }

    /*
    public Header getContactHeader()
    {
       return m_ContactHeader;
    }
     */
    public long getEasFailedSipDialogs() {
        return m_EasFailedSipDialogs.longValue();
    }

    public long getEasExpiredSipDialogs() {
        return DialogFragmentManager.getInstance().getEasExpiredSipDialogs();
    }

    public long getEasSuccessfulSipDialogs() {
        return m_EasSuccessfulSipDialogs.longValue();
    }

    public long getEasTotalSipDialogCount() {
        return DialogFragmentManager.getInstance().getEasTotalSipDialogCount();
    }

    public long getEasTotalSipDialogLifeTime() {
        return DialogFragmentManager.getInstance().getEasTotalSipDialogLifeTime();
    }

    public long getEasConcurrentSipDialogs() {
        return DialogFragmentManager.getInstance().getEasConcurrentSipDialogs();
    }

    public long getEasHeapMemoryUsage() {
        return Runtime.getRuntime().totalMemory() -
        Runtime.getRuntime().freeMemory();
    }
}
