/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2007. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.Map;

import javax.servlet.ServletOutputStream;
import javax.servlet.sip.Address;
import javax.servlet.sip.Proxy;
import javax.servlet.sip.Rel100Exception;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipServletResponse;
import javax.servlet.sip.TooManyHopsException;
import javax.servlet.sip.URI;

import org.jvnet.glassfish.comms.security.auth.impl.AuthHeaderProcessor;

import com.ericsson.ssa.config.LayerHandler;
import com.ericsson.ssa.sip.B2buaHelperImpl.TempSession;
import com.ericsson.ssa.sip.PathNode.Type;
import com.ericsson.ssa.sip.dns.TargetTuple;

/**
 * @author ekrigro
 * 
 */
public class SipServletResponseImpl extends SipServletMessageImpl implements
        SipServletResponse, Serializable {
    /**
     * Comment for <code>serialVersionUID</code>
     */
    public static final String CLIENT_CERT = "javax.servlet.response.X509Certificate";
    private static final long serialVersionUID = 3256728389854769457L;

    // Space enough for all headers (excl content)
    private static final int SIZE_ALLHEADERS = 8192;
    private static ThreadLocal<ByteBuffer> _bytebuffer = new ThreadLocal<ByteBuffer>() {
        protected synchronized ByteBuffer initialValue() {
            return ByteBuffer.allocate(SIZE_ALLHEADERS);
        }
    };

    public static final Map<Integer, String> REASON_PHRASE_MAP = new HashMap<Integer, String>();

    static {
        REASON_PHRASE_MAP.put(new Integer(100), "Trying");
        REASON_PHRASE_MAP.put(new Integer(180), "Ringing");
        REASON_PHRASE_MAP.put(new Integer(181), "Call Is Being Forwarded");
        REASON_PHRASE_MAP.put(new Integer(182), "Queued");
        REASON_PHRASE_MAP.put(new Integer(183), "Session Progress");
        REASON_PHRASE_MAP.put(new Integer(200), "OK");
        REASON_PHRASE_MAP.put(new Integer(202), "Accepted");
        REASON_PHRASE_MAP.put(new Integer(300), "Multiple Choices");
        REASON_PHRASE_MAP.put(new Integer(301), "Moved Permanently");
        REASON_PHRASE_MAP.put(new Integer(302), "Moved Temporarily");
        REASON_PHRASE_MAP.put(new Integer(305), "Use Proxy");
        REASON_PHRASE_MAP.put(new Integer(380), "Alternative Service");
        REASON_PHRASE_MAP.put(new Integer(400), "Bad Request");
        REASON_PHRASE_MAP.put(new Integer(401), "Unauthorized");
        REASON_PHRASE_MAP.put(new Integer(402), "Payment Required");
        REASON_PHRASE_MAP.put(new Integer(403), "Forbidden");
        REASON_PHRASE_MAP.put(new Integer(404), "Not Found");
        REASON_PHRASE_MAP.put(new Integer(405), "Method Not Allowed");
        REASON_PHRASE_MAP.put(new Integer(406), "Not Acceptable");
        REASON_PHRASE_MAP
                .put(new Integer(407), "Proxy Authentication Required");
        REASON_PHRASE_MAP.put(new Integer(408), "Request Timeout");
        REASON_PHRASE_MAP.put(new Integer(410), "Gone");
        REASON_PHRASE_MAP.put(new Integer(412), "Conditional Request Failed");
        REASON_PHRASE_MAP.put(new Integer(413), "Request Entity Too Large");
        REASON_PHRASE_MAP.put(new Integer(414), "Request-URI Too Long");
        REASON_PHRASE_MAP.put(new Integer(415), "Unsupported Media Type");
        REASON_PHRASE_MAP.put(new Integer(416), "Unsupported URI Scheme");
        REASON_PHRASE_MAP.put(new Integer(420), "Bad Extension");
        REASON_PHRASE_MAP.put(new Integer(421), "Extension Required");
        REASON_PHRASE_MAP.put(new Integer(422), "Session Interval Too Small");
        REASON_PHRASE_MAP.put(new Integer(423), "Interval Too Brief");
        REASON_PHRASE_MAP.put(new Integer(480), "Temporarily Unavailable");
        REASON_PHRASE_MAP.put(new Integer(481),
                "Call/Transaction Does Not Exist");
        REASON_PHRASE_MAP.put(new Integer(482), "Loop Detected");
        REASON_PHRASE_MAP.put(new Integer(483), "Too Many Hops");
        REASON_PHRASE_MAP.put(new Integer(484), "Address Incomplete");
        REASON_PHRASE_MAP.put(new Integer(485), "Ambiguous");
        REASON_PHRASE_MAP.put(new Integer(486), "Busy Here");
        REASON_PHRASE_MAP.put(new Integer(487), "Request Terminated");
        REASON_PHRASE_MAP.put(new Integer(488), "Not Acceptable Here");
        REASON_PHRASE_MAP.put(new Integer(489), "Bad Event");
        REASON_PHRASE_MAP.put(new Integer(491), "Request Pending");
        REASON_PHRASE_MAP.put(new Integer(493), "Undecipherable");
        REASON_PHRASE_MAP.put(new Integer(500), "Server Internal Error");
        REASON_PHRASE_MAP.put(new Integer(501), "Not Implemented");
        REASON_PHRASE_MAP.put(new Integer(502), "Bad Gateway");
        REASON_PHRASE_MAP.put(new Integer(503), "Service Unavailable");
        REASON_PHRASE_MAP.put(new Integer(504), "Server Time-out");
        REASON_PHRASE_MAP.put(new Integer(505), "Version Not Supported");
        REASON_PHRASE_MAP.put(new Integer(513), "Message Too Large");
        REASON_PHRASE_MAP.put(new Integer(600), "Busy Everywhere");
        REASON_PHRASE_MAP.put(new Integer(603), "Decline");
        REASON_PHRASE_MAP.put(new Integer(604), "Does Not Exist Anywhere");
        REASON_PHRASE_MAP.put(new Integer(606), "Not Acceptable");
    }

    private int _statusCode;
    private String _phrase;
    private SipServletRequestImpl _currentRequest;
    private byte[] _byteArr = null;
    protected List<Dispatcher> _retransmission_transactionStack = null;
    private boolean _isAlreadyACKGenerated = false;
    private boolean _isAlreadyPRACKGenerated = false;

    /**
     * Indicates if the Response is to be sent reliable
     * 
     */
    private boolean _reliableProvisionalResponse = false;
    private PathNode _current = null;
    private PathNode _previous = null;
    Header _viaOfCancel = null;
    private TempSession _sessionToLink = null;

    // Redirection 3xx
    private boolean _alreadyRedirected = false;

    public SipServletResponseImpl() {
        super();
        _messageType = SipMessageType.SipResponse;
    }

    public SipServletResponseImpl(SipServletRequestImpl originalRequest,
            String protocol, int code) {
        super(protocol);
        _messageType = SipMessageType.SipResponse;
        _statusCode = code;
        _phrase = REASON_PHRASE_MAP.get(new Integer(code));
        _currentRequest = originalRequest;

        // If phrase is null, use default phrase.
        if (_phrase == null) { // A range check is already preformed.

            int defaultCode = (code / 100);

            switch (defaultCode) {
            case 1:
                _phrase = REASON_PHRASE_MAP.get(new Integer(100));

                break;

            case 2:
                _phrase = REASON_PHRASE_MAP.get(new Integer(200));

                break;

            case 3:
                _phrase = REASON_PHRASE_MAP.get(new Integer(300));

                break;

            case 4:
                _phrase = REASON_PHRASE_MAP.get(new Integer(400));

                break;

            case 5:
                _phrase = REASON_PHRASE_MAP.get(new Integer(500));

                break;

            case 6:
                _phrase = REASON_PHRASE_MAP.get(new Integer(600));

                break;

            default:
                _phrase = REASON_PHRASE_MAP.get(new Integer(500));
            }
        }
    }

    public SipServletResponseImpl(SipServletRequestImpl originalRequest,
            String protocol, int code, String phrase) {
        super(protocol);
        _messageType = SipMessageType.SipResponse;
        _statusCode = code;
        _phrase = phrase;
        _currentRequest = originalRequest;
    }

    private void writeObject(ObjectOutputStream stream) throws IOException {
        try {
            ByteBuffer bb = _bytebuffer.get();

            if ((bb != null) && (bb.limit() > 0)) {
                stream.writeBoolean(true); // Compact format
                stream.writeObject(bb);
                stream.writeObject(_local);
                stream.writeObject(_remote);
            } else {
                stream.writeBoolean(false); // Standard format
                stream.defaultWriteObject();
            }
        } catch (Exception ignore) {
        }
    }

    private void readObject(ObjectInputStream stream) throws IOException {
        try {
            // stream.registerValidation(this, 0);
            boolean compact = stream.readBoolean();

            if (compact) {
                _bytebuffer.set((ByteBuffer) stream.readObject());
                _local = (InetSocketAddress) stream.readObject();
                _remote = (TargetTuple) stream.readObject();
            } else {
                stream.defaultReadObject();
            }

            _transactionStack = new ArrayList<Dispatcher>();
            _applicationStack = new ArrayList<Dispatcher>();
        } catch (Exception ignore) {
        }
    }

    /**
     * Returns true if it is a redirect 3xx response that already has been
     * redirected at least one of its contacts.
     */
    public boolean isAlreadyRedirected() {
        return _alreadyRedirected;
    }

    /**
     * Only possible to set this to true if it is a redirect 3xx response
     */
    public void setAlreadyRedirected() {
        if ((getStatus() / 100) == 3) {
            _alreadyRedirected = true;
        }
    }

    public SipServletRequest getRequest() {
        return getRequestImpl();
    }

    public SipServletRequestImpl getRequestImpl() {
        return _currentRequest;
    }

    /**
     * Enables the container to update the associated request of this response
     * before it is delivered to a servlet
     * 
     * @param request
     */
    public void setRequest(SipServletRequestImpl request) {
        _currentRequest = request;
    }

    // the specialized getMethod() below was introduced when fixing TR B0007
    /**
     * Overrides the definition of getMethod() in SipServletMessageImpl. The
     * idea is that we should never return an empty string. If it can be proved
     * that the _method field of the superclass always gets set then we would
     * not need to override it here, but it appears that the constructors of
     * SipServletResponseImpl really don't set the _method.
     */
    public String getMethod() {
        String s = super.getMethod();

        if (!s.equals("")) {
            return s;
        } else if (_currentRequest != null) {
            return _currentRequest.getMethod();
        }

        return "";
    }

    public int getStatus() {
        return _statusCode;
    }

    public void setStatus(int code) {
        _statusCode = code;
    }

    public void setStatus(int code, String phrase) {
        _statusCode = code;
        _phrase = phrase;
    }

    public String getReasonPhrase() {
        return _phrase;
    }

    public ServletOutputStream getOutputStream() throws IOException {
        // TODO Auto-generated method stub
        return null;
    }

    public PrintWriter getWriter() throws IOException {
        // TODO Auto-generated method stub
        return null;
    }

    public Proxy getProxy() {
        Proxy p = null;

        try {
            p = getRequest().getProxy();
        } catch (TooManyHopsException e) {
        }

        return p;
    }

    public void sendReliably() throws Rel100Exception {
        // 1. Check that the associated request is "INVITE"
        if (getRequestImpl().getMethod().equals("INVITE") == false) {
            throw new Rel100Exception(Rel100Exception.NOT_INVITE);
        }

        // 2. Check that response is a 1xx response
        if ((getStatus() < 101) || (getStatus() > 199)) {
            throw new Rel100Exception(Rel100Exception.NOT_1XX);
        }

        // Check that the UAC Support or Require the provisional
        // response to be sent reliable, i.e. The Request has
        // the header Require or Supported with the value "100rel"
        boolean rel100supported = false;
        ListIterator lIterator = getRequestImpl().getHeaders("Require");

        while (lIterator.hasNext()) {
            String headerValue = (String) lIterator.next();

            if (headerValue.equalsIgnoreCase("100rel")) {
                rel100supported = true;
            }
        }

        lIterator = getRequestImpl().getHeaders("Supported");

        while (lIterator.hasNext()) {
            String headerValue = (String) lIterator.next();

            if (headerValue.equalsIgnoreCase("100rel")) {
                rel100supported = true;
            }
        }

        if (!rel100supported) {
            throw new Rel100Exception(Rel100Exception.NO_UAC_SUPPORT);
        }

        // 3.Initialize a flag 1xx Reliable response ongoing in the SipSession
        // and specify if the response contain an SDP
        // 
        // Only one at a time is permitted (Cf IP on CR38 PRACK)
        if (!getSessionImpl().set1xxReliableOngoing(
                (getContentLength() == 0) ? false : true)) {
            // There is an ongoing 1xx Reliable response
            throw new Rel100Exception(Rel100Exception.NOT_SUPPORTED);
        }

        // All Error checks completed, send the response

        // Indicate that this response is reliable
        setReliableProvisionalResponse(true);

        setHeader(Header.REQUIRE, SipFactoryImpl.SUPPORTED_100REL);

        Header rseq = new SingleLineHeader(Header.RSEQ, true);
        rseq.setValue(Integer.toString(getRequestImpl().getAndIncrementRSeq()),
                true);
        setHeader(rseq);

        try {
            sendInternal();
        } catch (IOException e) {
            getSessionImpl().reset1xxReliable();
            throw new IllegalStateException("Problem while sending response.",
                    e);
        }
    }

    /**
     * Check if this SipServletResponseImpl is originally created in this
     * container
     * 
     * @return true if originally created here, false otherwise
     */
    private boolean isLocallyCreated() {
        // No to tag in dialog == no dialog is set up yet == response created
        // here
        if (getDialog().getToTag() == null) {
            return true;
        }

        // For existing dialogs: getLocalParty() == From header in locally
        // created requests
        // These should differ for locally created responses.
        return !(_currentRequest.getFrom().equals(getSessionImpl()
                .getLocalParty()));
    }

    public SipServletRequest createAck() throws IllegalStateException {
        /*
         * SSA1.0 tell us: Throws: java.lang.IllegalStateException - if the
         * transaction state is such that it doesn't allow an ACK to be sent
         * now, e.g. if the original request was not an INVITE, if this response
         * is provisional only, or if an ACK has already been generated
         */
        if (_currentRequest.getMethod().equals("INVITE") && (_statusCode > 199)
                && !_isAlreadyACKGenerated && !isLocallyCreated()) {
            // Indicate that we are to send an ACK.
            _isAlreadyACKGenerated = true;

            return createAckImpl();
        } else {
            throw new IllegalStateException(
                    "Not allowed to create an ACK at this transaction state.");
        }
    }

    public SipServletRequest createPrack() throws Rel100Exception {
        if (_currentRequest.getMethod().equals("INVITE") &&
                ((_statusCode > 100) && (_statusCode < 200)) &&
                !_isAlreadyPRACKGenerated && !isLocallyCreated()) {
            if (getHeader(Header.RSEQ) == null) {
                throw new Rel100Exception(Rel100Exception.NOT_1XX);
            }

            // Indicate that we are to send an PRACK.
            _isAlreadyPRACKGenerated = true;

            return getSession().createRequest("PRACK");
        } else if ((_statusCode < 101) || (_statusCode > 199)) {
            throw new Rel100Exception(Rel100Exception.NOT_1XX);
        }

        throw new IllegalStateException(
            "Not allowed to create an PRACK at this transaction state.");
    }

    public SipServletRequestImpl createHopAckImpl(SipServletRequestImpl req)
            throws IllegalStateException {
        /*
         * SSA1.0 tell us: Throws: java.lang.IllegalStateException - if the
         * transaction state is such that it doesn't allow an ACK to be sent
         * now, e.g. if the original request was not an INVITE, if this response
         * is provisional only, or if an ACK has already been generated
         */
        if (_currentRequest.getMethod().equals("INVITE") && (_statusCode > 199)
                && !_isAlreadyACKGenerated) {
            // Indicate that we are to send an ACK.
            _isAlreadyACKGenerated = true;

            Address fromCopy = (Address) ((AddressImpl) getFrom()).clone(true,
                    true);
            Address toCopy = (Address) ((AddressImpl) getTo())
                    .clone(true, true);
            SipServletRequestImpl ack = new SipServletRequestImpl("ACK", req
                    .getRequestURI(), SipFactoryImpl.PROTOCOL_LINE);

            // set To
            Header toHeader = new SingleLineHeader(Header.TO, true);
            toHeader.setAddressValue(toCopy, false);
            ack.setHeader(toHeader);

            // set From
            Header fromHeader = new SingleLineHeader(Header.FROM, true);
            fromHeader.setAddressValue(fromCopy, false);
            ack.setHeader(fromHeader);

            // set Max-Forwards
            Header maxForwardsHeader = new SingleLineHeader(
                    Header.MAX_FORWARDS, false);
            maxForwardsHeader.setValue("70", false);
            ack.setHeader(maxForwardsHeader);

            // set CallID
            Header callIDHeader = new SingleLineHeader(Header.CALL_ID, true);
            callIDHeader.setValue(getCallId(), false);
            ack.setHeader(callIDHeader);

            // set CSeq
            Header cSeqHeader = new SingleLineHeader(Header.CSEQ, true);
            cSeqHeader.setValue(Integer.toString(getCSeqNumber()) + " ACK",
                    false);
            ack.setHeader(cSeqHeader);

            // Get the top VIA of the request
            ViaImpl v = new ViaImpl(req.getHeader(Header.VIA));
            Header via = new MultiLineHeader(Header.VIA, true);
            via.setValue(v.toString(), true);
            ack.setHeader(via);
            // the hopack shall follow the same path as the request
            // (hopByHop) and is never resolved by ResolverManager
            // so we must assign remote target here
            ack.setRemote(req.getRemote());
            // subsequent request
            ack.setInitial(false);

            // add Routes
            Header rawRoute = req.getRawHeader(Header.ROUTE);

            if (rawRoute != null) {
                Header route = (Header) rawRoute.clone();
                ack.setHeader(route);
            }

            return ack;
        } else {
            throw new IllegalStateException(
                    "Not allowed to create an ACK in this transaction state.");
        }
    }

    public SipServletRequestImpl createAckImpl() throws IllegalStateException {
        Address fromCopy = (Address) ((AddressImpl) getFrom())
                .clone(true, true);
        Address toCopy = (Address) ((AddressImpl) getTo()).clone(true, true);
        DialogFragment df = getDialog();
        UA uac = null;

        if (isDirection() == Type.Caller) {
            // need to fetch the uac from the first item of the application path
            uac = (UA) df.getFirst();
        } else if (isDirection() == Type.Callee) {
            // need to fetch the uac from the last item of the application
            // path
            uac = (UA) df.getLast();
        } else {
            throw new IllegalStateException(
                    "Don't know the direction of the flow.");
        }

        URI remoteTarget = uac.getRemoteTarget();

        if (remoteTarget == null) {
            try {
                Address contact = getAddressHeader(Header.CONTACT);

                if (contact != null) {
                    remoteTarget = getAddressHeader(Header.CONTACT).getURI();
                } else {
                    throw new IllegalStateException(
                            "Missing Contact header field");
                }
            } catch (ServletParseException e) {
                throw new IllegalStateException(e);
            }
        }

        SipServletRequestImpl req = new SipServletRequestImpl("ACK",
                remoteTarget, SipFactoryImpl.PROTOCOL_LINE);

        // set To
        Header toHeader = new SingleLineHeader(Header.TO, true);
        toHeader.setAddressValue(toCopy, false);
        req.setHeader(toHeader);

        // set From
        Header fromHeader = new SingleLineHeader(Header.FROM, true);
        fromHeader.setAddressValue(fromCopy, false);
        req.setHeader(fromHeader);

        // set Max-Forwards
        Header maxForwardsHeader = new SingleLineHeader(Header.MAX_FORWARDS,
                false);
        maxForwardsHeader.setValue("70", false);
        req.setHeader(maxForwardsHeader);

        // set CallID
        Header callIDHeader = new SingleLineHeader(Header.CALL_ID, true);
        callIDHeader.setValue(getCallId(), false);
        req.setHeader(callIDHeader);

        // set CSeq
        Header cSeqHeader = new SingleLineHeader(Header.CSEQ, true);
        cSeqHeader.setValue(Integer.toString(getCSeqNumber()) + " ACK", false);
        req.setHeader(cSeqHeader);
        // update new request with session...
        req.setSession(getSessionImpl());
        // ...and dialog
        req.setDialog(getDialog());
        // find out direction
        req.setDirection(isDirection());
        // subsequent request
        req.setInitial(false);

        // Fetch the Application stack
        List<Layer> layers = LayerHandler.getInstance().getLayers();
        req._applicationStack.addAll(layers);

        return req;
    }

    public void setCharacterEncoding(String arg0) {
        // TODO Auto-generated method stub
    }

    public void setBufferSize(int arg0) {
        // TODO Auto-generated method stub
    }

    public int getBufferSize() {
        // TODO Auto-generated method stub
        return 0;
    }

    public void flushBuffer() throws IOException {
        // TODO Auto-generated method stub
    }

    public void resetBuffer() {
        // TODO Auto-generated method stub
    }

    public void reset() {
        // TODO Auto-generated method stub
    }

    public void setLocale(Locale arg0) {
        // TODO Auto-generated method stub
    }

    public Locale getLocale() {
        // TODO Auto-generated method stub
        return null;
    }

    public void restoreRetransmissionTransactionStack() {
        _transactionStack = new ArrayList<Dispatcher>(
                _retransmission_transactionStack);
    }

    public void serializeForTransmission() throws UnsupportedEncodingException {
        _byteArr = null;
        _currentRequest = null;
        _applicationStack.clear();
        _retransmission_transactionStack = new ArrayList<Dispatcher>(
                _transactionStack);

        // we do not need to serialize the content
        ByteBuffer theByteBuf = null;
        theByteBuf = _bytebuffer.get();
        theByteBuf.clear();
        toBuffer(theByteBuf);
        _byteArr = new byte[theByteBuf.position()];
        theByteBuf.flip();
        theByteBuf.get(_byteArr);

        // TODO (qbinjoe) the clearing of fields has been commented out since
        // the converged load balancer needs the information;
        // this is a short term solution, in the long term we need to find
        // another way to conserve memory

        // trim some data to be GC'd ASAP
        // _content_enc = null;
        // _content_byte = null;
        // attrib = null;
        // systemAttrib = null;
        // headerMap = null;
        // _roles = null;
        // keep _remote = _remote;
        // keep _local = _local;
        // _Type = null;
        // _sf = null;
    }

    public void copyTransactionStack() {
        if (_currentRequest != null) {
            _transactionStack.addAll(_currentRequest._transactionStack);
        }
    }

    public Object clone() {
        SipServletResponseImpl clone = new SipServletResponseImpl(
                _currentRequest, getProtocol(), getStatus(), getReasonPhrase());
        Iterator<Header> i = headerMap.values().iterator();

        // deep copy
        while (i.hasNext()) {
            Header next = i.next();
            clone.headerMap.put(next.getName(), (Header) next.clone());
        }

        // if (attrib != null)
        // {
        // clone.attrib.putAll(attrib);
        // }
        // if (systemAttrib != null)
        // {
        // clone.systemAttrib.putAll(systemAttrib);
        // }
        // /
        if (attrib != null) {
            if (systemAttrib != null) {
                clone.attrib = new HashMap<String, Object>(systemAttrib);
            } else {
                clone.attrib = new HashMap<String, Object>(attrib.size());
            }

            clone.attrib.putAll(attrib);
        }

        if (systemAttrib != null) {
            clone.systemAttrib = new HashMap<String, Object>(systemAttrib
                    .size());
            clone.systemAttrib.putAll(systemAttrib);
        }

        // shallow copy
        clone._reliableProvisionalResponse = _reliableProvisionalResponse;
        clone._content_enc = _content_enc;
        clone._content_byte = _content_byte;
        clone._content_obj = _content_obj;
        clone._roles = _roles;
        clone._remote = _remote;
        clone._local = _local;
        clone._session = _session;
        clone._dialog = _dialog;
        clone._method = _method;
        clone._sf = _sf;
        clone._headersComplete = _headersComplete;
        clone._messageComplete = _messageComplete;
        clone._current = _current;
        clone._previous = _previous;
        clone._viaOfCancel = _viaOfCancel;
        // find out direction
        clone.setDirection(isDirection());

        // copy transaction stack (omit application stack)
        if (_transactionStack != null) {
            clone._transactionStack.addAll(_transactionStack);
        }

        return clone;
    }

    public String toString() {
        if (_byteArr != null) {
            try {
                return new String(_byteArr, SipFactoryImpl.SIP_CHARSET);
            } catch (UnsupportedEncodingException e) {
                // Build string from parts instead.
            }
        }

        StringBuilder sb = new StringBuilder(SipFactoryImpl.PROTOCOL_LINE);
        sb.append(' ');
        sb.append(_statusCode);
        sb.append(' ');
        sb.append(_phrase);
        sb.append(SipFactoryImpl.NEW_LINE);

        Iterator<Header> i = headerMap.values().iterator();

        while (i.hasNext()) {
            sb.append(i.next());
        }

        if (!headerMap.containsKey(Header.CONTENT_LENGTH)) {
            int length = 0;

            if ((_content_byte != null) && (_content_byte.length > 0)) {
                length = _content_byte.length;
            }

            Header cl = new SingleLineHeader(Header.CONTENT_LENGTH, false);
            cl.setValue(String.valueOf(length), false);
            headerMap.put(Header.CONTENT_LENGTH, cl);
            sb.append(cl.toString());
        }

        sb.append(SipFactoryImpl.NEW_LINE);

        if ((_content_byte != null) && (_content_byte.length > 0)) {
            sb.append(_content_byte);
        }

        return sb.toString();
    }

    /**
     * Translates the response to a byte buffer using the encoding specified by
     * SipFactoryImpl.SIP_CHARSET. Note that this conversion method is only
     * guaranteed to succeed if the SipFactoryImpl.SIP_CHARSET constant is set
     * to "UTF-8".
     * 
     * @return The encoded response.
     * @throws UnsupportedEncodingException Thrown when
     *                 SipFactoryImpl.SIP_CHARSET encoding is not supported.
     */
    public ByteBuffer toBuffer() throws UnsupportedEncodingException {
        serializeForTransmission();
        ByteBuffer buf = ByteBuffer.wrap(_byteArr);
        buf.position(buf.limit());
        _byteArr = null;
        return buf;
    }

    /**
     * Translates the response to a byte buffer using the encoding specified by
     * SipFactoryImpl.SIP_CHARSET. Note that this conversion method is only
     * guaranteed to succeed if the SipFactoryImpl.SIP_CHARSET constant is set
     * to "UTF-8" and the response can be encoded into "UTF-8". Also note that
     * there is no way to safeguard from the possibility that the conversion
     * fails due to an encoding error.
     * 
     * @param bb The buffer to encode te response into.
     * @return The encoded buffer.
     * @throws UnsupportedEncodingException Thrown when
     *                 SipFactoryImpl.SIP_CHARSET encoding is not supported.
     */
    public ByteBuffer toBuffer(ByteBuffer bb)
            throws UnsupportedEncodingException {
        if (_byteArr != null) {
            bb.put(_byteArr);

            return bb;
        }

        bb.put(SipFactoryImpl.PROTOCOL_LINE.getBytes());
        bb.put((byte) 0x20);
        bb.put(String.valueOf(_statusCode).getBytes());
        bb.put((byte) 0x20);
        bb.put(_phrase.getBytes(SipFactoryImpl.SIP_CHARSET));
        bb.put(SipFactoryImpl.NEW_LINE.getBytes());

        Iterator<Header> i = headerMap.values().iterator();

        while (i.hasNext()) {
            bb.put(i.next().toString().getBytes(SipFactoryImpl.SIP_CHARSET));
        }

        if (!headerMap.containsKey(Header.CONTENT_LENGTH)) {
            int length = 0;

            if ((_content_byte != null) && (_content_byte.length > 0)) {
                length = _content_byte.length;
            }

            Header cl = new SingleLineHeader(Header.CONTENT_LENGTH, false);
            cl.setValue(String.valueOf(length), false);
            headerMap.put(Header.CONTENT_LENGTH, cl);
            bb.put(cl.toString().getBytes());
        }

        bb.put(SipFactoryImpl.NEW_LINE.getBytes());

        if ((_content_byte != null) && (_content_byte.length > 0)) {
            bb.put(_content_byte);
        }

        return bb;
    }

    public String toDebugString() {
        StringBuilder sb = new StringBuilder(Integer.toString(_statusCode));
        sb.append(" ");
        sb.append(_phrase);
        sb.append(" ");

        if (_byteArr == null) {
            sb.append(super.toDebugString());
        } else {
            sb.append("BLOB");
        }

        return sb.toString();
    }

    public void send() throws IOException {
        if (getRequestImpl().getMethod().equals("INVITE")) {
            // SendReliable must be called if the inital request contain
            // the header REQUIRE: 100Rel
            if ((getStatus() > 100) && (getStatus() < 200)) {
                ListIterator lIterator = getRequestImpl().getHeaders("Require");

                while (lIterator.hasNext()) {
                    String headerValue = (String) lIterator.next();

                    if (headerValue.equalsIgnoreCase("100rel")) {
                        throw new IllegalStateException(
                                "Request with Require header 100rel, Provisional Response must be sent reliable");
                    }
                }

                // Only none SDP 1XX Reliable response can be overpass by a
                // 200OK
            } else if ((getStatus() == 200)
                    && getSessionImpl().is1xxReliableSDP()) {
                throw new IllegalStateException(
                        "Final response not allowed, 1XX reliable response with SDP ongoing RFC3262");
            }
        }

        sendInternal();
    }

    /**
     * Support of JSR 289 10.2.3 Sending Responses Helper function of the
     * Proxy.startVirtualProxyBranch
     * 
     * @throws IOException
     */
    private void startVirtualProxyBranch() throws IOException {
        // check that the proxy has been created
        if (_currentRequest.isInitial()
                && (_currentRequest.getProxyContext() != null)) {
            _currentRequest.getProxyContext().getProxy()
                    .startVirtualProxyBranch(this);
            // must be added to response path
            pushTransactionDispatcher(_currentRequest.getProxyContext());

            // must update contact
            if (SipFactoryImpl.getContactRequirement(this) != HeaderRequirement.NOT_APPLICAPLE) {
                // since it's a new fragment id the contact must be replaced...
                Header contact = getRawHeader(Header.CONTACT);
                contact.setReadOnly(false);
                contact.removeValues();
                SessionManager.getInstance().addContact(this);
            }
        }
    }

    /**
     * Support of JSR 289 10.2.3 Sending Responses Adding the ProxyContext to
     * the path if this is a Proxy acting as a UAS NOTE: only added for response
     * to initial request
     * 
     * @throws IOException
     */
    private void loadVirtualProxyBranch() throws IOException {
        // check that the proxy has been created
        if (_currentRequest.isInitial()
                && (_currentRequest.getProxyContext() != null)) {
            // must be added to response path
            pushTransactionDispatcher(_currentRequest.getProxyContext());
        }
    }

    public void sendInternal() throws IOException {
        PathNode uas = null;

        if (getRequest().isInitial() && !_currentRequest.isInApplicationPath()) {
            // it's a raise condition between answers,
            // use double check locking pattern
            synchronized (_currentRequest) {
                if (!_currentRequest.isInApplicationPath()) {
                    // support of JSR289, 10.2.3 Sending Responses (as a Proxy)
                    startVirtualProxyBranch();

                    // create the uas
                    uas = new UA(getApplicationSessionImpl(), false);
                    // lets add the current uas to the application & transaction
                    // path
                    getDialog().addToPath(uas);
                    // lets notify request that it is in app path
                    _currentRequest.setInApplicationPath(true);
                    setCurrentVisited(uas);
                } else {
                    // need to fetch the uas from the last item of the
                    // application
                    // path
                    uas = getDialog().getLast();
                    setCurrentVisited(uas);
                }

                setSent(true);
            }
        } else {
            loadVirtualProxyBranch();

            // lets build the dispatcher list from the transaction path
            if (isDirection() == Type.Caller) {
                // need to fetch the uas from the last item of the application
                // path
                uas = getDialog().getLast();
            } else if (isDirection() == Type.Callee) {
                // need to fetch the uas from the first item of the application
                // path
                uas = getDialog().getFirst();
            } else {
                throw new IOException("Don't know the direction of the flow.");
            }
        }

        // Tell the request that we will send a final response.
        // Final responses are 2xx, 3xx, 4xx, 5xx and 6xx.
        if ((_statusCode > 199) && (_statusCode < 700)) {
            _currentRequest.setSentFinalResponse(_statusCode);
        }

        // send
        uas.send(this);
    }

    @Override
    public void setCertificate(X509Certificate[] cert) {
        super.setCertificate(cert);

        if (systemAttrib == null) {
            systemAttrib = new HashMap<String, Object>();
        }

        this.systemAttrib.put(CLIENT_CERT, cert);
    }

    public String getSessionCase() {
        // TODO Auto-generated method stub
        return null;
    }

    public void setSessionCase(String sessionCase) {
        // TODO Auto-generated method stub
    }

    public Header getCancelVia() {
        return _viaOfCancel;
    }

    public void setCancelVia(Header via) {
        _viaOfCancel = via;
    }

    /**
     * @return the previous path node visited from the dispatcher stack or null
     *         if no path node was available
     */
    public PathNode getPreviousVisited() {
        return _previous;
    }

    /**
     * @return the current path node visited from the dispatcher stack or null
     *         if no path node was available
     */
    public PathNode getCurrentVisited() {
        return _current;
    }

    /**
     * Set the start path node
     * 
     * @param p
     */
    private void setCurrentVisited(PathNode p) {
        _current = p;
    }

    public Dispatcher popDispatcher() {
        int size = _transactionStack.size();

        if (size > 0) {
            Dispatcher d = _transactionStack.remove(size - 1);

            if ((getRequestImpl() != null) && getRequestImpl().isInitial()
                    && d instanceof PathNode) {
                // the current path node is now set as previous
                _previous = _current;
                // save the poped path node as current
                _current = (PathNode) d;
            }

            return d;
        }

        return null;
    }

    public Dispatcher peekDispatcher() {
        int size = _transactionStack.size();

        if (size > 0) {
            return _transactionStack.get(size - 1);
        }

        return null;
    }

    public boolean isReliableProvisionalResponse() {
        return _reliableProvisionalResponse;
    }

    public void setReliableProvisionalResponse(
            boolean reliableProvisionalResponse) {
        _reliableProvisionalResponse = reliableProvisionalResponse;
    }

    private TempSession getB2buaHelperTempSession() {
        return _sessionToLink;
    }

    public void setB2buaHelperTempSession(TempSession s) {
        _sessionToLink = s;
    }

    public void setSession(SipSessionBase sessionImpl) {
        // signal of b2buaHelper to link sessions...
        if (getB2buaHelperTempSession() != null) {
            // session were not jet linked, lets link them
            getRequest().getB2buaHelper().linkSipSessions(sessionImpl,
                    getB2buaHelperTempSession().getDerivedUACSessionToLink());
            
            // move attributes from temp. session to real session
            String name = null;
            Enumeration<String> names = getB2buaHelperTempSession().getAttributeNames();
            while (names.hasMoreElements()){
                name = names.nextElement();
                sessionImpl.setAttribute(name, getB2buaHelperTempSession().getAttribute(name));
            }     
        }

        super.setSession(sessionImpl);
    }

    public String[] getChallengeRealm() {
        if (getStatus() == SipServletResponse.SC_UNAUTHORIZED) {
            AuthHeaderProcessor ahp = new AuthHeaderProcessor();

            return ahp.getHeaderValues(this, "WWW-Authenticate", "Realm");
        } else if (getStatus() == SipServletResponse.SC_PROXY_AUTHENTICATION_REQUIRED) {
            AuthHeaderProcessor ahp = new AuthHeaderProcessor();

            return ahp.getHeaderValues(this, "Proxy-Authenticate", "Realm");
        }

        return null;
    }
}
